package com.arms.api.report.export_service.model;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;

import java.util.ArrayList;
import java.util.List;

/**
 * 차트 데이터 모델
 *
 * @author HS.Yang
 * @since 25.12.02
 */
@Getter
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class ChartData {

    /** 차트 ID */
    private String chartId;

    /** 차트 제목 */
    private String title;

    /** 부제목 */
    private String subtitle;

    /** 차트 유형 */
    @Builder.Default
    private ChartType type = ChartType.BAR;

    /** X축 라벨 */
    @Builder.Default
    private List<String> labels = new ArrayList<>();

    /** 데이터 시리즈 */
    @Builder.Default
    private List<DataSeries> series = new ArrayList<>();

    /** 차트 옵션 */
    @Builder.Default
    private ChartOptions options = new ChartOptions();

    /** 렌더링된 이미지 (선택적) */
    private byte[] renderedImage;

    /**
     * 차트 유형
     */
    public enum ChartType {
        BAR, LINE, PIE, DOUGHNUT, AREA, SCATTER, RADAR
    }

    /**
     * 데이터 시리즈 클래스
     */
    @Getter
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    public static class DataSeries {
        private String name;

        @Builder.Default
        private List<Double> values = new ArrayList<>();

        private String color;
    }

    /**
     * 차트 옵션 클래스
     */
    @Getter
    @Builder
    @NoArgsConstructor
    @AllArgsConstructor
    public static class ChartOptions {
        @Builder.Default
        private boolean showLegend = true;

        @Builder.Default
        private boolean showValues = false;

        @Builder.Default
        private String xAxisLabel = "";

        @Builder.Default
        private String yAxisLabel = "";

        @Builder.Default
        private int width = 600;

        @Builder.Default
        private int height = 400;
    }

    public void addLabel(String label) {
        this.labels.add(label);
    }

    public void addSeries(DataSeries dataSeries) {
        this.series.add(dataSeries);
    }

    public void addSeries(String name, List<Double> values, String color) {
        this.series.add(DataSeries.builder()
                .name(name)
                .values(values)
                .color(color)
                .build());
    }

    public boolean hasRenderedImage() {
        return renderedImage != null && renderedImage.length > 0;
    }
}
