package com.arms.api.analysis.cost.service;

import com.arms.api.analysis.cost.model.dto.SampleDTO;
import com.arms.api.analysis.cost.model.vo.SalaryVO;
import com.arms.api.analysis.cost.model.vo.SalaryEntity;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@Slf4j
@RequiredArgsConstructor
public class SalaryServiceImpl extends TreeServiceImpl implements SalaryService {
    @Override
    public List<SampleDTO> getSampleSalaryInfo() {

        List<SampleDTO> 샘플데이터 = new ArrayList<>();

        AtomicInteger 번호 = new AtomicInteger(1);
        샘플데이터.add(new SampleDTO(String.valueOf(번호.getAndIncrement()), "이름", "ALM(Jira, Redmine, GitLab) 에 설정된 사용자 이름 입니다.", "ex) moon", "최초 등록 후 수정 불가"));
        샘플데이터.add(new SampleDTO(String.valueOf(번호.getAndIncrement()), "고유 키", "ALM(Jira, Redmine, GitLab) 에 설정된 고유한 사용자의 아이디 입니다.", "ex) 6265535bfff19d0069259584", "최초 등록 후 수정 불가"));
        샘플데이터.add(new SampleDTO(String.valueOf(번호.getAndIncrement()), "연봉", "사용자의 연봉 정보입니다. (단위: 만원)", "ex) 5000", "최초 등록 후 수정 가능"));

        return 샘플데이터;
    }

    @Override
    @Transactional
    public List<SalaryEntity> compareSalaryInfo(List<SalaryVO> SalaryVO) throws Exception {

        log.info(" [ {} :: 연봉정보비교 ]", this.getClass().getName());

        // DB에 저장된 데이터 조회
        SalaryEntity SalaryEntity = new SalaryEntity();
        List<SalaryEntity> DB_데이터 = this.getNodesWithoutRoot(SalaryEntity);

        // DB 데이터와 받은 데이터 비교
        if (DB_데이터.size() != 0) {
            Map<String, SalaryEntity> DB_맵 = DB_데이터.stream()
                    .collect(Collectors.toMap(엔티티 -> 엔티티.getC_name() + 엔티티.getC_key(), 엔티티 -> 엔티티));

            SalaryVO.stream()
                    .filter(데이터 -> DB_맵.containsKey(데이터.getC_name() + 데이터.getC_key()))
                    .forEach(데이터 -> 데이터.setC_annual_income(DB_맵.get(데이터.getC_name() + 데이터.getC_key()).getC_annual_income())); // DB에 저장된 값으로 세팅
        } else {
            log.info("[ {} :: 연봉정보비교 ] :: DB에 저장된 데이터가 없음", this.getClass().getName());
        }

        // 엔티티로 변환
        List<SalaryEntity> 연봉엔티티리스트 = SalaryVO.stream()
                .map(데이터 -> {
                    SalaryEntity 엔티티 = new SalaryEntity();
                    엔티티.setC_name(데이터.getC_name());
                    엔티티.setC_key(데이터.getC_key());
                    엔티티.setC_annual_income(데이터.getC_annual_income());
                    return 엔티티;
                })
                .collect(Collectors.toList());

        // json으로 변환
        ObjectMapper objectMapper = new ObjectMapper();
        String json = objectMapper.writeValueAsString(SalaryVO);
        log.info("[ {} :: 연봉정보비교 ] :: 연봉 데이터 -> {}", this.getClass().getName(), json);

        return 연봉엔티티리스트;

    }

    @Override
    @Transactional
    public List<SalaryVO> saveExcelDataToDB(List<SalaryEntity> salaryEntityList) throws Exception {

        SalaryEntity SalaryEntity = new SalaryEntity();
        List<SalaryEntity> DB_데이터 = this.getNodesWithoutRoot(SalaryEntity);

        List<SalaryEntity> 엑셀_데이터 = 이름_키_NULL_EMPTY_제외_목록(salaryEntityList);

        //////////////////////////////////////////
        // 업로드한 엑셀 데이터 중복 검사 시작
        //////////////////////////////////////////
        List<SalaryEntity> 중복_엔티티 = new ArrayList<>();
        Map<String, SalaryEntity> 엑셀데이터_키_엔티티_맵 = new HashMap<>();
        엑셀_데이터.forEach(엔티티 -> {
            String 연봉 = 엔티티.getC_annual_income();
            // 연봉 값이 없거나 빈 문자열인 경우 0으로 설정
            if (연봉 == null || 연봉.isEmpty()) {
                엔티티.setC_annual_income("0");
            }

            String key = 엔티티.getC_key();

            // 키가 이미 존재하는 경우 중복 리스트에 추가
            if (엑셀데이터_키_엔티티_맵.containsKey(key)) {
                중복_엔티티.add(엔티티); // 중복 엔티티 추가
            } else {
                엑셀데이터_키_엔티티_맵.put(key, 엔티티); // 중복되지 않은 경우만 맵에 추가
            }
        });

        // 중복된 엔티티가 있는지 확인
        if (!중복_엔티티.isEmpty()) {
            // 중복된 엔티티의 이름과 키를 포맷하여 문자열로 변환
            String 중복_정보 = 중복_엔티티.stream()
                    .map(dupEntity -> "이름: " + dupEntity.getC_name() + ", 키(key): " + dupEntity.getC_key())
                    .collect(Collectors.joining(" | ")); // 엔트리 간 구분자 사용
            // 로그에 중복 정보를 기록
            log.error("[ SalaryServiceImpl :: 엑셀데이터_DB저장 ] :: 중복된 데이터 -> {}", 중복_정보);

            // RuntimeException을 던져 중복 정보를 포함한 에러 응답 생성
            throw new RuntimeException("[ SalaryServiceImpl :: 엑셀데이터_DB저장 ] :: 중복된 키(key)가 존재합니다: " + 중복_정보);
        }
        //////////////////////////////////////////
        // 업로드한 엑셀 데이터 중복 검사 종료 (Exception 안보내면 정상실행)
        //////////////////////////////////////////

        // DB 데이터와 비교해서 없으면 insert, 있으면 update
        if (DB_데이터.size() != 0) {
            // DB 데이터와 비교
            Map<String, SalaryEntity> DB_맵 = DB_데이터.stream()
                    .collect(Collectors.toMap(엔티티 -> 엔티티.getC_key(), 엔티티 -> 엔티티));

            for (Map.Entry<String, SalaryEntity> entry : 엑셀데이터_키_엔티티_맵.entrySet()) {
                String 키 = entry.getKey();
                SalaryEntity 엑셀_연봉정보 = entry.getValue();

                // DB_맵에 같은 키의 엔티티가 있는 경우, 그 엔티티의 연봉을 업데이트
                if (DB_맵.containsKey(키)
                        && 연봉_값이_같지_않으면(DB_맵.get(키), 엑셀_연봉정보)) {

                    SalaryEntity DB_엔티티 = DB_맵.get(키);
                    log.info(" [ {} :: 엑셀데이터_DB저장 ] :: 연봉정보 업데이트 -> 이름: {} 키(key): {}"
                            , this.getClass().getName(), 엑셀_연봉정보.getC_name(), 엑셀_연봉정보.getC_key());

                    DB_엔티티.setC_annual_income(엑셀_연봉정보.getC_annual_income());
                    this.updateNode(DB_엔티티);
                }
                // DB_맵에 같은 키의 엔티티가 없는 경우, 1. 엑셀_엔티티를 추가 2. DB_맵에 추가
                else if(!DB_맵.containsKey(키)){
                    log.info(" [ {} :: 엑셀데이터_DB저장 ] :: 연봉정보 신규 저장 -> 이름 : {} 키(key): {}"
                            ,this.getClass().getName(), 엑셀_연봉정보.getC_name(), 엑셀_연봉정보.getC_key());
                    SalaryEntity 추가한_연봉정보_엔티티 = this.addNode(new SalaryEntity(엑셀_연봉정보.getC_name(), 엑셀_연봉정보.getC_key(), 엑셀_연봉정보.getC_annual_income()));
                    DB_맵.put(키, 추가한_연봉정보_엔티티);
                }
            }

            List<SalaryVO> 결과_연봉정보DTO_목록 = 연봉정보_엔티티_콜렉션_DTO_목록으로_변환(DB_맵.values());

            return 결과_연봉정보DTO_목록;

        } else {
            List<SalaryEntity> 추가한_연봉정보_목록 = new ArrayList<>();
            // DB에 처음 저장하는 경우
            for (SalaryEntity 연봉정보 : 엑셀_데이터) {
                log.info(" [ {} :: 엑셀데이터_DB저장 ] :: 연봉정보 신규 저장 -> 이름: {} 키(key): {}"
                        ,this.getClass().getName(), 연봉정보.getC_name(), 연봉정보.getC_key());
                SalaryEntity 추가한_연봉정보_엔티티 =
                        this.addNode(new SalaryEntity(연봉정보.getC_name(), 연봉정보.getC_key(), 연봉정보.getC_annual_income()));
                추가한_연봉정보_목록.add(추가한_연봉정보_엔티티);
            }

            List<SalaryVO> 결과_연봉정보DTO_목록 = 연봉정보_엔티티_콜렉션_DTO_목록으로_변환(추가한_연봉정보_목록);
            return 결과_연봉정보DTO_목록;
        }
    }

    @Override
    public Map<String, SalaryEntity> getAllSalariesMap() throws Exception {
        SalaryEntity salaryEntity = new SalaryEntity();
        Function<SalaryEntity, String> k = SalaryEntity::getC_key;
        Function<SalaryEntity, SalaryEntity> v = Function.identity();
        return getNodesWithoutRootMap(salaryEntity, k, v);
    }

    @Override
    public List<SalaryVO> getAllSalaries() throws Exception {
        SalaryEntity salaryEntity = new SalaryEntity();
        List<SalaryEntity> nodesWithoutRoot = this.getNodesWithoutRoot(salaryEntity);

        return convertEntitiesToSalaryVOs(nodesWithoutRoot);

    }

    private List<SalaryVO> convertEntitiesToSalaryVOs(List<SalaryEntity> salaryEntities) {
        return salaryEntities.stream()
                .map(entity -> SalaryVO.builder()
                        .c_name(entity.getC_name())
                        .c_key(entity.getC_key())
                        .c_annual_income(entity.getC_annual_income())
                        .build())
                .collect(Collectors.toList());

    }


    @Override
    @Transactional
    public int updateSalary(List<SalaryEntity> salaryEntityList) throws Exception {
        Map<String, SalaryEntity> allSalariesMap = this.getAllSalariesMap();

        for (SalaryEntity salaryEntity : salaryEntityList) {
            SalaryEntity existingSalaryInfo = allSalariesMap.get(salaryEntity.getC_key());
            if (existingSalaryInfo == null) {
                salaryEntity.setRef(TreeConstant.First_Node_CID);
                salaryEntity.setC_type(TreeConstant.Leaf_Node_TYPE);
                this.addNode(salaryEntity);
            } else {
                existingSalaryInfo.setC_annual_income(salaryEntity.getC_annual_income());
                this.updateNode(existingSalaryInfo);
            }
        }

        return 1;
    }

    private boolean 연봉_값이_같지_않으면(SalaryEntity entity1, SalaryEntity entity2) {
        return !Objects.equals(entity1.getC_annual_income(), entity2.getC_annual_income());
    }

    private static List<SalaryEntity> 이름_키_NULL_EMPTY_제외_목록(List<SalaryEntity> 엑셀데이터) {
        List<SalaryEntity> 엑셀_데이터 = 엑셀데이터.stream()
                .filter(엔티티 -> {
                    String 이름 = 엔티티.getC_name();
                    String 키 = 엔티티.getC_key();
                    // 이름과 키 모두 값이 있어야 함
                    return 이름 != null && !이름.isEmpty() && 키 != null && !키.isEmpty();
                }).collect(Collectors.toList());
        return 엑셀_데이터;
    }

    private List<SalaryVO> 연봉정보_엔티티_콜렉션_DTO_목록으로_변환(Collection<SalaryEntity> 연봉정보_엔티티_콜렉션) {
        return 연봉정보_엔티티_콜렉션.stream()
                .map(this::DTO로_변환)
                .collect(Collectors.toList());
    }

    private SalaryVO DTO로_변환(SalaryEntity 엔티티) {
        SalaryVO 데이터 = new SalaryVO();
        데이터.setC_name(엔티티.getC_name());
        데이터.setC_key(엔티티.getC_key());
        데이터.setC_annual_income(엔티티.getC_annual_income());
        return 데이터;
    }
}