package com.arms.api.analysis.resource.service;

import com.arms.api.analysis.resource.model.dto.ResourceDTO;
import com.arms.api.analysis.resource.model.dto.ResourceWithVersionIdNamesDTO;
import com.arms.api.analysis.resource.model.dto.VersionIdNameDTO;
import com.arms.api.analysis.resource.model.vo.StackedHorizontalBarChartVO;
import com.arms.api.analysis.resource.model.vo.UniqueAssigneeIssueStatusVO;
import com.arms.api.analysis.resource.model.vo.UniqueAssigneeVO;
import com.arms.api.analysis.resource.model.vo.WordCloudVO;
import com.arms.api.analysis.resource.model.vo.horizontalbar.HorizontalBarChartYAxisAndSeriesVO;
import com.arms.api.analysis.resource.model.vo.horizontalbar.ReqAndNotReqHorizontalBarChartVO;
import com.arms.api.analysis.resource.model.vo.pie.ReqAndNotReqPieChartVO;
import com.arms.api.analysis.resource.model.vo.pie.TotalIssueAndPieChartVO;
import com.arms.api.analysis.resource.model.vo.sankey.SankeyChartBaseVO;
import com.arms.api.analysis.resource.model.vo.sankey.VersionAssigneeSummaryVO;
import com.arms.api.analysis.resource.model.vo.treemap.TreeMapWorkerVO;
import com.arms.api.analysis.resource.model.vo.treemap.TreemapExcelVO;
import com.arms.api.analysis.resource.model.vo.wordcloud.WordCloudExcelVO;
import com.arms.api.dashboard.model.SankeyData;
import com.arms.api.dashboard.model.SankeyData.SankeyLink;
import com.arms.api.dashboard.model.SankeyData.SankeyNode;
import com.arms.api.product_service.pdservice.model.PdServiceEntity;
import com.arms.api.product_service.pdservice.service.PdService;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.product_service.pdserviceversion.service.PdServiceVersion;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class ResourceServiceImpl implements ResourceService {

    private final AggregationService aggregationService;

    private final PdService pdService;

    private final PdServiceVersion pdServiceVersion;

    @Override
    public List<UniqueAssigneeIssueStatusVO> issueStatusDataByAssignee(ResourceDTO resourceDTO) {
        // v2
        return Optional.ofNullable(aggregationService.issueStatusDataByAssignee(resourceDTO)).
                orElse(Collections.emptyList());
    }

    @Override
    public SankeyData sankeyChartAPIWithTopN(ResourceDTO resourceDTO, Integer topN) throws Exception {
        Long pdServiceLink = resourceDTO.pdServiceLink();
        List<Long> pdServiceVersionLinks = resourceDTO.pdServiceVersionLinks();

        PdServiceEntity pdServiceEntity = new PdServiceEntity();
        pdServiceEntity.setC_id(pdServiceLink);
        PdServiceEntity savedPdService = pdService.getNode(pdServiceEntity);

        if (pdServiceVersionLinks.isEmpty()) {
            return new SankeyData(Collections.emptyList(), Collections.emptyList());
        }

        List<SankeyNode> nodeList = new ArrayList<>();
        List<SankeyLink> linkList = new ArrayList<>();

        String pdServiceId = savedPdService.getC_id() + "-product";
        // 1. 제품 노드 추가
        nodeList.add(new SankeyData.SankeyNode(pdServiceId, savedPdService.getC_title(), "제품", ""));
        // 2. 제품 버전 노드 추가
        Set<Long> versionIds = savedPdService.getPdServiceVersionEntities().stream()
                .filter(version -> pdServiceVersionLinks.contains(version.getC_id()))
                .sorted(Comparator.comparing(PdServiceVersionEntity::getC_id))
                .map(version -> {
                    String versionId = version.getC_id() + "-version";
                    nodeList.add(new SankeyData.SankeyNode(versionId, version.getC_title(), "버전", ""));
                    // 2-1. 제품 노드와 제품 버전 노드를 연결하는 link 추가
                    linkList.add(new SankeyData.SankeyLink(pdServiceId, versionId));
                    return version.getC_id();
                })
                .collect(Collectors.toSet());

        Map<String, SankeyNode> workerNodeMap = new HashMap<>();

        // 3. SanketChartBaseVO 목록 조회 (from engine)
        List<SankeyChartBaseVO> sankeyChartBaseVOList = aggregationService.findVersionAssigneeSummaryData(resourceDTO);

        if (ObjectUtils.isEmpty(sankeyChartBaseVOList)) {
            log.info("[ ResourceServiceImpl :: sankeyChartAPI ] :: result of aggregationService.findVersionAssigneeSummaryData(resourceDTO) is null");
            return new SankeyData(Collections.emptyList(), Collections.emptyList());
        }

        Set<String> topNAccountIdSet = new HashSet<>();
        if (!ObjectUtils.isEmpty(topN)) {
            List<VersionAssigneeSummaryVO> aggregationData = aggregateAndSortByAccountId(sankeyChartBaseVOList);
            if (aggregationData.size() > topN) {
                int limit = Math.min(topN, aggregationData.size());
                List<VersionAssigneeSummaryVO> topNList = new ArrayList<>(aggregationData.subList(0, limit));
                topNAccountIdSet = topNList.stream().map(VersionAssigneeSummaryVO::getAccountId).collect(Collectors.toSet());
            }
        }

        for (SankeyChartBaseVO chartBaseVO : sankeyChartBaseVOList) {
            if(!versionIds.contains(chartBaseVO.getVersionId())) {
                continue;
            }

            String versionId = String.valueOf(chartBaseVO.getVersionId());
            List<VersionAssigneeSummaryVO> versionAssigneeSummaryVOList = chartBaseVO.getVersionAssigneeSummaryVOList();

            for (VersionAssigneeSummaryVO summaryVO : versionAssigneeSummaryVOList) {
                String accountId = summaryVO.getAccountId();
                String assigneeName = summaryVO.getName();

                if (!topNAccountIdSet.isEmpty() && !topNAccountIdSet.contains(accountId)) {
                    continue;
                }

                if (workerNodeMap.containsKey(accountId)) {
                    SankeyNode workerNode = workerNodeMap.get(accountId);
                    String parent = workerNode.getParent();
                    workerNode.setParent(parent + "," + versionId);
                } else {
                    SankeyNode workerNode = new SankeyNode(accountId, assigneeName, "작업자", versionId);
                    nodeList.add(workerNode);
                    workerNodeMap.put(accountId, workerNode);
                }

                linkList.add(new SankeyLink(versionId + "-version", accountId));
                versionIds.remove(chartBaseVO.getVersionId());
            }

        }

        // 4. 담당자가 없는 제품 버전 찾기 (계층적인 표현에 있어 UI 상 문제가 있기 때문에 가짜 노드와 링크를 추가해주어야함)
        boolean isWorkerNodeExist = false;
        List<SankeyNode> versionNodes = nodeList.stream().filter(node -> node.getType().equals("버전")).collect(Collectors.toList());
        for (SankeyNode versionNode : versionNodes) {
            if (nodeList.stream().noneMatch(workerNode -> workerNode.getParent().contains(versionNode.getId().split("-")[0]))) {
                isWorkerNodeExist = true;
                // 4-1. 제품 버전 노드와 가짜 노드를 연결하는 link 추가
                linkList.add(new SankeyLink(versionNode.getId(), "No-Worker"));
            }
        }

        // 4-1. 의 경우, 차트의 Level 이 맞지 않아 UI 가 이상해지기 때문에 가짜 노드를 추가
        if (isWorkerNodeExist) {
            // 4-2. 가짜 노드 추가
            nodeList.add(new SankeyNode("No-Worker", "No-Worker", "No-Worker", ""));
        }

        return new SankeyData(nodeList, linkList);
    }


    @Override
    public List<TreeMapWorkerVO> findTreeMapChartDataV3(ResourceWithVersionIdNamesDTO resourceWithVersionIdNamesDTO) throws Exception {
        PdServiceAndIsReqDTO pdServiceAndIsReq = resourceWithVersionIdNamesDTO.getPdServiceAndIsReq();
        List<VersionIdNameDTO> versionIdNames = pdServiceVersion.getVersionListByCids(pdServiceAndIsReq.getPdServiceVersionLinks())
                .stream()
                .map(entity -> {
                    VersionIdNameDTO dto = new VersionIdNameDTO();
                    dto.setC_id(String.valueOf(entity.getC_id()));
                    dto.setC_title(entity.getC_title());
                    return dto;
                }).collect(Collectors.toList());
        resourceWithVersionIdNamesDTO.setVersionIdNames(versionIdNames);

        return Optional.ofNullable(aggregationService.findTreeMapChartDataV3(resourceWithVersionIdNamesDTO).getBody()).orElse(Collections.emptyList());
    }

    @Override
    public List<TreemapExcelVO> assigneeRequirementInvolvementDetailData(ResourceWithVersionIdNamesDTO resourceWithVersionIdNamesDTO) throws Exception {
        PdServiceAndIsReqDTO pdServiceAndIsReq = resourceWithVersionIdNamesDTO.getPdServiceAndIsReq();
        List<VersionIdNameDTO> versionIdNames = pdServiceVersion.getVersionListByCids(pdServiceAndIsReq.getPdServiceVersionLinks())
                .stream()
                .map(entity -> {
                    VersionIdNameDTO dto = new VersionIdNameDTO();
                    dto.setC_id(String.valueOf(entity.getC_id()));
                    dto.setC_title(entity.getC_title());
                    return dto;
                }).collect(Collectors.toList());
        resourceWithVersionIdNamesDTO.setVersionIdNames(versionIdNames);

        return Optional.ofNullable(aggregationService.findAssigneeRequirementInvolvementData(resourceWithVersionIdNamesDTO).getBody()).orElse(Collections.emptyList());
    }

    @Override
    public List<UniqueAssigneeVO> findAssigneesInfoData(ResourceDTO resourceDTO) throws Exception {

        return Optional.ofNullable(aggregationService.findAssigneesInfo(resourceDTO).getBody()).orElse(new ArrayList<>());
    }

    @Override
    public List<WordCloudVO> findWordCloudData(ResourceDTO resourceDTO) {
        List<WordCloudExcelVO> wordCloudExcelVOS = aggregationService.wordCloudData(resourceDTO);
        if (!ObjectUtils.isEmpty(wordCloudExcelVOS)) {
            return convertWordCloudExcelVOtoWordCloudVO(wordCloudExcelVOS);
        } else {
            return Collections.emptyList();
        }
    }

    private List<WordCloudVO> convertWordCloudExcelVOtoWordCloudVO(List<WordCloudExcelVO> wordCloudExcelVOs) {
        return wordCloudExcelVOs.stream().map(excelVO -> {
                    return WordCloudVO.builder()
                            .text(excelVO.getName())
                            .weight(excelVO.getIssueTotal()).build();
                }).collect(Collectors.toList());
    }

    // 작업자_이슈상태_누적_차트_집계조회
    @Override
    public StackedHorizontalBarChartVO issueStatusStackedBarChartData(ResourceDTO resourceDTO) {
        return aggregationService.issueStatusStackedBarChartData(resourceDTO);
    }

    @Override
    public TotalIssueAndPieChartVO findPieChartDataExpand(ResourceDTO resourceDTO) {

        return aggregationService.findPieChartDataExpand(resourceDTO);
    }

    @Override
    public ReqAndNotReqPieChartVO findPieChartData(ResourceDTO resourceDTO) {

        ReqAndNotReqPieChartVO pieChartData = aggregationService.findPieChartData(resourceDTO);

        int numSize = resourceDTO.getNumSize();

        TotalIssueAndPieChartVO reqPieChartVO = pieChartData.getReqPieChartVO();
        reqPieChartVO.getAssigneeAndIssueCounts()
                .sort((vo1,vo2) -> vo2.getValue().compareTo(vo1.getValue()));

        TotalIssueAndPieChartVO notReqPieChartVO = pieChartData.getNotReqPieChartVO();
        notReqPieChartVO.getAssigneeAndIssueCounts()
                .sort((vo1,vo2) -> vo2.getValue().compareTo(vo1.getValue()));

        if (numSize != 0 && numSize > 0) {
            return ReqAndNotReqPieChartVO.builder()
                    .reqPieChartVO(filteredPieChartDataByNumSize(reqPieChartVO, numSize))
                    .notReqPieChartVO(filteredPieChartDataByNumSize(notReqPieChartVO, numSize))
                    .build();
        } else {
            return pieChartData;
        }
    }

    @Override
    public List<HorizontalBarChartYAxisAndSeriesVO> findHorizontalBarChartData(ResourceDTO resourceDTO) {
        return aggregationService.findHorizontalBarChartData(resourceDTO);
    }

    private TotalIssueAndPieChartVO filteredPieChartDataByNumSize(TotalIssueAndPieChartVO pieChartVO, int numSize) {
        return TotalIssueAndPieChartVO.builder()
                .totalIssueCount(pieChartVO.getTotalIssueCount())
                .assigneeAndIssueCounts(pieChartVO.getAssigneeAndIssueCounts().stream()
                        .limit(numSize)
                        .collect(Collectors.toList()))
                .build();
    }

    @Override
    public ReqAndNotReqHorizontalBarChartVO findHorizontalBarChartDataBySameAccounts(ResourceDTO resourceDTO) {

        return aggregationService.findHorizontalBarChartDataBySameAccounts(resourceDTO);
    }

    private List<VersionAssigneeSummaryVO> aggregateAndSortByAccountId(List<SankeyChartBaseVO> baseList) {
        // accountId를 키로 사용하여 집계할 Map 생성
        Map<String, VersionAssigneeSummaryVO> aggregatedMap = new HashMap<>();

        // 모든 VersionAssigneeSummaryVO 순회하며 accountId 기준으로 집계
        baseList.forEach(sankeyChart -> {
            sankeyChart.getVersionAssigneeSummaryVOList().forEach(summaryVO -> {
                String accountId = summaryVO.getAccountId();
                aggregatedMap.merge(accountId, summaryVO, (existing, newValue) -> {
                    return VersionAssigneeSummaryVO.builder()
                            .accountId(accountId)
                            .name(existing.getName())
                            .emailAddress(existing.getEmailAddress())
                            .totalIssue(existing.getTotalIssue() + newValue.getTotalIssue())
                            .reqIssueCount(existing.getReqIssueCount() + newValue.getReqIssueCount())
                            .notReqIssueCount(existing.getNotReqIssueCount() + newValue.getNotReqIssueCount())
                            .build();
                });
            });
        });

        // Map의 값들을 List로 변환하고 totalIssue 기준 내림차순 정렬
        return aggregatedMap.values()
                .stream()
                .sorted(Comparator.comparing(VersionAssigneeSummaryVO::getTotalIssue).reversed())
                .collect(Collectors.toList());

    }
}
