package com.arms.api.analysis.topmenu.service;

import com.arms.api.analysis.resource.model.dto.ResourceDTO;
import com.arms.api.analysis.resource.model.vo.UniqueAssigneeIssueStatusVO;
import com.arms.api.analysis.topmenu.model.dto.TopMenuAnalysisDTO;
import com.arms.api.analysis.topmenu.model.vo.*;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.product_service.pdserviceversion.service.PdServiceVersion;
import com.arms.api.requirement.reqadd.model.entity.ReqAddEntity;
import com.arms.api.requirement.reqadd.service.ReqAdd;
import com.arms.api.util.RangeUtil;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Restrictions;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;

@Service
@RequiredArgsConstructor
@Slf4j
public class TopMenuServiceImpl implements TopMenuService {

    private final AggregationService aggregationService;

    private final ReqAdd reqAdd;

    private final PdServiceVersion pdServiceVersion;

    @Override
    public TopMenuVO getTopMenuData(String changeReqTableName, TopMenuAnalysisDTO topMenuAnalysisDTO) throws Exception {
        PdServiceAndIsReqDTO pdServiceAndIsReq = topMenuAnalysisDTO.getPdServiceAndIsReq();
        List<Long> pdServiceVersionLinks = pdServiceAndIsReq.getPdServiceVersionLinks();

        SessionUtil.setAttribute("getReqAddListByFilter", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();
        if (pdServiceVersionLinks != null && !pdServiceVersionLinks.isEmpty()) {
            Disjunction orCondition = Restrictions.disjunction();
            for (Long version : pdServiceVersionLinks) {
                orCondition.add(Restrictions.like("c_req_pdservice_versionset_link",  "\\\"" + (version) + "\\\"", MatchMode.ANYWHERE));
            }
            reqAddEntity.getCriterions().add(orCondition);
        }
        ReqStatusCountVO reqStatusCountVO = new ReqStatusCountVO(reqAdd.getChildNode(reqAddEntity));

        SessionUtil.removeAttribute("getReqAddListByFilter");

        VersionPeriodVO versionPeriodVO = this.topMenuVersionPeriod(topMenuAnalysisDTO);
        ReqAndIssueTotalCountVO reqAndIssueTotalCountVO = this.topMenuTotalCountWithReqAndSubtask(topMenuAnalysisDTO);
        ReqAndSubtaskByAssigneeTotalVO reqAndSubtaskByAssigneeTotalVO = this.topMenuResourceInfoWithReqAndSubtask(topMenuAnalysisDTO);

        return TopMenuVO.builder()
                .reqStatusCountVO(reqStatusCountVO)
                .versionPeriodVO(versionPeriodVO)
                .reqAndIssueTotalCountVO(reqAndIssueTotalCountVO)
                .reqAndSubtaskByAssigneeTotalVO(reqAndSubtaskByAssigneeTotalVO)
                .build();
    }

    // 사용 가능.
    @Override
    public ReqStatusCountVO analysisTopMenuReqStatusSummary(String changeReqTableName, Long pdServiceId, List<Long> pdServiceVersionLinks) throws Exception {

        SessionUtil.setAttribute("getReqAddListByFilter", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();

        if (pdServiceVersionLinks != null && !pdServiceVersionLinks.isEmpty()) {
            Disjunction orCondition = Restrictions.disjunction();
            for (Long version : pdServiceVersionLinks) {
                orCondition.add(Restrictions.like("c_req_pdservice_versionset_link",  "\\\"" + (version) + "\\\"", MatchMode.ANYWHERE));
            }
            reqAddEntity.getCriterions().add(orCondition);
        }

        ReqStatusCountVO reqStatusCountVO = new ReqStatusCountVO(reqAdd.getChildNode(reqAddEntity));

        SessionUtil.removeAttribute("getReqAddListByFilter");
        log.info("[TopMenuServiceImpl  :: 톱메뉴_버전별_요구사항_자료] :: 버전_요구사항_상태별_합계 :: 총합 = {}, 열림_요구사항 = {}, 열림아닌_요구사항 = {}",
                reqStatusCountVO.getTotalCount(), reqStatusCountVO.getOpenCount(), reqStatusCountVO.getTotalCount()- reqStatusCountVO.getOpenCount());

        return reqStatusCountVO;
    }

    @Override
    public ReqAndIssueTotalCountVO topMenuTotalCountWithReqAndSubtask(TopMenuAnalysisDTO topMenuAnalysisDTO) {
        return aggregationService.topMenuTotalCountWithReqAndSubtask(topMenuAnalysisDTO).getBody();
    }

    @Override
    public ReqAndSubtaskByAssigneeTotalVO topMenuResourceInfoWithReqAndSubtask(TopMenuAnalysisDTO topMenuAnalysisDTO) throws Exception {

        PdServiceAndIsReqDTO pdServiceAndIsReq = topMenuAnalysisDTO.getPdServiceAndIsReq();
        ResourceDTO resourceDTO = new ResourceDTO();
        resourceDTO.setPdServiceAndIsReq(pdServiceAndIsReq);

        List<UniqueAssigneeIssueStatusVO> uniqueAssigneeIssueStatusList = aggregationService.issueStatusDataByAssignee(resourceDTO);

        if (uniqueAssigneeIssueStatusList == null || uniqueAssigneeIssueStatusList.isEmpty()) {
            return ReqAndSubtaskByAssigneeTotalVO.builder()
                    .resource(0L)
                    .reqIssue(0L)
                    .reqIssueMax(0L)
                    .reqIssueMin(0L)
                    .notReqIssue(0L)
                    .notReqIssueMax(0L)
                    .notReqIssueMin(0L)
                    .build();
        }

        Long resource = Long.valueOf(uniqueAssigneeIssueStatusList.size());
        Long reqTotal = 0L;
        Long notReqTotal = 0L;
        Long reqIssueMax = 0L;
        Long reqIssueMin = 1000000L;
        Long notReqIssueMax = 0L;
        Long notReqIssueMin = 1000000L;

        for (UniqueAssigneeIssueStatusVO vo : uniqueAssigneeIssueStatusList) {
            Long reqIssueCount = vo.getReqIssueCount();
            Long notReqIssueCount = vo.getNotReqIssueCount();

            reqTotal = reqTotal + reqIssueCount;
            notReqTotal = notReqTotal + notReqIssueCount;

            reqIssueMax = RangeUtil.returnIfGreater(reqIssueCount, reqIssueMax);
            reqIssueMin = RangeUtil.returnIfLess(reqIssueCount, reqIssueMin);
            notReqIssueMax = RangeUtil.returnIfGreater(notReqIssueCount, notReqIssueMax);
            notReqIssueMin = RangeUtil.returnIfLess(notReqIssueCount, notReqIssueMin);

        }

        return ReqAndSubtaskByAssigneeTotalVO.builder()
                .resource(resource)
                .reqIssue(reqTotal)
                .reqIssueMax(reqIssueMax)
                .reqIssueMin(reqIssueMin)
                .notReqIssue(notReqTotal)
                .notReqIssueMax(notReqIssueMax)
                .notReqIssueMin(notReqIssueMin)
                .build();
    }

    @Override
    public VersionPeriodVO topMenuVersionPeriod(TopMenuAnalysisDTO topMenuAnalysisDTO) throws Exception {

        List<PdServiceVersionEntity> pdServiceVersionList = pdServiceVersion.getVersionListByCids(topMenuAnalysisDTO.pdServiceVersionLinks());

        LocalDate today = LocalDate.now();

        Optional<LocalDate> optionalEarliestStartDate = pdServiceVersionList.stream()
                .map(versionEntity -> parseDateOrDefault(versionEntity.getC_pds_version_start_date(), today))
                .min(LocalDate::compareTo);

        Optional<LocalDate> optionalLatestEndDate = pdServiceVersionList.stream()
                .map(versionEntity -> parseDateOrDefault(versionEntity.getC_pds_version_end_date(), today))
                .max(LocalDate::compareTo);

        String earliestStartDate = optionalEarliestStartDate.orElse(today).toString();
        String latestEndDate = optionalLatestEndDate.orElse(today).toString();

        return new VersionPeriodVO(earliestStartDate, latestEndDate);
    }

    private LocalDate parseDateOrDefault(String dateString, LocalDate defaultDate) {
        if ("start".equalsIgnoreCase(dateString) || "end".equalsIgnoreCase(dateString)) {
            return defaultDate;
        }

        return LocalDateTime.parse(dateString, DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm")).toLocalDate();
    }
}
