package com.arms.api.blog.service;

import com.arms.api.blog.model.BlogDTO;
import com.arms.api.blog.model.BlogEntity;
import com.arms.api.blog.model.BlogVO;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.modelmapper.ModelMapper;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.persistence.EntityNotFoundException;
import java.util.*;

@Slf4j
@RequiredArgsConstructor
@Service("blogService")
public class BlogServiceImpl extends TreeServiceImpl implements BlogService {

    private final ModelMapper modelMapper;

    private final EngineService engineService;

    @Override
    @Transactional(readOnly = true)
    public BlogDTO getBlog(BlogDTO blogDTO) throws Exception {
        BlogEntity existingBlogEntity = this.getNode(modelMapper.map(blogDTO, BlogEntity.class));

        if (existingBlogEntity == null) {
            throw new EntityNotFoundException(blogDTO.getC_id() + "는 존재하지 않는 Blog입니다.");
        }

        BlogDTO resultDTO = modelMapper.map(existingBlogEntity, BlogDTO.class);
        BlogDTO existingBlogDTOByEngine = engineService.getBlog(resultDTO).getBody();

        if (existingBlogDTOByEngine != null) {
            resultDTO.setC_blog_contents(existingBlogDTOByEngine.getC_blog_contents());
            resultDTO.setC_blog_view_count(existingBlogDTOByEngine.getC_blog_view_count());
        }

        return resultDTO;
    }

    @Override
    @Transactional
    public BlogEntity addBlog(BlogDTO blogDTO) throws Exception {
        BlogEntity blogEntity = modelMapper.map(blogDTO, BlogEntity.class);
        blogEntity.setRef(TreeConstant.First_Node_CID);
        blogEntity.setC_blog_created(new Date());
        blogEntity.setC_type(TreeConstant.Leaf_Node_TYPE);

        BlogEntity createdEntity = this.addNode(blogEntity);

        engineService.addBlog(modelMapper.map(createdEntity, BlogDTO.class));
        return createdEntity;
    }

    @Override
    @Transactional
    public BlogEntity updateBlog(BlogDTO blogDTO) throws Exception {
        BlogEntity blogEntity = modelMapper.map(blogDTO, BlogEntity.class);
        if (blogDTO.getC_id() == null || this.getNode(blogEntity) == null) {
            throw new EntityNotFoundException(blogDTO.getC_id() + "는 존재하지 않는 Blog입니다.");
        }
        blogEntity.setC_blog_updated(new Date());
        this.updateNode(blogEntity);

        updateBlogEngine(blogDTO);

        return blogEntity;
    }

    @Async
    protected void updateBlogEngine(BlogDTO blogDTO) {
        try {
            engineService.updateBlog(blogDTO);
        } catch (Exception e) {
            log.error("블로그 엔진 업데이트 실패: {}", e.getMessage());
        }
    }

    @Override
    @Transactional
    public int removeBlog(BlogDTO blogDTO) throws Exception {
        BlogEntity blogEntity = modelMapper.map(blogDTO, BlogEntity.class);
        if (blogDTO.getC_id() == null || this.getNode(blogEntity) == null) {
            throw new EntityNotFoundException(blogDTO.getC_id() + "는 존재하지 않는 Blog입니다.");
        }

        removeBlogEngine(blogDTO);
        return this.removeNode(blogEntity);
    }

    @Async
    protected void removeBlogEngine(BlogDTO blogDTO) {
        try {
            engineService.deleteBlog(blogDTO.getC_id().toString());
        } catch (Exception e) {
            log.error("블로그 엔진 삭제  실패: {}", e.getMessage());
        }
    }

    @Override
    @Transactional(readOnly = true)
    public List<BlogEntity> getLatestBlogList(BlogDTO blogDto) throws Exception {
        BlogEntity blogEntity = modelMapper.map(blogDto, BlogEntity.class);

        ignoreRootAndFirstNode(blogEntity);
        orderByCreatedDesc(blogEntity);
        updatePageInfo(blogEntity, 1, 4);

        return this.getPaginatedChildNode(blogEntity);
    }

    @Override
    public List<BlogDTO> getPopularBlogList(BlogDTO blogDTO) {
        return engineService.getBlogListByViewCountDesc(blogDTO).getBody();
    }

    /**
     * 전체 블로그 리스트를 최신순 + 페이징 처리하여 반환
     *
     * @param blogDTO, pageIndex, pageUnit
     * @return Map<String, Object>
     * @throws Exception
     */
    @Override
    @Transactional(readOnly = true)
    public Map<String, Object> getPaginatedBlogList(BlogDTO blogDTO, int pageIndex, int pageUnit) throws Exception {
        BlogEntity blogEntity = modelMapper.map(blogDTO, BlogEntity.class);

        ignoreRootAndFirstNode(blogEntity);
        orderByCreatedDesc(blogEntity);
        updatePageInfo(blogEntity, pageIndex, pageUnit);

        List<BlogEntity> blogList = this.getPaginatedChildNode(blogEntity);

        HashMap<String, Object> resultMap = new HashMap<>();
        if (!(blogList == null || blogList.isEmpty())) {
            resultMap.put("paginationInfo", blogEntity.getPaginationInfo());
        } else {
            resultMap.put("paginationInfo", Map.of("totalRecordCount", 0, "lastPageNo", 0));
        }
        resultMap.put("data", blogList);

        return resultMap;
    }

    @Override
    @Transactional(readOnly = true)
    public List<BlogEntity> getRecommendedBlogList(BlogDTO blogDTO) throws Exception {
        BlogEntity blogEntity = modelMapper.map(blogDTO, BlogEntity.class);

        ignoreRootAndFirstNode(blogEntity);
        orderByCreatedDesc(blogEntity);
        updatePageInfo(blogEntity, 1, 5);

        return this.getPaginatedChildNode(blogEntity);
    }

    @Override
    @Transactional(readOnly = true)
    public BlogVO searchBlogList(BlogDTO blogDTO) {
        return engineService.getBlogListBySearchAfter(blogDTO).getBody();
    }

    private void ignoreRootAndFirstNode(BlogEntity blogEntity) {
        blogEntity.getCriterions().add(Restrictions.not(
                Restrictions.in("c_id", TreeConstant.ROOT_CID, TreeConstant.First_Node_CID)));
    }

    private void orderByCreatedDesc(BlogEntity blogEntity) {
        blogEntity.getOrder().add(Order.desc("c_blog_created"));
    }

    private void updatePageInfo(BlogEntity blogEntity, int pageIndex, int pageUnit) {
        blogEntity.setPageIndex(pageIndex);
        blogEntity.setPageUnit(pageUnit);
    }

}
