package com.arms.api.clientcase.controller;

import com.arms.api.clientcase.model.ClientCaseDTO;
import com.arms.api.clientcase.model.ClientCaseEntity;
import com.arms.api.clientcase.service.ClientCaseService;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.controller.TreeAbstractController;
import com.arms.egovframework.javaservice.treeframework.validation.group.AddNode;
import com.arms.egovframework.javaservice.treeframework.validation.group.UpdateNode;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import javax.annotation.PostConstruct;
import java.util.Map;

@Slf4j
@Controller
@RestController
@RequiredArgsConstructor
@RequestMapping(value = {"/anonymous/cover/experience"})
public class ClientCaseController extends TreeAbstractController<ClientCaseService, ClientCaseDTO, ClientCaseEntity> {

    private final ClientCaseService clientCaseService;

    @PostConstruct
    public void initialize() {
        setTreeService(clientCaseService);
        setTreeEntity(ClientCaseEntity.class);
    }

    @PostMapping("/addExperience")
    public ResponseEntity<CommonResponse.ApiResult<ClientCaseEntity>> addClientCase(
            @Validated({AddNode.class}) ClientCaseDTO clientCaseDTO) throws Exception {

        ClientCaseEntity clientCaseEntity = modelMapper.map(clientCaseDTO, ClientCaseEntity.class);
        return ResponseEntity.ok(CommonResponse.success(clientCaseService.addClientCase(clientCaseEntity)));
    }

    @GetMapping("/getExperience/{clientCaseId}")
    public ResponseEntity<CommonResponse.ApiResult<ClientCaseEntity>> getClientCase(
            @PathVariable("clientCaseId") Long clientCaseId) throws Exception {

        return ResponseEntity.ok(CommonResponse.success(clientCaseService.getClientCase(clientCaseId)));
    }

    @PutMapping("/updateExperience")
    public ResponseEntity<CommonResponse.ApiResult<ClientCaseEntity>> updateClientCase(
            @Validated({UpdateNode.class}) ClientCaseDTO clientCaseDTO) throws Exception {

        ClientCaseEntity clientCaseEntity = modelMapper.map(clientCaseDTO, ClientCaseEntity.class);
        return ResponseEntity.ok(CommonResponse.success(clientCaseService.updateClientCase(clientCaseEntity)));
    }

    @DeleteMapping("/removeExperience/{clientCaseId}")
    public ResponseEntity<CommonResponse.ApiResult<ClientCaseEntity>> removeClientCase(
            @PathVariable("clientCaseId") Long clientCaseId) throws Exception {

        return ResponseEntity.ok(CommonResponse.success(clientCaseService.removeClientCase(clientCaseId)));
    }

    @GetMapping("/getExperienceList")
    public ResponseEntity<CommonResponse.ApiResult<Map<String, Object>>> getClientCaseList(
            @ModelAttribute ClientCaseDTO clientCaseDTO,
            @RequestParam(defaultValue = "1") int pageIndex,
            @RequestParam(defaultValue = "12") int pageUnit
    ) throws Exception {
        log.info("getClientCaseList - pageIndex={}, pageUnit={}", pageIndex, pageUnit);

        Map<String, Object> result = clientCaseService.getPaginatedClientCaseList(clientCaseDTO, pageIndex, pageUnit);
        return ResponseEntity.ok(CommonResponse.success(result));
    }
}