package com.arms.api.clientcase.service;

import com.arms.api.clientcase.model.ClientCaseDTO;
import com.arms.api.clientcase.model.ClientCaseEntity;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.modelmapper.ModelMapper;
import org.springframework.stereotype.Service;

import javax.persistence.EntityNotFoundException;
import javax.transaction.Transactional;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

@Slf4j
@RequiredArgsConstructor
@Service("clientCaseService")
public class ClientCaseServiceImpl extends TreeServiceImpl implements ClientCaseService {

    private final ModelMapper modelMapper;

    private final EngineService engineService;

    @Override
    @Transactional
    public ClientCaseEntity addClientCase(ClientCaseEntity clientCaseEntity) throws Exception {

        // Contents와 Thumbnail 백업 (Elasticsearch 전송용)
        String contentsBackup = clientCaseEntity.getC_clientcase_contents();
        String thumbnailBackup = clientCaseEntity.getC_clientcase_thumbnail_image();

        // MySQL에는 메타데이터만 저장 (Contents와 Thumbnail은 null)
        clientCaseEntity.setC_title(clientCaseEntity.getC_clientcase_title());
        clientCaseEntity.setRef(TreeConstant.First_Node_CID);
        clientCaseEntity.setC_type(TreeConstant.Leaf_Node_TYPE);
        clientCaseEntity.setC_clientcase_created(LocalDateTime.now());
        clientCaseEntity.setC_clientcase_contents(null);
        clientCaseEntity.setC_clientcase_thumbnail_image(null);

        // MySQL에 메타데이터만 저장
        ClientCaseEntity createdEntity = this.addNode(clientCaseEntity);

        // Elasticsearch에 Contents만 저장
        Map<String, Object> esData = new HashMap<>();
        esData.put("c_id", String.valueOf(createdEntity.getC_id()));
        esData.put("c_clientcase_contents", contentsBackup);
        esData.put("c_clientcase_thumbnail_image", thumbnailBackup);

        engineService.addClientCase(esData);

        return createdEntity;
    }

    @Override
    @Transactional
    public ClientCaseEntity getClientCase(Long clientCaseId) throws Exception {

        // MySQL에서 메타데이터 조회
        ClientCaseEntity clientCaseEntity = new ClientCaseEntity();
        clientCaseEntity.setC_id(clientCaseId);
        ClientCaseEntity existingClientCase = this.getNode(clientCaseEntity);

        // Elasticsearch에서 Contents와 Thumbnail 조회
        ClientCaseDTO dtoFromEngine = engineService.getClientCase(clientCaseId.toString()).getBody();
        if (dtoFromEngine != null) {
            existingClientCase.setC_clientcase_contents(dtoFromEngine.getC_clientcase_contents());
            existingClientCase.setC_clientcase_thumbnail_image(dtoFromEngine.getC_clientcase_thumbnail_image());
        }

        return existingClientCase;
    }

    @Override
    @Transactional
    public ClientCaseEntity updateClientCase(ClientCaseEntity clientCaseEntity) throws Exception {

        ClientCaseEntity preClientCase = this.getNode(clientCaseEntity);

        // Contents와 Thumbnail 백업 (Elasticsearch 전송용)
        String contentsBackup = clientCaseEntity.getC_clientcase_contents();
        String thumbnailBackup = clientCaseEntity.getC_clientcase_thumbnail_image();

        // MySQL에는 메타데이터만 업데이트 (Contents와 Thumbnail은 null)
        clientCaseEntity.setC_clientcase_contents(null);
        clientCaseEntity.setC_clientcase_thumbnail_image(null);
        clientCaseEntity.setC_clientcase_updated(LocalDateTime.now());

        int result = this.updateNode(clientCaseEntity);

        // Elasticsearch Contents만 업데이트
        Map<String, Object> esData = new HashMap<>();
        esData.put("c_id", String.valueOf(clientCaseEntity.getC_id()));
        esData.put("c_clientcase_contents", contentsBackup);
        esData.put("c_clientcase_thumbnail_image", thumbnailBackup);

        engineService.updateClientCase(esData);


        if (result == 1) {
            return this.getNode(clientCaseEntity);
        } else {
            return preClientCase;
        }
    }

    @Override
    @Transactional
    public ClientCaseEntity removeClientCase(Long clientCaseId) throws Exception {

        ClientCaseEntity key = new ClientCaseEntity();
        key.setC_id(clientCaseId);

        ClientCaseEntity pre = this.getNode(key);
        if (pre == null) {
            throw new EntityNotFoundException("해당 아이디의 ClientCase를 찾지 못하였음 : " + clientCaseId);
        }

        // Elasticsearch Contents 삭제
        engineService.deleteClientCase(clientCaseId.toString());

        // MySQL 메타데이터 삭제
        int result = this.removeNode(pre);
        if (result != 0) {
            throw new IllegalStateException("삭제 실패: id=" + clientCaseId + ", result=" + result);
        }

        return pre;
    }

    @Override
    @Transactional
    public Map<String, Object> getPaginatedClientCaseList(ClientCaseDTO clientCaseDTO, int pageIndex, int pageUnit) throws Exception {
        ClientCaseEntity clientCaseEntity = modelMapper.map(clientCaseDTO, ClientCaseEntity.class);

        // 루트 노드와 첫 번째 노드 제외
        ignoreRootAndFirstNode(clientCaseEntity);

        // 최신순 정렬
        orderByCreatedDesc(clientCaseEntity);

        // 페이지 정보 설정
        updatePageInfo(clientCaseEntity, pageIndex, pageUnit);

        // 페이징된 데이터 조회
        List<ClientCaseEntity> clientCaseList = this.getPaginatedChildNode(clientCaseEntity);

        // Entity를 DTO로 변환하여 Hibernate dirty checking 방지
        List<ClientCaseDTO> clientCaseDTOList = new java.util.ArrayList<>();

        if (clientCaseList != null && !clientCaseList.isEmpty()) {
            for (ClientCaseEntity entity : clientCaseList) {
                ClientCaseDTO dto = modelMapper.map(entity, ClientCaseDTO.class);

                // Elasticsearch에서 thumbnail 조회
                try {
                    ClientCaseDTO dtoFromEngine = engineService.getClientCase(entity.getC_id().toString()).getBody();
                    if (dtoFromEngine != null && dtoFromEngine.getC_clientcase_thumbnail_image() != null) {
                        dto.setC_clientcase_thumbnail_image(dtoFromEngine.getC_clientcase_thumbnail_image());
                    }
                } catch (Exception e) {
                    log.warn("[ClientCase-LIST] c_id: {} - Elasticsearch에서 thumbnail 조회 실패: {}", entity.getC_id(), e.getMessage());
                }

                clientCaseDTOList.add(dto);
            }
        }

        HashMap<String, Object> resultMap = new HashMap<>();
        if (!(clientCaseList == null || clientCaseList.isEmpty())) {
            resultMap.put("paginationInfo", clientCaseEntity.getPaginationInfo());
        } else {
            resultMap.put("paginationInfo", Map.of("totalRecordCount", 0, "lastPageNo", 0));
        }
        resultMap.put("data", clientCaseDTOList);

        return resultMap;
    }


    /**
     * 루트 노드와 첫 번째 노드 제외 (root, drive)
     */
    private void ignoreRootAndFirstNode(ClientCaseEntity clientCaseEntity) {
        clientCaseEntity.getCriterions().add(Restrictions.not(
                Restrictions.in("c_id", TreeConstant.ROOT_CID, TreeConstant.First_Node_CID)));
    }

    /**
     * 생성일시 기준 내림차순 정렬
     */
    private void orderByCreatedDesc(ClientCaseEntity clientCaseEntity) {
        clientCaseEntity.getOrder().add(Order.desc("c_clientcase_created"));
    }

    /**
     * 페이지 정보 설정
     */
    private void updatePageInfo(ClientCaseEntity clientCaseEntity, int pageIndex, int pageUnit) {
        clientCaseEntity.setPageIndex(pageIndex);
        clientCaseEntity.setPageUnit(pageUnit);
    }

}
