package com.arms.api.dashboard.service;

import com.arms.api.analysis.resource.model.dto.ResourceDTO;
import com.arms.api.analysis.resource.model.dto.ResourceWithVersionIdNamesDTO;
import com.arms.api.analysis.resource.model.dto.VersionIdNameDTO;
import com.arms.api.analysis.resource.model.vo.treemap.TreeMapWorkerVO;
import com.arms.api.analysis.resource.service.ResourceService;
import com.arms.api.dashboard.model.RequirementJiraIssueAggregationResponse;
import com.arms.api.dashboard.model.SankeyData;
import com.arms.api.dashboard.model.dto.DashboardDTO;
import com.arms.api.dashboard.model.vo.Top5AssigneeReqStatusInfoVO;
import com.arms.api.dashboard.model.vo.TopNReqAssigneeReqLinkVO;
import com.arms.api.dashboard.model.vo.TopNReqAssigneeVO;
import com.arms.api.product_service.pdserviceversion.service.PdServiceVersion;
import com.arms.api.requirement.reqadd.model.entity.ReqAddEntity;
import com.arms.api.requirement.reqadd.service.ReqAdd;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import com.arms.egovframework.javaservice.treeframework.util.StringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Restrictions;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@Service
@RequiredArgsConstructor
public class DashboardServiceImpl implements DashboardService {

    private final AggregationService aggregationService;

    private final PdServiceVersion pdServiceVersion;

    private final ReqAdd reqAdd;

    private final ResourceService resourceService;

    @Override
    public Map<String, RequirementJiraIssueAggregationResponse> requirementsJiraIssueStatuses(DashboardDTO dashboardDTO) {

        return aggregationService.aggregateWeeklyReqIssueStatus(dashboardDTO).getBody();
    }

    @Override
    public SankeyData sankeyChartAPIWithTopN(DashboardDTO dashboardDTO, Integer topN) throws Exception {

        ResourceDTO resourceDTO = new ResourceDTO();
        resourceDTO.setPdServiceAndIsReq(dashboardDTO.getPdServiceAndIsReq());

        return resourceService.sankeyChartAPIWithTopN(resourceDTO, topN);
    }


    @Override
    public List<TreeMapWorkerVO> findTreeMapChartDataV3(ResourceWithVersionIdNamesDTO resourceWithVersionIdNamesDTO) throws Exception {

        PdServiceAndIsReqDTO pdServiceAndIsReq = resourceWithVersionIdNamesDTO.getPdServiceAndIsReq();
        List<VersionIdNameDTO> versionIdNames = pdServiceVersion.getVersionListByCids(pdServiceAndIsReq.getPdServiceVersionLinks())
                .stream()
                .map(entity -> {
                    VersionIdNameDTO dto = new VersionIdNameDTO();
                    dto.setC_id(String.valueOf(entity.getC_id()));
                    dto.setC_title(entity.getC_title());
                    return dto;
                }).collect(Collectors.toList());
        resourceWithVersionIdNamesDTO.setVersionIdNames(versionIdNames);

        return resourceService.findTreeMapChartDataV3(resourceWithVersionIdNamesDTO);
    }

    @Override
    public List<TopNReqAssigneeVO> findTopNReqAssignee(DashboardDTO dashboardDTO, Integer topN) {
        return aggregationService.topNReqAssignee(dashboardDTO, topN);
    }

    @Override
    public List<Top5AssigneeReqStatusInfoVO> findTopNAssigneeReqStateInfo(String changeReqTableName, DashboardDTO dashboardDTO, Integer topN) throws Exception {

        // 1. (K-V) 요구사항 아이디 - 상태 아이디
        Map<Long, Long> reqLinkReqStateMap = getReqLinkReqState(changeReqTableName, dashboardDTO);

        // 2. 담당자 아이디 : 요구사항 아이디 목록 맵 (작업자별, 요구사항 아이디 목록 조회)
        List<TopNReqAssigneeReqLinkVO> topNReqAssigneeReqLinkVOS = aggregationService.topNAssigneeReqLinks(dashboardDTO, topN);

        // 3. 작업자별 상태 아이디 : 상태 개수를 집계
        List<Top5AssigneeReqStatusInfoVO> result = new ArrayList<>();

        for (TopNReqAssigneeReqLinkVO topNReqAssigneeReqLinkVO : topNReqAssigneeReqLinkVOS) {
            String nameWithEmailId = topNReqAssigneeReqLinkVO.getNameWithEmailId();
            List<Long> reqLinks = topNReqAssigneeReqLinkVO.getReqLinks().stream().map(Long::parseLong).collect(Collectors.toList());
            Map<Long, Long> statusCounts = calculateStatusCounts(reqLinks, reqLinkReqStateMap);
            // 4. vo로 반환
            statusCounts.forEach((statusId, count) -> {
                result.add(Top5AssigneeReqStatusInfoVO.builder()
                        .assignee(nameWithEmailId)
                        .issueStatusId(statusId)
                        .issueStatusCount(count)
                        .build());
            });
        }

        return result;
    }

    // 요구사항 목록에서 상태별 개수를 계산하는 메서드
    private Map<Long, Long> calculateStatusCounts(List<Long> reqLinkIds, Map<Long, Long> reqLinkStatusMap) {
        Map<Long, Long> statusCountMap = new HashMap<>();
        for (Long reqLinkId : reqLinkIds) {
            if (reqLinkStatusMap.containsKey(reqLinkId)) {
                Long statusId = reqLinkStatusMap.get(reqLinkId);
                if (statusId != null) {
                    statusCountMap.put(statusId, statusCountMap.getOrDefault(statusId, 0L) + 1);
                }
            }
        }
        return statusCountMap;
    }

    private Map<Long, Long> getReqLinkReqState(String changeReqTableName, DashboardDTO dashboardDTO) throws Exception {

        SessionUtil.setAttribute("getReqAddListByFilter",changeReqTableName);

        List<Long> pdServiceVersionLinks = dashboardDTO.pdServiceVersionLinks();

        ReqAddEntity searchEntity = new ReqAddEntity();

        if (pdServiceVersionLinks != null && !pdServiceVersionLinks.isEmpty()) {
            Disjunction orCondition = Restrictions.disjunction();
            for (Long version : pdServiceVersionLinks) {
                String versionString = "\\\"" + String.valueOf(version) + "\\\"";
                orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", versionString, MatchMode.ANYWHERE));
            }
            searchEntity.getCriterions().add(orCondition);
        }

        List<ReqAddEntity> searchResult = reqAdd.getChildNode(searchEntity);

        Map<Long, Long> reqLinkReqStateMap = new HashMap<>();
        for(ReqAddEntity entity : searchResult) {
            if(StringUtils.equals(entity.getC_type(),"default")) {
                Long reqAddId = entity.getC_id();
                Long stateId = Optional.ofNullable(entity)
                        .map(e -> e.getReqStateEntity())
                        .map(e -> e.getC_id())
                        .orElse(null);
                reqLinkReqStateMap.put(reqAddId,stateId);
            }
        }

        SessionUtil.removeAttribute("getReqAddListByFilter");

        return reqLinkReqStateMap;
    }
}

