/*
 * @author Dongmin.lee
 * @since 2023-03-28
 * @version 23.03.28
 * @see <pre>
 *  Copyright (C) 2007 by 313 DEV GRP, Inc - All Rights Reserved
 *  Unauthorized copying of this file, via any medium is strictly prohibited
 *  Proprietary and confidential
 *  Written by 313 developer group <313@313.co.kr>, December 2010
 * </pre>
 */
package com.arms.api.jira.jiraserver.controller;

import com.arms.api.jira.jiraserver.model.JiraServerDTO;
import com.arms.api.jira.jiraserver.model.JiraServerEntity;
import com.arms.api.jira.jiraserver.model.enums.EntityType;
import com.arms.api.jira.jiraserver.service.JiraServer;
import com.arms.api.util.aes.AES256Encryption;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.controller.TreeAbstractController;
import com.arms.egovframework.javaservice.treeframework.util.ParameterParser;
import com.arms.egovframework.javaservice.treeframework.validation.group.AddNode;
import com.arms.egovframework.javaservice.treeframework.validation.group.UpdateNode;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import java.util.concurrent.CompletableFuture;

@Slf4j
@Controller
@RequestMapping(value = {"/arms/jiraServer"})
public class JiraServerController extends TreeAbstractController<JiraServer, JiraServerDTO, JiraServerEntity> {

    @Autowired
    @Qualifier("jiraServer")
    private JiraServer jiraServer;

    @PostConstruct
    public void initialize() {
        setTreeService(jiraServer);
        setTreeEntity(JiraServerEntity.class);
    }

    @Autowired
    private AES256Encryption aes256Encryption;

    @ResponseBody
    @RequestMapping(
            value = { "/addJiraServerNode.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> addJiraServerNode(@Validated({AddNode.class}) JiraServerDTO jiraServerDTO,
                                            BindingResult bindingResult, ModelMap model) throws  Exception {

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);
        JiraServerEntity result = jiraServer.addJiraServer(jiraServerEntity);

        CompletableFuture<Boolean> future = jiraServer.refreshALMServer(result);
        future.thenAccept(process -> {
            try {
                jiraServer.cacheStatusMappingData(jiraServerEntity);
            }
            catch (Exception e) {
                log.error(e.getMessage());
                chat.sendMessageByEngine("캐쉬 서버 업데이트 중 오류가 발생하였습니다. :: " + e.getMessage());
            }
        });

        return ResponseEntity.ok(CommonResponse.success(result));
    }

    @ResponseBody
    @RequestMapping(
            value= { "/getNodeWithEncryption.do"},
            method= {RequestMethod.GET}
    )
    public ResponseEntity<String> getNodeWithEncryption(JiraServerDTO jiraServerDTO) throws Exception {

        JiraServerEntity jiraServerEntity = jiraServer.getNode(modelMapper.map(jiraServerDTO, JiraServerEntity.class));

        String encryptedAlmConnPw = aes256Encryption.encrypt(jiraServerEntity.getC_jira_server_connect_pw());

        return ResponseEntity.ok(encryptedAlmConnPw);
    }



    @ResponseBody
    @RequestMapping(
            value={"{refreshTarget}/renewNode.do"},
            method = {RequestMethod.PUT }
    )
    public ResponseEntity<?> refreshALMServerByTarget(@PathVariable String refreshTarget,
                                                String projectCId,
                                                JiraServerDTO jiraServerDTO) throws Exception{

        log.info("JiraServerController :: refreshALMServerByTarget refreshTarget=> {}", refreshTarget);
        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        EntityType entityType = EntityType.fromString(refreshTarget);
        JiraServerEntity result = jiraServer.서버_엔티티_항목별_갱신(entityType, projectCId, jiraServerEntity);

        if (entityType == EntityType.이슈상태) {
            jiraServer.cacheStatusMappingData(result);
        }

        return ResponseEntity.ok(CommonResponse.success(result));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/getJiraProjectPure.do"},
            method={RequestMethod.GET}
    )
    public ResponseEntity<?> getJiraProjectPure(JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception {

        log.info("JiraServerController :: getJiraProjectPure");

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(CommonResponse.success(jiraServer.getProjectPureList(jiraServerEntity)));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/getJiraIssueType.do"},
            method={RequestMethod.GET}
    )
    public ResponseEntity<?> getJiraIssueTypeList(JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception {

        log.info("JiraServerController :: getJiraIssueTypeList");

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(CommonResponse.success(jiraServer.getIssueTypeOnlyList(jiraServerEntity)));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/getIssueStatusList"},
            method={RequestMethod.GET}
    )
    public ResponseEntity<?> getIssueStatusList(JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception {

        log.info("JiraServerController :: getIssueStatusList");

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(CommonResponse.success(jiraServer.getServerIssueStatusOnlyList(jiraServerEntity)));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/getJiraIssuePriority.do"},
            method={RequestMethod.GET}
    )
    public ResponseEntity<?> getJiraIssuePriorityList(JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception {

        log.info("JiraServerController :: getJiraIssuePriorityList");

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(CommonResponse.success(jiraServer.getServerIssuePriorityOnlyList(jiraServerEntity)));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/getJiraIssueResolution.do"},
            method={RequestMethod.GET}
    )
    public ResponseEntity<?> getJiraIssueResolutionList(JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception {

        log.info("JiraServerController :: getJiraIssueResolutionList");

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(CommonResponse.success(jiraServer.getServerIssueResolutionList(jiraServerEntity)));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{defaultTarget}/makeDefault.do"},
            method = {RequestMethod.PUT}
    )
    public ResponseEntity<?> setDefaultValueForTarget(@PathVariable(name="defaultTarget") String targetEntity,
                                                    @RequestParam(name= "targetCId") Long targetCId,
                                                    JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception{

        log.info("JiraServerController :: setDefaultValueForTarget, 설정할_항목 : {}, 항목_c_id : {}", targetEntity, targetCId);

        ParameterParser parser = new ParameterParser(request);
        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }
        if (parser.getInt("targetCId") <= 0) {
            throw new RuntimeException("targetCId is minus value");
        }

        EntityType entityType = EntityType.fromString(targetEntity);

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(CommonResponse.success(jiraServer.setDefaultValueForTarget(entityType, targetCId, jiraServerEntity)));
    }

    @ResponseBody
    @RequestMapping(value = "/updateNodeAndEngineServerInfoUpdate.do", method = RequestMethod.PUT)
    public ModelAndView updateArmsAndEngineServerData(@Validated(value = UpdateNode.class) JiraServerDTO JiraServerDTO,
                                        BindingResult bindingResult, HttpServletRequest request, ModelMap model) throws Exception {

        log.info("JiraServerController :: updateArmsAndEngineServerData");
        JiraServerEntity treeSearchEntity = modelMapper.map(JiraServerDTO, JiraServerEntity.class);

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", jiraServer.updateArmsAndEngineServerData(treeSearchEntity));
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(
            value = {"/refresh.do"},
            method={RequestMethod.PUT}
    )
    public ResponseEntity<?> refreshALMServer(JiraServerDTO jiraServerDTO) throws Exception {

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        CompletableFuture<Boolean> future = jiraServer.refreshALMServer(jiraServerEntity);
        future.thenAccept(process -> {
            try {
                jiraServer.cacheStatusMappingData(jiraServerEntity);
            }
            catch (Exception e) {
                log.error(e.getMessage());
                chat.sendMessageByEngine("캐쉬 서버 업데이트 중 오류가 발생하였습니다. :: " + e.getMessage());
            }
        });

        return ResponseEntity.ok(jiraServerEntity);
    }

    @ResponseBody
    @RequestMapping(
            value = {"/getRuleSet.do"},
            method={RequestMethod.GET}
    )
    public ResponseEntity<?> getStatusRuleSet(JiraServerDTO jiraServerDTO, HttpServletRequest request) throws Exception {

        log.info("JiraServerController :: getStatusRuleSet");

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);

        return ResponseEntity.ok(jiraServer.getStatusRuleSet(jiraServerEntity));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/preSetIssueStatusMapping.do"},
            method={RequestMethod.POST}
    )
    public ResponseEntity<?> preSetIssueStatusMapping(String projectCId, JiraServerDTO jiraServerDTO) throws Exception {

        log.info("JiraServerController :: preSetIssueStatusMapping 프리셋 대상 프로젝트 아이디: {}, 대상 서버 아이디: {}",projectCId,jiraServerDTO.getC_id());

        JiraServerEntity jiraServerEntity = modelMapper.map(jiraServerDTO, JiraServerEntity.class);
        jiraServer.preSetIssueStatusMapping(projectCId, jiraServerEntity);

        return ResponseEntity.ok(jiraServerEntity);
    }

}
