/*
 * @author Dongmin.lee
 * @since 2023-03-28
 * @version 23.03.28
 * @see <pre>
 *  Copyright (C) 2007 by 313 DEV GRP, Inc - All Rights Reserved
 *  Unauthorized copying of this file, via any medium is strictly prohibited
 *  Proprietary and confidential
 *  Written by 313 developer group <313@313.co.kr>, December 2010
 * </pre>
 */
package com.arms.api.jira.jiraserver.service;

import com.arms.api.globaltreemap.model.GlobalTreeMapEntity;
import com.arms.api.globaltreemap.service.GlobalTreeMapService;
import com.arms.api.jira.jiraissuepriority.model.JiraIssuePriorityEntity;
import com.arms.api.jira.jiraissuepriority.service.JiraIssuePriority;
import com.arms.api.jira.jiraissueresolution.model.JiraIssueResolutionEntity;
import com.arms.api.jira.jiraissueresolution.service.JiraIssueResolution;
import com.arms.api.jira.jiraissuestatus.model.JiraIssueStatusEntity;
import com.arms.api.jira.jiraissuestatus.model.StatusRuleSetEntity;
import com.arms.api.jira.jiraissuestatus.service.JiraIssueStatus;
import com.arms.api.jira.jiraissuestatuscategorymap.model.JiraIssueStatusCategoryMapEntity;
import com.arms.api.jira.jiraissuestatuscategorymap.service.JiraIssueStatusCategoryMap;
import com.arms.api.jira.jiraissuetype.model.JiraIssueTypeEntity;
import com.arms.api.jira.jiraissuetype.service.JiraIssueType;
import com.arms.api.jira.jiraproject.model.JiraProjectEntity;
import com.arms.api.jira.jiraproject.service.JiraProject;
import com.arms.api.jira.jiraproject_pure.model.JiraProjectPureEntity;
import com.arms.api.jira.jiraserver.model.JiraServerEntity;
import com.arms.api.jira.jiraserver.model.enums.EntityType;
import com.arms.api.jira.jiraserver.model.enums.ServerType;
import com.arms.api.jira.jiraserver.model.pure.JiraServerIssuePriorityOnlyEntity;
import com.arms.api.jira.jiraserver.model.pure.JiraServerIssueStatusOnlyEntity;
import com.arms.api.jira.jiraserver.model.pure.JiraServerIssueTypeOnlyEntity;
import com.arms.api.jira.jiraserver.model.pure.JiraServerProjectOnlyEntity;
import com.arms.api.util.aes.AES256Encryption;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.external.MiddleProxyService;
import com.arms.api.util.communicate.external.request.redis.AlmDataRequestDTO;
import com.arms.api.util.communicate.external.request.redis.IssueStatusDTO;
import com.arms.api.util.communicate.external.request.ServerInfo;
import com.arms.api.util.communicate.external.response.jira.*;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.model.TreeSearchEntity;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import com.arms.egovframework.javaservice.treeframework.util.StringUtils;
import com.arms.egovframework.javaservice.treeframework.util.Util_TitleChecker;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Restrictions;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.security.SecureRandom;
import java.util.*;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.arms.egovframework.javaservice.treeframework.remote.Global.chat;


@Slf4j
@AllArgsConstructor
@Service("jiraServer")
public class JiraServerImpl extends TreeServiceImpl implements JiraServer {

    private static final SecureRandom RANDOM = new SecureRandom();

    private final GlobalTreeMapService globalTreeMapService;
    private final EngineService engineService;
    private final MiddleProxyService middleProxyService;

    private final JiraProject jiraProject;
    private final JiraIssueType jiraIssueType;
    private final JiraIssueStatus jiraIssueStatus;
    private final JiraIssuePriority jiraIssuePriority;
    private final JiraIssueResolution jiraIssueResolution;
    private final JiraIssueStatusCategoryMap jiraIssueStatusCategoryMap;

    private final AES256Encryption aes256Encryption;

    @Override
    @Transactional
    public JiraServerEntity setDefaultValueForTarget(EntityType entityType, Long targetCId, JiraServerEntity searchEntity) throws Exception {

        JiraServerEntity jiraServerEntity = this.getNode(searchEntity);

        if (jiraServerEntity == null) {
            throw new Exception(entityType + " 항목 기본값 설정에 필요한 서버 정보가 없습니다. :: " + targetCId);
        }

        //이슈 유형
        if (EntityType.이슈유형 == entityType) {
            Optional.ofNullable(jiraServerEntity.getJiraIssueTypeEntities())
                    .orElse(Collections.emptySet())
                    .forEach(issueType -> {
                        if (Objects.equals(issueType.getC_id(), targetCId)) {
                            issueType.setC_check("true");
                        } else {
                            issueType.setC_check("false");
                        }
                    });
        }

        //이슈 상태
        if (EntityType.이슈상태 == entityType) {
            Optional.ofNullable(jiraServerEntity.getJiraIssueStatusEntities())
                    .orElse(Collections.emptySet())
                    .forEach(issueStatus -> {
                        if (Objects.equals(issueStatus.getC_id(), targetCId)) {
                            issueStatus.setC_check("true");
                        } else {
                            issueStatus.setC_check("false");
                        }
                    });
        }

        //이슈 우선순위
        if (EntityType.이슈우선순위 == entityType) {
            Optional.ofNullable(jiraServerEntity.getJiraIssuePriorityEntities())
                    .orElse(Collections.emptySet())
                    .forEach(issuePriority -> {
                        if (Objects.equals(issuePriority.getC_id(), targetCId)) {
                            issuePriority.setC_check("true");
                        } else {
                            issuePriority.setC_check("false");
                        }
                    });
        }

        //이슈 해결책
        if (EntityType.이슈해결책 == entityType) {
            Optional.ofNullable(jiraServerEntity.getJiraIssueResolutionEntities())
                    .orElse(Collections.emptySet())
                    .forEach(issueResolution -> {
                        if (Objects.equals(issueResolution.getC_id(), targetCId)) {
                            issueResolution.setC_check("true");
                        } else {
                            issueResolution.setC_check("false");
                        }
                    });
        }

        this.updateNode(jiraServerEntity);

        return jiraServerEntity;
    }

    @Override
    public List<JiraProjectPureEntity> getProjectPureList(JiraServerEntity jiraServerEntity) throws Exception {

        JiraServerProjectOnlyEntity searchEntity = new JiraServerProjectOnlyEntity();
        searchEntity.setC_id(jiraServerEntity.getC_id());

        JiraServerProjectOnlyEntity jiraServerProjectOnlyEntity = this.getNode(searchEntity);

        List<JiraProjectPureEntity> projectPureList =
                Optional.ofNullable(jiraServerProjectOnlyEntity.getJiraProjectPureEntities())
                        .orElseGet(Collections::emptySet)
                        .stream()
                        .filter(Objects::nonNull)
                        .filter(project -> project.getC_etc() == null || !StringUtils.equals("delete", project.getC_etc()))
                        .collect(Collectors.toList());

        return projectPureList;
    }

    @Override
    public List<JiraIssueTypeEntity> getIssueTypeOnlyList(JiraServerEntity jiraServerEntity) throws Exception {

        JiraServerIssueTypeOnlyEntity searchEntity = new JiraServerIssueTypeOnlyEntity();
        searchEntity.setC_id(jiraServerEntity.getC_id());

        JiraServerIssueTypeOnlyEntity jiraServerIssueTypeOnlyEntity = this.getNode(searchEntity);

        List<JiraIssueTypeEntity> issueTypeList = Optional.ofNullable(jiraServerIssueTypeOnlyEntity.getJiraIssueTypeEntities())
                .orElseGet(Collections::emptySet)
                .stream()
                .filter(Objects::nonNull)
                .filter(issueType -> issueType.getC_etc() == null || !StringUtils.equals("delete", issueType.getC_etc()))
                .filter(issueType -> issueType.getC_desc() == null || !StringUtils.equals("true", issueType.getC_desc()))
                .collect(Collectors.toList());

        return issueTypeList;
    }

    @Override
    public List<JiraIssueStatusEntity> getServerIssueStatusOnlyList(JiraServerEntity jiraServerEntity) throws Exception {

        JiraServerIssueStatusOnlyEntity searchEntity = new JiraServerIssueStatusOnlyEntity();
        searchEntity.setC_id(jiraServerEntity.getC_id());

        JiraServerIssueStatusOnlyEntity jiraServerIssueStatusOnlyEntity = this.getNode(searchEntity);

        List<JiraIssueStatusEntity> jiraIssueStatusList = Optional.ofNullable(jiraServerIssueStatusOnlyEntity.getJiraIssueStatusEntities())
                .orElseGet(Collections::emptySet)
                .stream()
                .filter(Objects::nonNull)
                .filter(issueStatus -> issueStatus.getC_etc() == null || !StringUtils.equals("delete", issueStatus.getC_etc()))
                .collect(Collectors.toList());

        return jiraIssueStatusList;
    }

    @Override
    public List<JiraIssuePriorityEntity> getServerIssuePriorityOnlyList(JiraServerEntity jiraServerEntity) throws Exception {

        JiraServerIssuePriorityOnlyEntity searchEntity = new JiraServerIssuePriorityOnlyEntity();
        searchEntity.setC_id(jiraServerEntity.getC_id());

        JiraServerIssuePriorityOnlyEntity jiraServerIssuePriorityOnlyEntity = this.getNode(searchEntity);

        List<JiraIssuePriorityEntity> issuePriorityList = Optional.ofNullable(jiraServerIssuePriorityOnlyEntity.getJiraIssuePriorityEntities())
                .orElseGet(Collections::emptySet)
                .stream()
                .filter(Objects::nonNull)
                .filter(issuePriority -> issuePriority.getC_etc() == null || !StringUtils.equals("delete", issuePriority.getC_etc()))
                .collect(Collectors.toList());

        return issuePriorityList;
    }

    @Override
    public List<JiraIssueResolutionEntity> getServerIssueResolutionList(JiraServerEntity jiraServerEntity) throws Exception {
        GlobalTreeMapEntity searchEntity = new GlobalTreeMapEntity();
        searchEntity.setJiraserver_link(jiraServerEntity.getC_id());
        List<GlobalTreeMapEntity> globalTreeMapList = globalTreeMapService.findAllBy(searchEntity);

        List<Long> issueResolutionCIdList = new ArrayList<>();
        for (GlobalTreeMapEntity globalTreeMapEntity : globalTreeMapList) {
            if (globalTreeMapEntity.getJiraissueresolution_link() != null) {
                issueResolutionCIdList.add(globalTreeMapEntity.getJiraissueresolution_link());
            }
        }

        JiraIssueResolutionEntity issueResolutionEntity = new JiraIssueResolutionEntity();
        Criterion criterion1 = Restrictions.in("c_id", issueResolutionCIdList);
        Criterion criterion2 = Restrictions.or(Restrictions.isNull("c_etc"), Restrictions.ne("c_etc", "delete"));
        Criterion criterion = Restrictions.and(criterion1, criterion2);
        issueResolutionEntity.getCriterions().add(criterion);

        List<JiraIssueResolutionEntity> issueResolutionList = jiraIssueResolution.getChildNode(issueResolutionEntity);

        return issueResolutionList;
    }

    @Override
    @Transactional
    public int updateArmsAndEngineServerData(JiraServerEntity updateServerEntity) throws Exception {
        // 수정대상의 PW가 null 과 공란이 아님 > 암호화하여 새로 업데이트 할 대상
        if (updateServerEntity.getC_jira_server_connect_pw() != null && !updateServerEntity.getC_jira_server_connect_pw().trim().isEmpty()) {
            updateServerEntity.setC_jira_server_connect_pw(aes256Encryption.encrypt(updateServerEntity.getC_jira_server_connect_pw()));
        }

        int result = this.updateNode(updateServerEntity);
        JiraServerEntity jiraServerEntity = this.getNode(updateServerEntity);

        ServerInfo serverInfo = new ServerInfo();
        serverInfo.setConnectId(jiraServerEntity.getC_jira_server_etc());
        serverInfo.setType(jiraServerEntity.getC_jira_server_type());
        serverInfo.setUri(jiraServerEntity.getC_jira_server_base_url());
        serverInfo.setUserId(jiraServerEntity.getC_jira_server_connect_id());
        serverInfo.setPasswordOrToken(jiraServerEntity.getC_jira_server_connect_pw());

        /**
         * 엔진 - 암스 오류발생 시 동기화 필요
         **/
        ServerInfoEntity updateResult = engineService.registerOrUpdateAlmServerInfo(serverInfo);

        if (updateResult != null) {
            log.info(" [ " + this.getClass().getName() + " :: 암스_및_엔진_서버정보수정 ] :: 수정결과 -> " + updateResult.toString() );
        }

        return result;
    }

    @Override
    @Transactional
    public JiraServerEntity addJiraServer(JiraServerEntity jiraServerEntity) throws Exception {
        // 프로젝트 생성 //
        String randomConnectId = String.valueOf(RANDOM.nextLong() & Long.MAX_VALUE);
        // pw 암호화
        if (jiraServerEntity.getC_jira_server_connect_pw() != null && !jiraServerEntity.getC_jira_server_connect_pw().trim().isEmpty()) {
            jiraServerEntity.setC_jira_server_connect_pw(aes256Encryption.encrypt(jiraServerEntity.getC_jira_server_connect_pw()));
        }

        String serverUrl = StringUtils.removeTrailingSlash(jiraServerEntity.getC_jira_server_base_url());
//        서버는 1개만 등록 시킬 지 여부
//        if (serverUrl != null) {
//            JiraServerEntity searchServerEntity = new JiraServerEntity();
//            searchServerEntity.getCriterions().add(Restrictions.eq("c_jira_server_base_url", serverUrl));
//            List<JiraServerEntity> jiraServerEntities = this.getChildNode(searchServerEntity);
//            if (jiraServerEntities.size() > 0) {
//                throw new Exception("이미 등록된 서버입니다.");
//            }
//        }

        jiraServerEntity.setC_jira_server_base_url(serverUrl);
        jiraServerEntity.setC_title(Util_TitleChecker.StringReplace(jiraServerEntity.getC_title()));
        jiraServerEntity.setC_jira_server_etc(randomConnectId); // 엔진과 통신할 connectId

        JiraServerEntity addedNodeEntity = this.addNode(jiraServerEntity);

        ServerInfo serverInfo = new ServerInfo();
        serverInfo.setConnectId(randomConnectId);
        serverInfo.setType(addedNodeEntity.getC_jira_server_type());
        serverInfo.setUri(addedNodeEntity.getC_jira_server_base_url());
        serverInfo.setUserId(addedNodeEntity.getC_jira_server_connect_id());
        serverInfo.setPasswordOrToken(addedNodeEntity.getC_jira_server_connect_pw());
        ServerInfoEntity result = engineService.registerOrUpdateAlmServerInfo(serverInfo);

        if (result != null) {
            log.info(" [ 암스_및_엔진_서버정보등록 ] :: 등록결과 -> " + result.toString() );
            chat.sendMessageByEngine("서버 등록이 완료되었습니다. ALM 서버 데이터 갱신에 시간이 걸릴 수 있습니다.");
        }

        return addedNodeEntity;
    }

    @Override
    @Transactional
    public JiraServerEntity 서버_엔티티_항목별_갱신(EntityType 갱신할_항목, String projectCId, JiraServerEntity jiraServerEntity) throws Exception {
        JiraServerEntity 검색용_서버_엔티티 = new JiraServerEntity();
        검색용_서버_엔티티.setC_id(jiraServerEntity.getC_id());

        JiraServerEntity 검색된_ALM_서버 = this.getNode(검색용_서버_엔티티);

        if (검색된_ALM_서버 == null
                || 검색된_ALM_서버.getC_jira_server_type() == null
                || 검색된_ALM_서버.getC_jira_server_etc() == null) {
            throw new Exception("갱신에 필요한 서버 정보가 없습니다.");
        }

        String 서버유형 = 검색된_ALM_서버.getC_jira_server_type();
        log.info("[ JiraServerImpl :: 서버_엔티티_항목별_갱신 ] :: 갱신할_항목 → {}, 서버유형 → {}", 갱신할_항목.getType(), 서버유형);

        String 갱신_메세지;
        switch (갱신할_항목) {
            case 프로젝트:
                갱신_메세지 = this.프로젝트_갱신(검색된_ALM_서버);
                break;
            case 이슈유형:
                갱신_메세지 = this.이슈유형_갱신(검색된_ALM_서버, projectCId);
                break;
            case 이슈상태:
                갱신_메세지 = this.이슈상태_갱신(검색된_ALM_서버, projectCId);
                break;
            case 이슈우선순위:
                갱신_메세지 = this.이슈우선순위_갱신(검색된_ALM_서버);
                break;
            case 이슈해결책:
                갱신_메세지 = this.이슈해결책_갱신(검색된_ALM_서버);
                break;
            default:
                throw new IllegalArgumentException("알 수 없는 갱신 항목: " + 갱신할_항목);
        }

        if (갱신_메세지 == null) {
            this.updateNode(검색된_ALM_서버);
            갱신_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신되었습니다.";
        }

        chat.sendMessageByEngine(갱신_메세지);

        return 검색된_ALM_서버;
    }

    private String 프로젝트_갱신(JiraServerEntity 검색된_ALM_서버) {
        EntityType 갱신할_항목 = EntityType.프로젝트;
        String 엔진_통신_아이디 = 검색된_ALM_서버.getC_jira_server_etc();

        Set<JiraProjectEntity> 해당_서버_프로젝트_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraProjectEntities())
                .orElse(new HashSet<>());

        Map<String, JiraProjectEntity> 기존프로젝트_맵 = 해당_서버_프로젝트_목록.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(프로젝트 -> 프로젝트.getC_jira_url(), 프로젝트 -> 프로젝트));

        List<지라프로젝트_데이터> 가져온_프로젝트_목록;
        try {
            가져온_프로젝트_목록 = engineService.ALM_프로젝트_목록_가져오기(엔진_통신_아이디);
        }
        catch (Exception e) {
            String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
            log.error(오류_메세지);
            return 오류_메세지;
        }

        Set<JiraProjectEntity> 프로젝트_동기화목록 = 서버_엔티티_동기화(해당_서버_프로젝트_목록, 기존프로젝트_맵, 가져온_프로젝트_목록,
                검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

        검색된_ALM_서버.setJiraProjectEntities(프로젝트_동기화목록);
        return null;
    }

    private String 이슈유형_갱신(JiraServerEntity 검색된_ALM_서버, String 프로젝트_C아이디) throws Exception {
        EntityType 갱신할_항목 = EntityType.이슈유형;
        String 엔진_통신_아이디 = 검색된_ALM_서버.getC_jira_server_etc();
        String 서버유형 = 검색된_ALM_서버.getC_jira_server_type();

        if (StringUtils.equals(ServerType.JIRA_ON_PREMISE.getType(), 서버유형)) {
            Set<JiraIssueTypeEntity> 해당_서버_이슈유형_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraIssueTypeEntities())
                    .orElse(new HashSet<>());

            Map<String, JiraIssueTypeEntity> 기존이슈유형_맵 = 해당_서버_이슈유형_목록.stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toMap(이슈유형 -> 이슈유형.getC_issue_type_url(), 이슈유형 -> 이슈유형));

            List<지라이슈유형_데이터> 가져온_이슈유형_목록;
            try {
                가져온_이슈유형_목록 = engineService.ALM_이슈_유형_가져오기(엔진_통신_아이디);
            }
            catch (Exception e) {
                String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
                log.error(오류_메세지);
                return 오류_메세지;
            }

            Set<JiraIssueTypeEntity> 이슈유형_동기화목록 = 서버_엔티티_동기화(해당_서버_이슈유형_목록, 기존이슈유형_맵, 가져온_이슈유형_목록,
                    검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

            검색된_ALM_서버.setJiraIssueTypeEntities(이슈유형_동기화목록);
        }
        else if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버유형) || StringUtils.equals(ServerType.REDMINE_ON_PREMISE.getType(), 서버유형)) {

            // 프로젝트_아이디가 null - 서버 등록. 전체 프로젝트에 대한 이슈유형을 갱신시키는 로직
            // 프로젝트 아이디가 있는 경우 - 해당 프로젝트만 이슈유형을 갱신시키는 로직
            if (프로젝트_C아이디 == null || 프로젝트_C아이디.isEmpty()) {
                Set<JiraProjectEntity> 해당_서버_프로젝트_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraProjectEntities())
                        .orElse(new HashSet<>());

                for (JiraProjectEntity 프로젝트 : 해당_서버_프로젝트_목록) {
                    if (프로젝트.getC_etc() != null && StringUtils.equals(프로젝트.getC_etc(), "delete")) {
                        continue;
                    }

                    Set<JiraIssueTypeEntity> 프로젝트의_이슈유형_목록 = Optional.ofNullable(프로젝트.getJiraIssueTypeEntities())
                            .orElse(new HashSet<>());

                    Map<String, JiraIssueTypeEntity> 기존이슈유형_맵 = 프로젝트의_이슈유형_목록.stream()
                            .filter(Objects::nonNull)
                            .collect(Collectors.toMap(이슈유형 -> 이슈유형.getC_issue_type_url(), 이슈유형 -> 이슈유형));

                    List<지라이슈유형_데이터> 가져온_이슈유형_목록;
                    try {
                        가져온_이슈유형_목록 = engineService.클라우드_프로젝트별_이슈_유형_목록(엔진_통신_아이디, 프로젝트.getC_desc());
                    }
                    catch (Exception e) {
                        String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
                        log.error(오류_메세지);
                        return 오류_메세지;
                    }

                    Set<JiraIssueTypeEntity> 이슈유형_동기화목록 = 서버_엔티티_동기화(프로젝트의_이슈유형_목록, 기존이슈유형_맵, 가져온_이슈유형_목록,
                            검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

                    프로젝트.setJiraIssueTypeEntities(이슈유형_동기화목록);
                    해당_서버_프로젝트_목록.add(프로젝트);
                }

                검색된_ALM_서버.setJiraProjectEntities(해당_서버_프로젝트_목록);
            }
            else {
                JiraProjectEntity 프로젝트_검색전용 = new JiraProjectEntity();
                프로젝트_검색전용.setC_id(Long.parseLong(프로젝트_C아이디));
                JiraProjectEntity 프로젝트 = jiraProject.getNode(프로젝트_검색전용);
                Set<JiraIssueTypeEntity> 프로젝트의_이슈유형_목록 = Optional.ofNullable(프로젝트.getJiraIssueTypeEntities())
                        .orElse(new HashSet<>());

                Map<String, JiraIssueTypeEntity> 기존이슈유형_맵 = 프로젝트의_이슈유형_목록.stream()
                        .filter(Objects::nonNull)
                        .collect(Collectors.toMap(이슈유형 -> 이슈유형.getC_issue_type_url(), 이슈유형 -> 이슈유형));

                List<지라이슈유형_데이터> 가져온_이슈유형_목록;
                try {
                    가져온_이슈유형_목록 = engineService.클라우드_프로젝트별_이슈_유형_목록(엔진_통신_아이디, 프로젝트.getC_desc());
                }
                catch (Exception e) {
                    String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
                    log.error(오류_메세지);
                    return 오류_메세지;
                }

                Set<JiraIssueTypeEntity> 이슈유형_동기화목록 = 서버_엔티티_동기화(프로젝트의_이슈유형_목록, 기존이슈유형_맵, 가져온_이슈유형_목록,
                        검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

                프로젝트.setJiraIssueTypeEntities(이슈유형_동기화목록);
                Set<JiraProjectEntity> jiraProjectEntities = 검색된_ALM_서버.getJiraProjectEntities();

                if (jiraProjectEntities.removeIf(entity -> Objects.equals(entity.getC_id(), 프로젝트.getC_id()))) {
                    jiraProjectEntities.add(프로젝트);
                }

                검색된_ALM_서버.setJiraProjectEntities(jiraProjectEntities);
            }
        }

        return null;
    }

    private String 이슈상태_갱신(JiraServerEntity 검색된_ALM_서버, String 프로젝트_C아이디) throws Exception {
        EntityType 갱신할_항목 = EntityType.이슈상태;
        String 엔진_통신_아이디 = 검색된_ALM_서버.getC_jira_server_etc();
        String 서버유형 = 검색된_ALM_서버.getC_jira_server_type();

        if (StringUtils.equals(ServerType.JIRA_ON_PREMISE.getType(), 서버유형) || StringUtils.equals(ServerType.REDMINE_ON_PREMISE.getType(), 서버유형)) {
            Set<JiraIssueStatusEntity> 해당_서버_이슈_상태_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraIssueStatusEntities())
                    .orElse(new HashSet<>());

            Map<String, JiraIssueStatusEntity> 기존이슈상태_맵 = 해당_서버_이슈_상태_목록.stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toMap(이슈상태 ->
                            이슈상태.getC_issue_status_url(), 이슈상태 -> 이슈상태));

            List<지라이슈상태_데이터> 가져온_이슈상태_목록;
            try {
                가져온_이슈상태_목록 = engineService.ALM_이슈_상태_가져오기(엔진_통신_아이디);
            }
            catch (Exception e) {
                String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
                log.error(오류_메세지);
                return 오류_메세지;
            }

            Set<JiraIssueStatusEntity> 이슈상태_동기화목록 = 서버_엔티티_동기화(해당_서버_이슈_상태_목록, 기존이슈상태_맵, 가져온_이슈상태_목록,
                    검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

            검색된_ALM_서버.setJiraIssueStatusEntities(이슈상태_동기화목록);
        }
        else if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버유형)) {

            // 프로젝트_아이디가 null - 서버 등록. 전체 프로젝트에 대한 이슈상태 갱신로직
            // 프로젝트 아이디가 있는 경우 - 해당 프로젝트만 이슈상태 갱신시키는 로직
            if (프로젝트_C아이디 == null || 프로젝트_C아이디.isEmpty()) {
                Set<JiraProjectEntity> 동기화_프로젝트_목록 = new HashSet<>();
                Set<JiraProjectEntity> 해당_서버_프로젝트_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraProjectEntities())
                        .orElse(new HashSet<>());

                for (JiraProjectEntity 프로젝트 : 해당_서버_프로젝트_목록) {
                    if (프로젝트.getC_etc() != null && StringUtils.equals(프로젝트.getC_etc(), "delete")) {
                        continue;
                    }

                    Set<JiraIssueTypeEntity> 프로젝트의_이슈유형_목록 = Optional.ofNullable(프로젝트.getJiraIssueTypeEntities())
                            .orElse(new HashSet<>());

                    List<클라우드_지라_이슈유형별_상태_데이터> 가져온_이슈상태_목록;
                    try {
                        가져온_이슈상태_목록 = engineService.클라우드_프로젝트별_이슈_상태_목록(엔진_통신_아이디, 프로젝트.getC_desc());
                    }
                    catch (Exception e) {
                        String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
                        log.error(오류_메세지);
                        return 오류_메세지;
                    }

                    Map<String, List<지라이슈상태_데이터>> 이슈유형별_이슈상태목록_맵 = 가져온_이슈상태_목록.stream()
                            .filter(Objects::nonNull)
                            .collect(Collectors.toMap(
                                    이슈유형 -> 이슈유형.getId(),
                                    이슈유형 -> 이슈유형.getStatuses().stream()
                                            .map(issueStatus -> {
                                                issueStatus.setIssueTypeId(이슈유형.getId());
                                                return issueStatus;
                                            })
                                            .collect(Collectors.toList())
                            ));

                    for (JiraIssueTypeEntity issueType : 프로젝트의_이슈유형_목록) {
                        List<지라이슈상태_데이터> 이슈유형_이슈상태목록 = 이슈유형별_이슈상태목록_맵.get(issueType.getC_issue_type_id());
                        Map<String, JiraIssueStatusEntity> 기존이슈상태_맵 =
                                Optional.ofNullable(issueType.getJiraIssueStatusEntities())
                                        .orElse(Collections.emptySet())
                                        .stream()
                                        .filter(Objects::nonNull)
                                        .collect(Collectors.toMap(이슈상태 -> 이슈상태.getC_issue_status_url(), 이슈상태 -> 이슈상태));

                        Set<JiraIssueStatusEntity> 기존이슈상태_목록 = Optional.ofNullable(issueType.getJiraIssueStatusEntities())
                                .orElse(new HashSet<>())
                                .stream()
                                .filter(Objects::nonNull)
                                .collect(Collectors.toSet());

                        Set<JiraIssueStatusEntity> 이슈상태_동기화목록 = 서버_엔티티_동기화(기존이슈상태_목록, 기존이슈상태_맵, 이슈유형_이슈상태목록,
                                검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);
                        issueType.setJiraIssueStatusEntities(이슈상태_동기화목록);
                    }

                    프로젝트.setJiraIssueTypeEntities(프로젝트의_이슈유형_목록);
                    동기화_프로젝트_목록.add(프로젝트);
                }

                검색된_ALM_서버.setJiraProjectEntities(동기화_프로젝트_목록);
            }
            else {
                JiraProjectEntity 프로젝트_검색전용 = new JiraProjectEntity();
                프로젝트_검색전용.setC_id(Long.parseLong(프로젝트_C아이디));
                JiraProjectEntity 프로젝트 = jiraProject.getNode(프로젝트_검색전용);

                Set<JiraIssueTypeEntity> 프로젝트의_이슈유형_목록 = Optional.ofNullable(프로젝트.getJiraIssueTypeEntities())
                        .orElse(new HashSet<>());

                List<클라우드_지라_이슈유형별_상태_데이터> 가져온_이슈상태_목록;
                try {
                    가져온_이슈상태_목록 = engineService.클라우드_프로젝트별_이슈_상태_목록(엔진_통신_아이디, 프로젝트.getC_desc());
                }
                catch (Exception e) {
                    String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
                    log.error(오류_메세지);
                    return 오류_메세지;
                }

                Map<String, List<지라이슈상태_데이터>> 이슈유형별_이슈상태목록_맵 = 가져온_이슈상태_목록.stream().filter(Objects::nonNull)
                        .collect(Collectors.toMap(이슈유형 -> 이슈유형.getId(), 이슈유형 -> 이슈유형.getStatuses()));

                for (JiraIssueTypeEntity issueType : 프로젝트의_이슈유형_목록) {
                    List<지라이슈상태_데이터> 이슈유형_이슈상태목록 = 이슈유형별_이슈상태목록_맵.get(issueType.getC_issue_type_id());
                    Map<String, JiraIssueStatusEntity> 기존이슈상태_맵 =
                            issueType.getJiraIssueStatusEntities()
                                    .stream()
                                    .filter(Objects::nonNull)
                                    .collect(Collectors.toMap(이슈상태 -> 이슈상태.getC_issue_status_url(), 이슈상태 -> 이슈상태));

                    Set<JiraIssueStatusEntity> 기존이슈상태_목록 = Optional.ofNullable(issueType.getJiraIssueStatusEntities())
                            .orElse(new HashSet<>())
                            .stream()
                            .filter(Objects::nonNull)
                            .collect(Collectors.toSet());

                    Set<JiraIssueStatusEntity> 이슈상태_동기화목록 = 서버_엔티티_동기화(기존이슈상태_목록, 기존이슈상태_맵, 이슈유형_이슈상태목록,
                            검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);
                    issueType.setJiraIssueStatusEntities(이슈상태_동기화목록);
                }

                프로젝트.setJiraIssueTypeEntities(프로젝트의_이슈유형_목록);
                Set<JiraProjectEntity> jiraProjectEntities = 검색된_ALM_서버.getJiraProjectEntities();

                if (jiraProjectEntities.removeIf(entity -> Objects.equals(entity.getC_id(), 프로젝트.getC_id()))) {
                    jiraProjectEntities.add(프로젝트);
                }

                검색된_ALM_서버.setJiraProjectEntities(jiraProjectEntities);
            }
        }

        return null;
    }

    private String 이슈우선순위_갱신(JiraServerEntity 검색된_ALM_서버) {
        EntityType 갱신할_항목 = EntityType.이슈우선순위;
        String 엔진_통신_아이디 = 검색된_ALM_서버.getC_jira_server_etc();

        Set<JiraIssuePriorityEntity> 해당_서버_이슈_우선순위_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraIssuePriorityEntities())
                .orElse(new HashSet<>());

        Map<String, JiraIssuePriorityEntity> 기존이슈우선순위_맵 = 해당_서버_이슈_우선순위_목록.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(우선순위 -> 우선순위.getC_issue_priority_url(), 우선순위 -> 우선순위));

        List<지라이슈우선순위_데이터> 가져온_이슈우선순위_목록 = null;
        try {
            가져온_이슈우선순위_목록 = engineService.ALM_이슈_우선순위_가져오기(엔진_통신_아이디);
        }
        catch (Exception e) {
            String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
            log.error(오류_메세지);
            return 오류_메세지;
        }

        Set<JiraIssuePriorityEntity> 우선순위_동기화목록 = 서버_엔티티_동기화(해당_서버_이슈_우선순위_목록, 기존이슈우선순위_맵, 가져온_이슈우선순위_목록,
                검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

        검색된_ALM_서버.setJiraIssuePriorityEntities(우선순위_동기화목록);
        return null;
    }

    private String 이슈해결책_갱신(JiraServerEntity 검색된_ALM_서버) {
        EntityType 갱신할_항목 = EntityType.이슈해결책;
        String 엔진_통신_아이디 = 검색된_ALM_서버.getC_jira_server_etc();

        Set<JiraIssueResolutionEntity> 해당_서버_이슈_해결책_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraIssueResolutionEntities())
                .orElse(new HashSet<>());

        Map<String, JiraIssueResolutionEntity> 기존이슈해결책_맵 = 해당_서버_이슈_해결책_목록.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(이슈해결책 -> 이슈해결책.getC_issue_resolution_url(), 이슈해결책 -> 이슈해결책));

        List<지라이슈해결책_데이터> 가져온_이슈해결책_목록;
        try {
            가져온_이슈해결책_목록 = engineService.ALM_이슈_해결책_가져오기(엔진_통신_아이디);
        }
        catch (Exception e) {
            String 오류_메세지 = 검색된_ALM_서버.getC_jira_server_name()+"의 "+ 갱신할_항목 + " 이(가) 갱신 실패하습니다." + e.getMessage();
            log.error(오류_메세지);
            return 오류_메세지;
        }

        Set<JiraIssueResolutionEntity> 해결책_동기화목록 = 서버_엔티티_동기화(해당_서버_이슈_해결책_목록, 기존이슈해결책_맵, 가져온_이슈해결책_목록,
                검색된_ALM_서버, 엔진_통신_아이디, 갱신할_항목);

        검색된_ALM_서버.setJiraIssueResolutionEntities(해결책_동기화목록);
        return null;
    }

    private <E extends TreeSearchEntity, T extends ALM_데이터> Set<E> 서버_엔티티_동기화(Set<E> 등록된_엔티티_목록, Map<String, E> 기존_맵, List<T> 가져온_엔티티_목록,
                                                                              JiraServerEntity 검색된_ALM_서버, String 엔진_연결_아이디, EntityType 갱신항목 ) {
        String 서버유형 = 검색된_ALM_서버.getC_jira_server_type();

        if (가져온_엔티티_목록 == null || 가져온_엔티티_목록.size() == 0) {
            return null;
        }

        for (T 엔진_엔티티 : 가져온_엔티티_목록) {
            boolean 엔티티존재여부 = 기존_맵 != null && 기존_맵.containsKey(엔진_엔티티.getSelf());
            E 백엔드_엔티티 = 엔티티존재여부 ? 기존_맵.get(엔진_엔티티.getSelf()) : null;
            E 결과 = null;
            try {
                if (백엔드_엔티티 instanceof JiraProjectEntity || EntityType.프로젝트 == 갱신항목) {
                    결과 = (E) 프로젝트_저장_또는_업데이트((지라프로젝트_데이터) 엔진_엔티티, (JiraProjectEntity) 백엔드_엔티티);
                }
                else if (백엔드_엔티티 instanceof JiraIssueTypeEntity || EntityType.이슈유형 == 갱신항목) {
                    결과 = (E) 이슈유형_저장_또는_업데이트((지라이슈유형_데이터) 엔진_엔티티, (JiraIssueTypeEntity) 백엔드_엔티티, 서버유형);
                }
                else if (백엔드_엔티티 instanceof JiraIssueStatusEntity || EntityType.이슈상태 == 갱신항목) {
                    결과 = (E) 이슈상태_저장_또는_업데이트( (지라이슈상태_데이터) 엔진_엔티티, (JiraIssueStatusEntity) 백엔드_엔티티 );
                }
                else if (백엔드_엔티티 instanceof JiraIssuePriorityEntity || EntityType.이슈우선순위 == 갱신항목) {
                    결과 = (E) 이슈우선순위_저장_또는_업데이트((지라이슈우선순위_데이터) 엔진_엔티티, (JiraIssuePriorityEntity) 백엔드_엔티티, 서버유형);
                }
                else if (백엔드_엔티티 instanceof JiraIssueResolutionEntity || EntityType.이슈해결책 == 갱신항목) {
                    결과 = (E) 이슈해결책_저장_또는_업데이트((지라이슈해결책_데이터) 엔진_엔티티, (JiraIssueResolutionEntity) 백엔드_엔티티, 서버유형);
                }
            }
            catch (Exception e) {
                log.error("서버_엔티티_동기화 오류 발생 :: 연결아이디 :: {} 서버유형 :: {} :: 갱신항목 ::  {} :: 엔티티 :: {}",
                        엔진_연결_아이디, 서버유형, 갱신항목, 엔진_엔티티.toString());
                log.error(e.getMessage());
            }

            if (결과 != null && !엔티티존재여부) {
                등록된_엔티티_목록.add(결과);
            }

            기존_맵.remove(엔진_엔티티.getSelf());
        }

        Set<E> 반환할_엔티티_목록 = 소프트_딜리트(기존_맵, 등록된_엔티티_목록);

        return 반환할_엔티티_목록;
    }

    private <T extends TreeSearchEntity> Set<T> 소프트_딜리트(Map<String, T> 기존_맵, Set<T> 등록된_엔티티_목록) {
        /**
         * 이 메소드는 JiraProjectEntity, JiraIssueTypeEntity, JiraIssueStatusEntity, JiraIssueResolutionEntity,
         * JiraIssuePriorityEntity의 [ c_etc ] 컬럼을 Soft Delete 동작을 수행합니다.
         * Soft Delete 처리와 함께 c_check(JiraProjectEntity 제외) 의 값을 false로 변경해줍니다.
         * @author Advanc2d
         * @since 2024-03-31
         **/
        if (기존_맵 != null && 기존_맵.size() > 0) {
            기존_맵.forEach((엔티티키, 엔티티) -> {
                T 반환결과 = null;
                try {
                    int 처리결과 = 0;

                    if (엔티티 instanceof JiraProjectEntity) {
                        JiraProjectEntity 프로젝트 = (JiraProjectEntity) 엔티티;
                        if (프로젝트.getC_etc() == null || !StringUtils.equals("delete", 프로젝트.getC_etc())) {
                            프로젝트.setC_etc("delete");
                        }
                        처리결과 = jiraProject.updateNode(프로젝트);

                        반환결과 = (T) 프로젝트;
                    }
                    else if (엔티티 instanceof JiraIssueTypeEntity) {
                        JiraIssueTypeEntity 이슈유형 = (JiraIssueTypeEntity) 엔티티;
                        if (이슈유형.getC_etc() == null || !StringUtils.equals("delete", 이슈유형.getC_etc())) {
                            이슈유형.setC_etc("delete");
                            이슈유형.setC_check("false");
                        }
                        처리결과 = jiraIssueType.updateNode(이슈유형);

                        반환결과 = (T) 이슈유형;
                    }
                    else if (엔티티 instanceof JiraIssueStatusEntity) {
                        JiraIssueStatusEntity 이슈상태 = (JiraIssueStatusEntity) 엔티티;
                        if (이슈상태.getC_etc() == null || !StringUtils.equals("delete", 이슈상태.getC_etc())) {
                            이슈상태.setC_etc("delete");
                            이슈상태.setC_check("false");
                        }
                        처리결과 = jiraIssueStatus.updateNode(이슈상태);

                        반환결과 = (T) 이슈상태;
                    }
                    else if (엔티티 instanceof JiraIssuePriorityEntity) {
                        JiraIssuePriorityEntity 이슈우선순위 = (JiraIssuePriorityEntity) 엔티티;
                        if (이슈우선순위.getC_etc() == null || !StringUtils.equals("delete", 이슈우선순위.getC_etc())) {
                            이슈우선순위.setC_etc("delete");
                            이슈우선순위.setC_check("false");
                        }
                        처리결과 = jiraIssuePriority.updateNode(이슈우선순위);

                        반환결과 = (T) 이슈우선순위;
                    }
                    else if (엔티티 instanceof JiraIssueResolutionEntity) {
                        JiraIssueResolutionEntity 이슈해결책 = (JiraIssueResolutionEntity) 엔티티;
                        if (이슈해결책.getC_etc() == null || !StringUtils.equals("delete", 이슈해결책.getC_etc())) {
                            이슈해결책.setC_etc("delete");
                            이슈해결책.setC_check("false");
                        }
                        처리결과 = jiraIssueResolution.updateNode(이슈해결책);

                        반환결과 = (T) 이슈해결책;
                    }

                    if(처리결과 == 1) {
                        등록된_엔티티_목록.add(반환결과);
                    }
                }
                catch (Exception e) {
                    log.error("소프트 딜리트 처리 오류 발생 :: 엔티티 ::  {}", 엔티티.toString());
                    log.error(e.getMessage());
                }
            });
        }

        return 등록된_엔티티_목록;
    }

    private JiraProjectEntity 프로젝트_저장_또는_업데이트(지라프로젝트_데이터 엔진_프로젝트, JiraProjectEntity 백엔드_프로젝트) throws Exception {

        boolean isNew = false;

        if (백엔드_프로젝트 == null) {
            isNew= true;
            백엔드_프로젝트 = new JiraProjectEntity();
        }

        // 저장 or 업데이트 공통
        백엔드_프로젝트.setC_jira_key(엔진_프로젝트.getKey());
        백엔드_프로젝트.setC_desc(엔진_프로젝트.getId());
        백엔드_프로젝트.setC_jira_url(엔진_프로젝트.getSelf());

        String projectName = 엔진_프로젝트.getName();
        if (projectName != null) {
            String formattedProjectName = projectName.replaceAll("&nbsp;-&nbsp;", " - ");
            백엔드_프로젝트.setC_title(formattedProjectName);
            백엔드_프로젝트.setC_jira_name(formattedProjectName);
        }

        if (isNew) {
            백엔드_프로젝트.setRef(TreeConstant.First_Node_CID);
            백엔드_프로젝트.setC_type(TreeConstant.Leaf_Node_TYPE);
            return jiraProject.addNode(백엔드_프로젝트);
        }
        else {
            /***
             *삭제된 내용이었는데 복구된 경우 Soft Delete 복구처리를 위한 로직
             ***/
            if (백엔드_프로젝트.getC_etc() != null && StringUtils.equals("delete", 백엔드_프로젝트.getC_etc())) {
                백엔드_프로젝트.setC_etc(null);
            }
            int 업데이트결과 = jiraProject.updateNode(백엔드_프로젝트);
            return 업데이트결과 == 1 ? 백엔드_프로젝트 : null;
        }
    }

    private JiraIssueTypeEntity 이슈유형_저장_또는_업데이트(지라이슈유형_데이터 엔진_이슈유형, JiraIssueTypeEntity 백엔드_이슈유형, String 서버유형) throws Exception {

        boolean isNew = false;

        if (백엔드_이슈유형 == null) {
            isNew= true;
            백엔드_이슈유형 = new JiraIssueTypeEntity();
        }

        // 공통
        백엔드_이슈유형.setC_issue_type_id(엔진_이슈유형.getId());
        백엔드_이슈유형.setC_issue_type_name(엔진_이슈유형.getName());
        백엔드_이슈유형.setC_issue_type_url(엔진_이슈유형.getSelf());
        백엔드_이슈유형.setC_issue_type_desc(엔진_이슈유형.getDescription());
        백엔드_이슈유형.setC_desc(엔진_이슈유형.getSubtask().toString());

        if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(),서버유형)) {
            백엔드_이슈유형.setC_contents(엔진_이슈유형.getHierarchyLevel().toString()); //Integer
        }

        if (isNew) {
            if (엔진_이슈유형.getName().equals("arms-requirement")) {
                백엔드_이슈유형.setC_check("true"); //기본값 false 설정
            }
            else {
                백엔드_이슈유형.setC_check("false"); //기본값 false 설정
            }
            백엔드_이슈유형.setRef(TreeConstant.First_Node_CID);
            백엔드_이슈유형.setC_type(TreeConstant.Leaf_Node_TYPE);
            return jiraIssueType.addNode(백엔드_이슈유형);
        }
        else {
            /***
             *삭제된 내용이었는데 복구된 경우 Soft Delete 복구처리를 위한 로직
             ***/
            if (백엔드_이슈유형.getC_etc() != null && StringUtils.equals("delete", 백엔드_이슈유형.getC_etc())) {
                백엔드_이슈유형.setC_etc(null);
            }
            int 업데이트결과 = jiraIssueType.updateNode(백엔드_이슈유형);
            return 업데이트결과 == 1 ? 백엔드_이슈유형 : null;
        }
    }

    private JiraIssueStatusEntity 이슈상태_저장_또는_업데이트(지라이슈상태_데이터 엔진_이슈상태, JiraIssueStatusEntity 백엔드_이슈상태) throws Exception {

        boolean isNew = false;

        if (백엔드_이슈상태 == null) {
            isNew = true;
            백엔드_이슈상태 = new JiraIssueStatusEntity();
        }

        // 공통 필드 설정
        백엔드_이슈상태.setC_issue_status_id(엔진_이슈상태.getId());
        백엔드_이슈상태.setC_issue_status_name(엔진_이슈상태.getName());
        백엔드_이슈상태.setC_issue_status_url(엔진_이슈상태.getSelf());
        백엔드_이슈상태.setC_issue_status_desc(엔진_이슈상태.getDescription());
        백엔드_이슈상태.setC_issue_type_mapping_id(엔진_이슈상태.getIssueTypeId());

        if (isNew) {
            백엔드_이슈상태.setC_check("false");
            백엔드_이슈상태.setRef(TreeConstant.First_Node_CID);
            백엔드_이슈상태.setC_type(TreeConstant.Leaf_Node_TYPE);
            return jiraIssueStatus.addNode(백엔드_이슈상태);
        } else {
            // Soft Delete 복구 처리
            if (백엔드_이슈상태.getC_etc() != null && StringUtils.equals("delete", 백엔드_이슈상태.getC_etc())) {
                백엔드_이슈상태.setC_etc(null);
            }
            int 업데이트결과 = jiraIssueStatus.updateNode(백엔드_이슈상태);
            return 업데이트결과 == 1 ? 백엔드_이슈상태 : null;
        }
    }

    private JiraIssuePriorityEntity 이슈우선순위_저장_또는_업데이트(지라이슈우선순위_데이터 엔진_이슈우선순위, JiraIssuePriorityEntity 백엔드_이슈우선순위, String 서버유형) throws Exception {

        boolean isNew = false;

        if (백엔드_이슈우선순위 == null) {
            isNew= true;
            백엔드_이슈우선순위 = new JiraIssuePriorityEntity();
        }

        백엔드_이슈우선순위.setC_issue_priority_id(엔진_이슈우선순위.getId());
        백엔드_이슈우선순위.setC_issue_priority_name(엔진_이슈우선순위.getName());
        백엔드_이슈우선순위.setC_issue_priority_url(엔진_이슈우선순위.getSelf());
        백엔드_이슈우선순위.setC_issue_priority_desc(엔진_이슈우선순위.getDescription());

        if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버유형)
                || StringUtils.equals(ServerType.REDMINE_ON_PREMISE.getType(), 서버유형)) {
            백엔드_이슈우선순위.setC_desc(String.valueOf(엔진_이슈우선순위.isDefault()));
        }

        if (isNew) {
            if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버유형)
                    || StringUtils.equals(ServerType.REDMINE_ON_PREMISE.getType(), 서버유형)) {
                백엔드_이슈우선순위.setC_check(String.valueOf(엔진_이슈우선순위.isDefault()));
            }
            백엔드_이슈우선순위.setRef(TreeConstant.First_Node_CID);
            백엔드_이슈우선순위.setC_type(TreeConstant.Leaf_Node_TYPE);
            return jiraIssuePriority.addNode(백엔드_이슈우선순위);
        }
        else {
            /***
             *삭제된 내용이었는데 복구된 경우 Soft Delete 복구처리를 위한 로직
             ***/
            if (백엔드_이슈우선순위.getC_etc() != null && StringUtils.equals("delete", 백엔드_이슈우선순위.getC_etc())) {
                백엔드_이슈우선순위.setC_etc(null);
            }
            int 업데이트결과 = jiraIssuePriority.updateNode(백엔드_이슈우선순위);
            return 업데이트결과 == 1 ? 백엔드_이슈우선순위 : null;
        }
    }

    private JiraIssueResolutionEntity 이슈해결책_저장_또는_업데이트(지라이슈해결책_데이터 엔진_이슈해결책, JiraIssueResolutionEntity 백엔드_이슈해결책, String 서버유형) throws Exception {

        boolean isNew = false;

        if (백엔드_이슈해결책 == null) {
            isNew= true;
            백엔드_이슈해결책 = new JiraIssueResolutionEntity();
        }

        백엔드_이슈해결책.setC_issue_resolution_id(엔진_이슈해결책.getId());
        백엔드_이슈해결책.setC_issue_resolution_name(엔진_이슈해결책.getName());
        백엔드_이슈해결책.setC_issue_resolution_url(엔진_이슈해결책.getSelf());
        백엔드_이슈해결책.setC_issue_resolution_desc(엔진_이슈해결책.getDescription());
        백엔드_이슈해결책.setC_desc(String.valueOf(엔진_이슈해결책.isDefault()));

        if (isNew) {
            if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버유형) || StringUtils.equals(ServerType.REDMINE_ON_PREMISE.getType(), 서버유형)) {
                백엔드_이슈해결책.setC_check(String.valueOf(엔진_이슈해결책.isDefault()));
            }
            백엔드_이슈해결책.setRef(TreeConstant.First_Node_CID);
            백엔드_이슈해결책.setC_type(TreeConstant.Leaf_Node_TYPE);
            return jiraIssueResolution.addNode(백엔드_이슈해결책);
        }
        else {
            /***
             *삭제된 내용이었는데 복구된 경우 Soft Delete 복구처리를 위한 로직
             ***/
            if (백엔드_이슈해결책.getC_etc() != null && StringUtils.equals("delete", 백엔드_이슈해결책.getC_etc())) {
                백엔드_이슈해결책.setC_etc(null);
            }
            int 업데이트결과 = jiraIssueResolution.updateNode(백엔드_이슈해결책);
            return 업데이트결과 == 1 ? 백엔드_이슈해결책 : null;
        }
    }

    @Async
    @Override
    @Transactional
    public CompletableFuture<Boolean> refreshALMServer(JiraServerEntity jiraServerEntity) {

        boolean 결과 = false;

        try {
            chat.sendMessageByEngine("서버 갱신을 시작합니다. 서버 데이터에 따라 걸리는 시간 차가 발생할 수 있습니다.");

            서버_엔티티_항목별_갱신(EntityType.프로젝트, null, jiraServerEntity);
            서버_엔티티_항목별_갱신(EntityType.이슈유형, null, jiraServerEntity);
            서버_엔티티_항목별_갱신(EntityType.이슈상태, null, jiraServerEntity);
            서버_엔티티_항목별_갱신(EntityType.이슈우선순위, null, jiraServerEntity);
            서버_엔티티_항목별_갱신(EntityType.이슈해결책, null, jiraServerEntity);

            결과 = true;
        }
        catch (Exception e) {
            log.error(e.getMessage());
            chat.sendMessageByEngine("서버 갱신 중 오류가 발생하였습니다. :: " + e.getMessage());
        }

        if (결과) {
            chat.sendMessageByEngine("전체 서버 데이터 갱신이 완료되었습니다.");
        }

        return CompletableFuture.completedFuture(true);
    }

    @Override
    @Transactional
    public Map<String, Map<String, StatusRuleSetEntity>> getStatusRuleSet(JiraServerEntity searchEntity) throws Exception {
        JiraServerEntity jiraServerEntity = this.getNode(searchEntity);

        if (jiraServerEntity == null
                || jiraServerEntity.getC_jira_server_type() == null
                || jiraServerEntity.getC_jira_server_etc() == null) {
            throw new Exception("룰셋 설정에 필요한 서버 정보가 없습니다.");
        }

        String 엔진_통신_아이디 = jiraServerEntity.getC_jira_server_etc();
        String 서버유형 = jiraServerEntity.getC_jira_server_type();

        if (!StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버유형)) {
            throw new Exception("상태 프리셋 방식은 클라우드 지라 서버 타입의 경우에만 사용됩니다.");
        }

        Map<String, Map<String, StatusRuleSetEntity>> statusRuleSetMap = new HashMap<>();
        Set<JiraProjectEntity> almProjectEntities = Optional.ofNullable(jiraServerEntity.getJiraProjectEntities())
                .orElse(Collections.emptySet());

        for (JiraProjectEntity almProjectEntity : almProjectEntities) {
            if (almProjectEntity.getC_etc() != null && StringUtils.equals(almProjectEntity.getC_etc(), "delete")) {
                continue;
            }

            List<클라우드_지라_이슈유형별_상태_데이터> almProjectIssueStatusList = null;
            try {
                almProjectIssueStatusList = engineService.클라우드_프로젝트별_이슈_상태_목록(엔진_통신_아이디, almProjectEntity.getC_desc());
            } catch (Exception e) {
                log.error("Error updating " + jiraServerEntity.getC_jira_server_name() + ": " + e.getMessage(), e);
            }

            Optional.ofNullable(almProjectIssueStatusList)
                    .orElseGet(Collections::emptyList)
                    .forEach(issueType -> {
                        Optional.ofNullable(issueType.getStatuses())
                                .orElseGet(Collections::emptyList)
                                .forEach(status -> {
                                    statusRuleSetMap
                                            // status category key로 ruleset map key 설정
                                            .computeIfAbsent(status.getStatusCategory().getKey(), k -> new HashMap<>())
                                            // status name을 key로 상태 map 설정
                                            .compute(status.getName(), (name, existingStatus) -> {
                                                if (existingStatus == null) {
                                                    // 존재하지 않는 상태일 시 RuleSetEntity 반환
                                                    StatusRuleSetEntity ruleSetEntity = new StatusRuleSetEntity();
                                                    ruleSetEntity.setStatusCategoryKey(status.getStatusCategory().getKey());
                                                    ruleSetEntity.setStatusName(status.getName());
                                                    ruleSetEntity.setCount(1);
                                                    return ruleSetEntity;
                                                } else {
                                                    // 이미 있는 상태 데이터일 시 count + 1
                                                    existingStatus.setCount(existingStatus.getCount() + 1);
                                                    return existingStatus;
                                                }
                                            });
                                });
                    });
        }

        statusRuleSetMap.forEach((key, stateMap) -> {
            List<StatusRuleSetEntity> stateList = new ArrayList<>(stateMap.values());

            // category key별 상태 count 기반 내림차순 정렬 & 같은 count일 경우 상태명 오름차순 정렬
            stateList.sort(
                    // 빈도수 기반 내림차순 설정
                    Comparator.comparingInt(StatusRuleSetEntity::getCount).reversed()
                            // 같은 빈도수 일 경우 상태명 기준 오름차순 설정
                            .thenComparing(StatusRuleSetEntity::getStatusName)
            );

            // 정렬된 상태기반 우선순위 설정
            int priority = 1;
            for (StatusRuleSetEntity entity : stateList) {
                entity.setPriority(priority++);
            }
        });

        return statusRuleSetMap;
    }

    @Async
    @Override
    @Transactional
    public void preSetIssueStatusMapping(String projectCid, JiraServerEntity jiraServer) throws Exception {

        boolean 프리셋_결과 = false;

        JiraServerEntity 검색용_서버_엔티티 = new JiraServerEntity();
        검색용_서버_엔티티.setC_id(jiraServer.getC_id());

        JiraServerEntity 검색된_ALM_서버 = this.getNode(검색용_서버_엔티티);

        if (검색된_ALM_서버 == null
                || 검색된_ALM_서버.getC_jira_server_type() == null
                || 검색된_ALM_서버.getC_jira_server_etc() == null) {
            throw new Exception("프리셋 설정에 필요한 서버 정보가 없습니다.");
        }

        String 서버_유형 = 검색된_ALM_서버.getC_jira_server_type();

        if (StringUtils.equals(ServerType.JIRA_CLOUD.getType(), 서버_유형)) {  // 클라우드 지라의 경우
            if (projectCid == null || projectCid.isEmpty()) {
                // 프로젝트 아이디 없음 서버에 해당하는 모든 프로젝트의 프리셋 진행
                프리셋_결과 = this.setCloudJiraProjectPreset(검색된_ALM_서버);
            }
        }

        String 결과_메세지 = 프리셋_결과 ? 검색된_ALM_서버.getC_jira_server_name() + "의 상태 프리셋 연결이 완료되었습니다."
                : 검색된_ALM_서버.getC_jira_server_name() + "의 상태 프리셋 연결에 실패하였습니다.";

        chat.sendMessageByEngine(결과_메세지);
    }

    private boolean setCloudJiraProjectPreset(JiraServerEntity 검색된_ALM_서버) throws Exception {

        boolean result = false;

        String 엔진_통신_아이디 = 검색된_ALM_서버.getC_jira_server_etc();
        Map<String, Map<String, StatusRuleSetEntity>> 룰셋 = this.getStatusRuleSet(검색된_ALM_서버);

        JiraIssueStatusCategoryMapEntity 상태카테고리매핑_테이블_검색 = new JiraIssueStatusCategoryMapEntity();
        List<JiraIssueStatusCategoryMapEntity> 상태카테고리매핑_목록 = jiraIssueStatusCategoryMap.getNodesWithoutRoot(상태카테고리매핑_테이블_검색);

        // 상태카테고리 이름 - 상태 아이디 값 맵
        Map<String, Long> 상태카테고리매핑_맵 = 상태카테고리매핑_목록.stream()
                .filter(Objects::nonNull) // 널 체크
                .collect(Collectors.toMap(
                        JiraIssueStatusCategoryMapEntity::getC_title,
                        JiraIssueStatusCategoryMapEntity::getC_req_state_mapping_id
                ));

        Set<JiraProjectEntity> 동기화_프로젝트_목록 = new HashSet<>();
        Set<JiraProjectEntity> 해당_서버_프로젝트_목록 = Optional.ofNullable(검색된_ALM_서버.getJiraProjectEntities())
                .orElse(new HashSet<>());

        for (JiraProjectEntity 프로젝트 : 해당_서버_프로젝트_목록) {
            if (프로젝트.getC_etc() != null && StringUtils.equals(프로젝트.getC_etc(), "delete")) {
                continue;
            }

            Set<JiraIssueTypeEntity> 프로젝트의_이슈유형_목록 = this.setCloudJiraIssueType(프로젝트, 엔진_통신_아이디, 룰셋, 상태카테고리매핑_맵);

            /* API 오류 혹은 삭제된 프로젝트 시 null 반환 방어로직  */
            if (프로젝트의_이슈유형_목록 != null) {
                프로젝트.setJiraIssueTypeEntities(프로젝트의_이슈유형_목록);
            }

            동기화_프로젝트_목록.add(프로젝트);
        }

        검색된_ALM_서버.setJiraProjectEntities(동기화_프로젝트_목록);

        try {
            this.updateNode(검색된_ALM_서버);
            result = true;
        }
        catch (Exception e){
            log.error(e.getMessage());
        }

        return result;
    }

    private Set<JiraIssueTypeEntity> setCloudJiraIssueType(JiraProjectEntity jiraProjectEntity, String 엔진_통신_아이디,
                                                           Map<String, Map<String, StatusRuleSetEntity>> 룰셋,
                                                           Map<String, Long> 상태카테고리매핑_맵) {

        Set<JiraIssueTypeEntity> 프로젝트의_이슈유형_목록 = Optional.ofNullable(jiraProjectEntity.getJiraIssueTypeEntities())
                .orElse(new HashSet<>());

        List<클라우드_지라_이슈유형별_상태_데이터> 가져온_이슈상태_목록;
        try {
            가져온_이슈상태_목록 = engineService.클라우드_프로젝트별_이슈_상태_목록(엔진_통신_아이디, jiraProjectEntity.getC_desc());
        }
        catch (Exception e) {
            String 오류_메세지 = 엔진_통신_아이디 + "서버의 " +jiraProjectEntity.getC_desc() + " 프로젝트 이슈상태 갱신에 실패하였습니다. 프로젝트 전체 프리셋 중 이슈 상태 조회 오류 발생" + e.getMessage();
            log.error(오류_메세지);
            chat.sendMessageByEngine(오류_메세지);
            return null;
        }

        Map<String, List<지라이슈상태_데이터>> 이슈유형별_이슈상태목록_맵 = 가져온_이슈상태_목록.stream()
                .filter(Objects::nonNull)
                .collect(Collectors.toMap(
                        이슈유형 -> 이슈유형.getId(),
                        이슈유형 -> 이슈유형.getStatuses().stream()
                                .map(issueStatus -> {
                                    issueStatus.setIssueTypeId(이슈유형.getId());
                                    return issueStatus;
                                })
                                .collect(Collectors.toList())
                ));

        for (JiraIssueTypeEntity issueType : 프로젝트의_이슈유형_목록) {
            List<지라이슈상태_데이터> 이슈유형_이슈상태목록 = 이슈유형별_이슈상태목록_맵.get(issueType.getC_issue_type_id());

            if (이슈유형_이슈상태목록 == null || 이슈유형_이슈상태목록.size() == 0) {
                continue;
            }

            Set<JiraIssueStatusEntity> 기존이슈상태_목록 = Optional.ofNullable(issueType.getJiraIssueStatusEntities())
                    .orElse(new HashSet<>())
                    .stream()
                    .filter(Objects::nonNull)
                    .collect(Collectors.toSet());

            Map<String, JiraIssueStatusEntity> 기존이슈상태_맵 =
                    Optional.ofNullable(기존이슈상태_목록)
                            .orElse(Collections.emptySet())
                            .stream()
                            .filter(Objects::nonNull)
                            .collect(Collectors.toMap(이슈상태 -> 이슈상태.getC_issue_status_url(), 이슈상태 -> 이슈상태));

            Set<JiraIssueStatusEntity> 이슈상태_동기화목록 = this.setIssueStatusEntitiesOfIssueType(기존이슈상태_목록, 기존이슈상태_맵, 이슈유형_이슈상태목록, 룰셋, 상태카테고리매핑_맵);
            issueType.setJiraIssueStatusEntities(이슈상태_동기화목록);
        }

        return 프로젝트의_이슈유형_목록;
    }

    private Set<JiraIssueStatusEntity> setIssueStatusEntitiesOfIssueType(Set<JiraIssueStatusEntity> 기존이슈상태_목록,
                                                                         Map<String, JiraIssueStatusEntity> 기존이슈상태_맵,
                                                                         List<지라이슈상태_데이터> 이슈유형_이슈상태목록,
                                                                         Map<String, Map<String, StatusRuleSetEntity>> 룰셋,
                                                                         Map<String, Long> 상태카테고리매핑_맵) {

        Map<String, 지라이슈상태_데이터> 우선순위_가장높은_상태_맵 = 이슈유형_이슈상태목록.stream()
                .collect(Collectors.toMap(
                        조회된_이슈상태 -> 조회된_이슈상태.getStatusCategory().getKey(),
                        Function.identity(),
                        (existing, replacement) -> {
                            // 우선 순위 비교 하여 높은 우선순위
                            int 기존_우선순위 = 룰셋.get(existing.getStatusCategory().getKey())
                                    .get(existing.getName()).getPriority();
                            int 새로운_우선순위 = 룰셋.get(replacement.getStatusCategory().getKey())
                                    .get(replacement.getName()).getPriority();
                            return 새로운_우선순위 < 기존_우선순위 ? replacement : existing;
                        }
                ));

        for (지라이슈상태_데이터 이슈상태_데이터 : 이슈유형_이슈상태목록) {
            boolean 엔티티존재여부 = 기존이슈상태_맵 != null && 기존이슈상태_맵.containsKey(이슈상태_데이터.getSelf());
            JiraIssueStatusEntity jiraIssueStatusEntity = 엔티티존재여부 ? 기존이슈상태_맵.get(이슈상태_데이터.getSelf()) : null;
            JiraIssueStatusEntity 결과 = null;

            String 상태_카테고리 = 이슈상태_데이터.getStatusCategory().getKey();
            지라이슈상태_데이터 우선순위_가장높은_상태 = 우선순위_가장높은_상태_맵.get(상태_카테고리);

            try {
                결과 = 이슈상태_저장_또는_업데이트_매핑(이슈상태_데이터, jiraIssueStatusEntity, 우선순위_가장높은_상태, 상태카테고리매핑_맵);
            }
            catch (Exception e) {
                log.error(e.getMessage());
            }

            if (결과 != null && !엔티티존재여부) {
                기존이슈상태_목록.add(결과);
            }

            기존이슈상태_맵.remove(이슈상태_데이터.getSelf());
        }

        return 소프트_딜리트(기존이슈상태_맵, 기존이슈상태_목록);
    }

    private JiraIssueStatusEntity 이슈상태_저장_또는_업데이트_매핑(지라이슈상태_데이터 엔진_이슈상태,
                                                     JiraIssueStatusEntity 백엔드_이슈상태,
                                                     지라이슈상태_데이터 우선순위_가장높은_상태,
                                                     Map<String, Long> 상태카테고리_테이블_맵) throws Exception {

        boolean isNew = false;

        if (백엔드_이슈상태 == null) {
            isNew = true;
            백엔드_이슈상태 = new JiraIssueStatusEntity();
        }

        // 공통 필드 설정
        백엔드_이슈상태.setC_issue_status_id(엔진_이슈상태.getId());
        백엔드_이슈상태.setC_issue_status_name(엔진_이슈상태.getName());
        백엔드_이슈상태.setC_issue_status_url(엔진_이슈상태.getSelf());
        백엔드_이슈상태.setC_issue_status_desc(엔진_이슈상태.getDescription());
        백엔드_이슈상태.setC_issue_type_mapping_id(엔진_이슈상태.getIssueTypeId());

        // 우선순위 상태 1대1 매핑 부분
        if (isHighestPriorityStatus(엔진_이슈상태, 우선순위_가장높은_상태)) {
            setStatusPresetMapping(백엔드_이슈상태, 엔진_이슈상태, 상태카테고리_테이블_맵);
        }
        else {
            // 그 외 상태는 기타 상태로 매핑
            백엔드_이슈상태.setC_req_state_mapping_link(14L);
        }

        if (isNew) {
            백엔드_이슈상태.setC_check("false");
            백엔드_이슈상태.setRef(TreeConstant.First_Node_CID);
            백엔드_이슈상태.setC_type(TreeConstant.Leaf_Node_TYPE);
            return jiraIssueStatus.addNode(백엔드_이슈상태);
        }
        else {
            // Soft Delete 복구 처리
            if (백엔드_이슈상태.getC_etc() != null && StringUtils.equals("delete", 백엔드_이슈상태.getC_etc())) {
                백엔드_이슈상태.setC_etc(null);
            }
            int 업데이트결과 = jiraIssueStatus.updateNode(백엔드_이슈상태);
            return 업데이트결과 == 1 ? 백엔드_이슈상태 : null;
        }
    }

    private boolean isHighestPriorityStatus(지라이슈상태_데이터 currentStatus, 지라이슈상태_데이터 highestPriorityStatus) {
        return highestPriorityStatus != null && currentStatus.getSelf().equals(highestPriorityStatus.getSelf());
    }

    private void setStatusPresetMapping(JiraIssueStatusEntity issueStatus,
                                        지라이슈상태_데이터 engineIssueStatus,
                                        Map<String, Long> statusCategoryMap) {

        String statusCategory = engineIssueStatus.getStatusCategory().getKey();
        Long mappingValue = statusCategoryMap.get(statusCategory);

        if (mappingValue != null) {
            issueStatus.setC_req_state_mapping_link(mappingValue);
        }
    }

    @Override
    public void cacheStatusMappingData(JiraServerEntity jiraServerEntity) throws Exception {

        List<JiraServerEntity> jiraServerEntityList = this.getNodesWithoutRoot(jiraServerEntity);

        jiraServerEntityList.stream()
                .filter(Objects::nonNull)
                .forEach(this::processAlmServer);
    }

    private void processAlmServer(JiraServerEntity jiraServerEntity) {
        ServerType serverType = ServerType.fromString(jiraServerEntity.getC_jira_server_type());
        String serverId = jiraServerEntity.getC_jira_server_etc();

        if (serverType == ServerType.JIRA_CLOUD) {
            processJiraCloud(serverId, jiraServerEntity);
        } else if (serverType == ServerType.JIRA_ON_PREMISE || serverType == ServerType.REDMINE_ON_PREMISE) {
            processOnPremise(serverId, jiraServerEntity);
        }
    }

    private void processJiraCloud(String serverId, JiraServerEntity jiraServerEntity) {
        jiraServerEntity.getJiraProjectEntities().forEach(jiraProjectEntity ->
                jiraProjectEntity.getJiraIssueTypeEntities().forEach(jiraIssueTypeEntity ->
                        cacheALMServerDTO(
                                convertToIssueStatusDTOs(jiraIssueTypeEntity.getJiraIssueStatusEntities()),
                                serverId,
                                jiraProjectEntity.getC_jira_key(),
                                jiraIssueTypeEntity.getC_issue_type_id()
                        )
                )
        );
    }

    private List<IssueStatusDTO> convertToIssueStatusDTOs(Set<JiraIssueStatusEntity> issueStatusEntities) {
        return issueStatusEntities.stream()
                .filter(Objects::nonNull)
                .map(issueStatus -> new IssueStatusDTO(
                        issueStatus.getC_issue_status_id(),
                        issueStatus.getC_issue_status_name(),
                        issueStatus.getC_req_state_mapping_link() != null ? issueStatus.getC_req_state_mapping_link().toString() : null
                ))
                .collect(Collectors.toList());
    }

    private void processOnPremise(String serverId, JiraServerEntity jiraServerEntity) {
        cacheALMServerDTO(
                convertToIssueStatusDTOs(jiraServerEntity.getJiraIssueStatusEntities()),
                serverId,
                null,
                null
        );
    }

    public void cacheALMServerDTO(List<IssueStatusDTO> issueStatusDTOs, String serverId, String projectKeyOrId, String issueTypeId) {

        log.info("cacheALMServerDTO :: serverId = {}, projectKeyOrId = {}, issueTypeId = {}", serverId, projectKeyOrId, issueTypeId);

        middleProxyService.setAlmIssueStatusDTO(
                AlmDataRequestDTO.builder()
                        .serverId(serverId)
                        .projectKeyOrId(projectKeyOrId)
                        .issueTypeId(issueTypeId)
                        .issueStatusDTOs(issueStatusDTOs)
                        .build()
        );
    }
}