package com.arms.api.report.export_service.enums;

import lombok.Getter;
import lombok.RequiredArgsConstructor;

/**
 * 리포트 내보내기 타입을 정의하는 열거형
 *
 * <p>지원하는 내보내기 형식:
 * <ul>
 *   <li>PPT - PowerPoint 프레젠테이션 (.pptx)</li>
 *   <li>PDF - PDF 문서 (.pdf)</li>
 * </ul>
 *
 * @author HS.Yang
 * @since 25.12.01
 */
@Getter
@RequiredArgsConstructor
public enum ExportType { // PPT, PDF 등 export type

    /**
     * PowerPoint 형식
     * MIME 타입: application/vnd.openxmlformats-officedocument.presentationml.presentation
     */
    PPT("application/vnd.openxmlformats-officedocument.presentationml.presentation", ".pptx"),
    /**
     * PDF 형식
     * MIME 타입: application/pdf
     */
    PDF("application/pdf", ".pdf");

    /**
     * HTTP Content-Type 헤더에 사용될 MIME 타입
     */
    private final String contentType;
    /**
     * 파일 확장자 (점 포함)
     */
    private final String extension;

    /**
     * 문자열로부터 ExportType을 찾아 반환
     *
     * @param type 내보내기 타입 문자열 (대소문자 무관)
     * @return 해당하는 ExportType
     * @throws IllegalArgumentException 지원하지 않는 타입인 경우
     */
    public static ExportType fromString(String type) {
        try {
            return ExportType.valueOf(type.toUpperCase());
        } catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Unsupported export type: " + type);
        }
    }
}