package com.arms.api.report.export_service.model;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 리포트 메인 데이터 모델
 *
 * <p>PPT/PDF 내보내기에 사용되는 모든 데이터를 포함합니다.
 *
 * @author HS.Yang
 * @since 2025.12.02
 */
@Getter
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class ReportData {

    /** 리포트 ID */
    private String reportId;

    /** 리포트 제목 */
    private String title;

    /** 부제목 */
    private String subtitle;

    /** 설명 */
    private String description;

    /** 생성 일시 */
    @Builder.Default
    private LocalDateTime reportDate = LocalDateTime.now();

    /** 작성자 */
    private String author;

    /** 부서 */
    private String department;

    /** 템플릿 이름 */
    private String templateName;

    /** 섹션 목록 */
    @Builder.Default
    private List<ReportSection> sections = new ArrayList<>();

    /** 테이블 목록 */
    @Builder.Default
    private List<TableData> tables = new ArrayList<>();

    /** 차트 목록 */
    @Builder.Default
    private List<ChartData> charts = new ArrayList<>();

    /** 플레이스홀더 맵 */
    @Builder.Default
    private Map<String, String> placeholders = new HashMap<>();

    /** 메타데이터 */
    @Builder.Default
    private Map<String, Object> metadata = new HashMap<>();

    /**
     * 플레이스홀더 추가
     */
    public void addPlaceholder(String key, String value) {
        this.placeholders.put(key, value);
    }

    /**
     * 섹션 추가
     */
    public void addSection(ReportSection section) {
        this.sections.add(section);
    }

    /**
     * 테이블 추가
     */
    public void addTable(TableData table) {
        this.tables.add(table);
    }

    /**
     * 차트 추가
     */
    public void addChart(ChartData chart) {
        this.charts.add(chart);
    }

    /**
     * 기본 플레이스홀더 맵 생성
     *
     * @return 기본값 + 사용자 정의 플레이스홀더
     */
    public Map<String, String> buildDefaultPlaceholders() {
        Map<String, String> defaults = new HashMap<>();

        defaults.put("{{TITLE}}", title != null ? title : "");
        defaults.put("{{SUBTITLE}}", subtitle != null ? subtitle : "");
        defaults.put("{{DESCRIPTION}}", description != null ? description : "");
        defaults.put("{{AUTHOR}}", author != null ? author : "");
        defaults.put("{{DEPARTMENT}}", department != null ? department : "");
        defaults.put("{{DATE}}", reportDate != null ? reportDate.toLocalDate().toString() : "");
        defaults.put("{{DATETIME}}", reportDate != null ? reportDate.toString() : "");

        // 사용자 정의 플레이스홀더 병합 (우선)
        defaults.putAll(placeholders);

        return defaults;
    }
}
