package com.arms.api.report.export_service.model;

import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Getter;
import lombok.NoArgsConstructor;

import java.util.ArrayList;
import java.util.List;

/**
 * 리포트 섹션 모델
 *
 * <p>각 섹션은 PPT의 슬라이드 또는 PDF의 페이지로 변환됩니다.
 *
 * @author HS.Yang
 * @since 2025.12.02
 */
@Getter
@Builder
@NoArgsConstructor
@AllArgsConstructor
public class ReportSection {

    /** 섹션 ID */
    private String sectionId;

    /** 섹션 제목 */
    private String title;

    /** 본문 내용 */
    private String content;

    /** 섹션 타입 */
    @Builder.Default
    private SectionType type = SectionType.TEXT;

    /** 정렬 순서 */
    @Builder.Default
    private int order = 0;

    /** 불릿 포인트 목록 */
    @Builder.Default
    private List<String> bulletPoints = new ArrayList<>();

    /** 참조 테이블 ID */
    private String tableRef;

    /** 참조 차트 ID */
    private String chartRef;

    /** 이미지 데이터 */
    private byte[] image;

    /** 이미지 캡션 */
    private String imageCaption;

    /** 하위 섹션 */
    @Builder.Default
    private List<ReportSection> subSections = new ArrayList<>();

    /**
     * 섹션 타입 열거형
     */
    public enum SectionType {
        TEXT,        // 일반 텍스트
        BULLET_LIST, // 불릿 목록
        TABLE,       // 테이블
        CHART,       // 차트
        IMAGE,       // 이미지
        MIXED        // 복합
    }

    public void addBulletPoint(String point) {
        this.bulletPoints.add(point);
    }

    public void addSubSection(ReportSection subSection) {
        this.subSections.add(subSection);
    }

    public boolean hasTable() {
        return tableRef != null && !tableRef.isEmpty();
    }

    public boolean hasChart() {
        return chartRef != null && !chartRef.isEmpty();
    }

    public boolean hasImage() {
        return image != null && image.length > 0;
    }
}
