package com.arms.api.report.export_service.service;

import com.arms.api.report.export_service.enums.ExportType;
import com.arms.api.report.export_service.exception.ReportExportException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.util.List;

/**
 * 리포트 내보내기 서비스 팩토리
 *
 * <p>Factory 패턴으로 ExportType에 따른 서비스를 반환합니다.
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class ReportExportServiceFactory {

    private final List<ReportExportService> exportServices;

    /**
     * ExportType에 맞는 서비스 반환
     */
    public ReportExportService getService(ExportType type) {
        log.debug("Looking for export service for type: {}", type);

        for (ReportExportService service : exportServices) {
            if (service.supports(type)) {
                log.debug("Found service: {}", service.getClass().getSimpleName());
                return service;
            }
        }

        throw new ReportExportException(
                "UNSUPPORTED_TYPE",
                "No export service found for type: " + type
        );
    }

    /**
     * 타입 문자열로 서비스 반환
     */
    public ReportExportService getService(String typeName) {
        ExportType type = ExportType.fromString(typeName);
        return getService(type);
    }
}