package com.arms.api.report.export_service.template;

import com.arms.api.report.export_service.config.ReportConfig;
import com.arms.api.report.export_service.exception.TemplateNotFoundException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.xslf.usermodel.XMLSlideShow;
import org.springframework.core.io.Resource;
import org.springframework.stereotype.Component;

import java.io.IOException;
import java.io.InputStream;

/**
 * PPT 템플릿 로딩 컴포넌트
 *
 * <p>classpath 또는 파일시스템에서 .pptx 템플릿을 로드합니다.
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@Component
@RequiredArgsConstructor
public class TemplateLoader {

    private final ReportConfig reportConfig;

    /**
     * 템플릿 로드
     *
     * @param templateName 템플릿 논리명
     * @return XMLSlideShow 객체
     * @throws TemplateNotFoundException 템플릿 미발견 시
     */
    public XMLSlideShow loadPptTemplate(String templateName) {
        Resource templateResource = reportConfig.getTemplateResource(templateName);

        if (!templateResource.exists()) {
            log.error("Template not found: {}", templateName);
            throw new TemplateNotFoundException(templateName);
        }

        try (InputStream is = templateResource.getInputStream()) {
            log.info("Loading PPT template: {}", templateName);
            return new XMLSlideShow(is);
        } catch (IOException e) {
            log.error("Failed to load template: {}", templateName, e);
            throw new TemplateNotFoundException(templateName, e);
        }
    }

    /**
     * 기본 템플릿 로드
     */
    public XMLSlideShow loadDefaultTemplate() {
        return loadPptTemplate("default");
    }

    /**
     * 빈 프레젠테이션 생성
     */
    public XMLSlideShow createEmptyPresentation() {
        log.info("Creating empty presentation (no template)");
        return new XMLSlideShow();
    }

    /**
     * 템플릿 존재 여부 확인
     */
    public boolean templateExists(String templateName) {
        Resource templateResource = reportConfig.getTemplateResource(templateName);
        boolean exists = templateResource.exists();
        log.debug("Template '{}' exists: {}", templateName, exists);
        return exists;
    }

    /**
     * 폰트 로드
     */
    public InputStream loadFont(String fontName) throws IOException {
        Resource fontResource = reportConfig.getFontResource(fontName);
        if (!fontResource.exists()) {
            log.error("Font not found: {}", fontName);
            throw new IOException("Font not found: " + fontName);
        }
        log.debug("Loading font: {}", fontName);
        return fontResource.getInputStream();
    }

    /**
     * 기본 폰트 로드
     */
    public InputStream loadDefaultFont() throws IOException {
        Resource fontResource = reportConfig.getDefaultFontResource();
        return fontResource.getInputStream();
    }
}
