package com.arms.api.report.fulldata.service;

import com.arms.api.jira.jiraproject_pure.model.JiraProjectPureEntity;
import com.arms.api.jira.jiraproject_pure.service.JiraProjectPure;
import com.arms.api.product_service.pdservice.model.PdServiceAndVersionListDTO;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.report.fulldata.model.*;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.external.response.jira.AlmIssue;

import com.arms.api.util.communicate.internal.InternalService;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Slf4j
@Service("FullDataService")
@RequiredArgsConstructor
public class FullDataServiceImpl implements FullDataService {

    private final InternalService internalCommunicator;

    private final EngineService engineService;

    private final JiraProjectPure jiraProjectPure;

    @Override
    public List<작업자_정보> getAssigneeList() {
        ResponseEntity<List<작업자_정보>> listResponseEntity = engineService.getAssigneeList();
        return listResponseEntity.getBody();
    }

    @Override
    public List<ExcelDataDTO> getExcelDataDown(FullDataRequestDTO fullDataRequestDTO) throws Exception {
        return getExcelDataMapping(fullDataRequestDTO,engineService::이슈목록_가져오기);
    }

    @Override
    public List<ExcelDataDTO> getExcelData(FullDataRequestDTO fullDataRequestDTO) throws Exception {
        return getExcelDataMapping(fullDataRequestDTO,engineService::이슈목록_가져오기);
    }

    // key 가 pdServiceVerion c_id, value가 버전명(시작일~종료일) 인 맵 만들기.
    private Map<Long, String> 버전아이디_이름및일정_맵_구성 (List<PdServiceVersionEntity> 버전엔티티_세트) throws ParseException {
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

        SimpleDateFormat inputFormat = new SimpleDateFormat("yyyy/MM/dd HH:mm");
        SimpleDateFormat outputFormat = new SimpleDateFormat("yyyy-MM-dd");


        // Map 초기화
        Map<Long, String> 버전키_이름및일정_맵 = new HashMap<>();

        for (PdServiceVersionEntity entity : 버전엔티티_세트) {
            Long key = entity.getC_id();

            String startDate = outputFormat.format(inputFormat.parse(entity.getC_pds_version_start_date()));
            String endDate = outputFormat.format(inputFormat.parse(entity.getC_pds_version_end_date()));

            String value = entity.getC_title() + " (" + startDate + " ~ " + endDate + ")";
            버전키_이름및일정_맵.put(key, value);
        }

        return 버전키_이름및일정_맵;
    }

    private String 이슈_키_가져오기(String docId) {
        if (docId == null || docId.isEmpty()) {
            return "(알수없음)";
        }
        String[] parts = docId.split("_");
        return parts[parts.length - 1];
    }

    private List<ExcelDataDTO> getExcelDataMapping(FullDataRequestDTO fullDataRequestDTO, Function<FullDataRequestDTO, ResponseEntity<FullDataResponseDTO> > function ) throws Exception {
        // 반환할 엑셀 데이터 리스트
        List<ExcelDataDTO> result = new ArrayList<>();

        ResponseEntity<List<PdServiceAndVersionListDTO>> internalResponse = internalCommunicator.getPdServiceEntityAndVersionList();

        if (internalResponse.getBody() == null) {
            log.error("[ FullDataServiceImpl :: getExcelData ] :: internalCommunicator.result => Information of PdServiceAndVersion is null.");
            return result;
        }

        List<PdServiceAndVersionListDTO> 제품서비스_엔티티_버전목록 = internalResponse.getBody();

        Map<Long, String> 제품서비스_아이디_이름_맵 = new HashMap<>();
        Map<Long, String> 버전_아이디_이름_맵 = new HashMap<>();

        for (PdServiceAndVersionListDTO dto : 제품서비스_엔티티_버전목록) {

            제품서비스_아이디_이름_맵.put(dto.getC_id(),dto.getC_title());
            List<PdServiceVersionEntity> pdServiceVersionEntityList = dto.getPdServiceVersionEntityList();
            Map<Long, String> 버전아이디_이름및일정_맵_구성 = 버전아이디_이름및일정_맵_구성(pdServiceVersionEntityList);

            // 현재 entity의 맵을 total_version_map에 병합
            for (Map.Entry<Long, String> entry : 버전아이디_이름및일정_맵_구성.entrySet()) {
                버전_아이디_이름_맵.put(entry.getKey(), entry.getValue());
            }
        }

        List<Long> almProjectList = Optional.ofNullable(fullDataRequestDTO.getAlmProjectIds()).orElse(Collections.emptyList());

        if(!almProjectList.isEmpty()) {
            log.info("[ FullDataServiceImpl :: getExcelDataMapping ] :: almProjectList.size() => {}", almProjectList.size());
            List<JiraProjectPureEntity> almProjects = jiraProjectPure.getJiraProjects(fullDataRequestDTO.getAlmProjectIds());
            List<String> almProjectUrls = almProjects.stream().map(JiraProjectPureEntity::getC_jira_url).collect(Collectors.toList());
            fullDataRequestDTO.setAlmProjectUrls(almProjectUrls);
            log.info(fullDataRequestDTO.getAlmProjectUrls().toString());
        }

        ResponseEntity<FullDataResponseDTO> excelDataFromEngine = function.apply(fullDataRequestDTO);

        List<AlmIssue> AlmIssue_목록 = excelDataFromEngine.getBody().getIssueEntityList();

        if(AlmIssue_목록.isEmpty()) {
            log.info("[ FullDataServiceImpl :: getExcelData ] :: issue list is empty => 0");
            return result;
        }

        List<AlmIssue> 요구사항_목록 = 날짜_포메팅된_요구사항_이슈_목록(AlmIssue_목록);

        for (AlmIssue 요구사항_이슈 : 요구사항_목록) {

            ExcelDataDTO 요구사항이슈_데이터 = mapping(제품서비스_아이디_이름_맵, 버전_아이디_이름_맵, 요구사항_이슈);
            result.add(요구사항이슈_데이터);

            List<AlmIssue> 하위이슈_연결이슈_목록 = AlmIssue_목록.stream()
                            .filter(지라이슈 -> 지라이슈.getParentReqKey() != null
                                    && 지라이슈.getParentReqKey().equals(요구사항_이슈.getKey()))
                            .collect(Collectors.toList());
            String[] linkedIssues = 요구사항_이슈.getLinkedIssues();

            // 연결이슈 배열에 요소(element) 존재
            if (linkedIssues != null && linkedIssues.length > 0) {
                Set<String> linkedIssuesSet = Arrays.stream(linkedIssues).collect(Collectors.toSet());
                for (AlmIssue 하위_또는_연결이슈 : 하위이슈_연결이슈_목록) {
                    if (linkedIssuesSet.contains(하위_또는_연결이슈.getRecentId())) {
                        String etcContent = 요구사항_이슈.getKey() + "의 연결이슈";
                        AlmIssue 연결이슈 = 하위_또는_연결이슈.create연결이슈();
                        연결이슈.setEtc(etcContent);

                        ExcelDataDTO 연결이슈_데이터 = mapping(제품서비스_아이디_이름_맵, 버전_아이디_이름_맵, 연결이슈);
                        연결이슈_데이터.setReqTitle(요구사항_이슈.getSummary());
                        연결이슈_데이터.setReqState(Optional.ofNullable(요구사항_이슈.getCReqProperty())
                                .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                                .orElse(""));
                        result.add(연결이슈_데이터);

                        log.warn("[ FullDataServiceImpl :: getExcelDataMapping ] :: 연결이슈 발견 => {}", 연결이슈_데이터.toString());

                    } else {
                        ExcelDataDTO 하위이슈_데이터 = mapping(제품서비스_아이디_이름_맵, 버전_아이디_이름_맵, 하위_또는_연결이슈);
                        하위이슈_데이터.setReqTitle(요구사항_이슈.getSummary());
                        하위이슈_데이터.setReqState(Optional.ofNullable(요구사항_이슈.getCReqProperty())
                                .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                                .orElse(""));
                        result.add(하위이슈_데이터);

                    }
                }
            }
            // 전부 하위이슈
            else {
                for (AlmIssue 하위이슈 : 하위이슈_연결이슈_목록) {
                    ExcelDataDTO 하위이슈_데이터 = mapping(제품서비스_아이디_이름_맵, 버전_아이디_이름_맵, 하위이슈);
                    하위이슈_데이터.setReqTitle(요구사항_이슈.getSummary());
                    하위이슈_데이터.setReqState(Optional.ofNullable(요구사항_이슈.getCReqProperty())
                            .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                            .orElse(""));
                    result.add(하위이슈_데이터);
                }

            }
        }

        for (ExcelDataDTO excelDataDTO : result) {
            log.warn(excelDataDTO.getEtc());
        }

        return result;
    }

    private List<AlmIssue> 날짜_포메팅된_요구사항_이슈_목록 (List<AlmIssue> AlmIssue_목록) {

        AlmIssue_목록.stream()
            .filter(지라이슈 -> 지라이슈.getIsReq().equals(Boolean.TRUE)) // 요구사항 이슈만 필터링
            .forEach(지라이슈 -> { // 날짜 필드 처리
                지라이슈.setCreated(
                        Optional.ofNullable(지라이슈.getCreated()).map(this::timeFormatting).orElse("")
                );
                지라이슈.setUpdated(
                        Optional.ofNullable(지라이슈.getUpdated()).map(this::timeFormatting).orElse("")
                );
                지라이슈.setResolutiondate(
                        Optional.ofNullable(지라이슈.getResolutiondate()).map(this::timeFormatting).orElse("")
                );
            });

        return AlmIssue_목록.stream()
                .filter(지라이슈 -> 지라이슈.getIsReq().equals(Boolean.TRUE))
                .sorted(Comparator.comparing(AlmIssue::getUpdated, Comparator.nullsLast(Comparator.reverseOrder()))) // updated 기준 내림차순 정렬
                .collect(Collectors.toList());
    }

    private ExcelDataDTO mapping(Map<Long, String> 제품서비스_아이디_이름_맵, Map<Long, String> 버전_아이디_이름_맵, AlmIssue issue) {
        if(issue.getKey()==null){
            System.out.println(issue);
        }
        Long[] pdServiceVersions = issue.getPdServiceVersions();

        String 버전명
                = Optional.ofNullable(pdServiceVersions)
                .map(versions->Arrays.stream(versions)
                        .filter(버전_아이디_이름_맵::containsKey)
                        .map(버전_아이디_이름_맵::get)
                        .collect(Collectors.joining(",")))
                .filter(result -> !result.isEmpty())
                .orElse("버전 정보 없음");

        ExcelDataDTO.ExcelDataDTOBuilder 엑셀데이터_빌더 = ExcelDataDTO.builder()
                .pdServiceId(issue.getPdServiceId())
                .pdServiceName(Optional.ofNullable(제품서비스_아이디_이름_맵.get(issue.getPdServiceId())).orElse("제품(서비스) 정보 없음"))
                .pdServiceVersionNames(버전명) // 가져온 이슈의 버전으로 보여준다는 점이 "중요"
                .cReqLink(issue.getCReqLink());

        엑셀데이터_빌더.upperKey(issue.getUpperKey())
                .parentReqKey(issue.getParentReqKey());

        엑셀데이터_빌더
                .almProjectName(issue.projectName())
                .key(issue.getKey())
                .issueID(issue.getIssueID())
                .docId(issue.getRecentId())
                .issueTitle(issue.getSummary())
                .issueStatus(issue.statusName())
                .assigneeName(Optional.ofNullable(issue.getAssignee()).map(AlmIssue.담당자::getDisplayName).orElse("담당자 정보 없음"))
                .assigneeEmail(Optional.ofNullable(issue.getAssignee()).map(AlmIssue.담당자::getEmailAddress).orElse("담당자 메일 없음"));

        String isReqName = "";
        // 요구사항 이슈
        if (issue.getIsReq().equals(Boolean.TRUE)) {
            isReqName = "요구사항";
            엑셀데이터_빌더.reqTitle(issue.getSummary())
                    .isReq(Boolean.TRUE)
                    .isReqName(isReqName)
                    .createDate(Optional.ofNullable(issue.getCreated()).orElse("생성일 정보 없음"))
                    .updatedDate(Optional.ofNullable(issue.getUpdated()).orElse("수정일 정보 없음"))
                    .resolutionDate(Optional.ofNullable(issue.getResolutiondate()).orElse(""))
                    .reqState(Optional.ofNullable(issue.getCReqProperty())
                            .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                            .orElse("상태 정보 없음"));
        }
        // 연결 이슈
        else if (issue.getEtc() != null) {
            isReqName = String.valueOf(issue.getEtc());
            엑셀데이터_빌더
                    .isReq(Boolean.FALSE)
                    .isReqName(isReqName)
                    .reqTitle("").reqState("")
                    .createDate(Optional.ofNullable(issue.getCreated()).map(this::timeFormatting).orElse("생성일 정보 없음"))
                    .updatedDate(Optional.ofNullable(issue.getUpdated()).map(this::timeFormatting).orElse("수정일 정보 없음"))
                    .resolutionDate(Optional.ofNullable(issue.getResolutiondate()).map(this::timeFormatting).orElse(""))
                    .etc(String.valueOf(issue.getEtc()));
        // 하위이슈(isReq != True, issue.getEtc()가 null)
        } else {
            if (!Objects.equals(issue.getParentReqKey(), issue.getUpperKey()) && issue.getUpperKey() != null) {
                isReqName = issue.getUpperKey()+"의 하위이슈";
            } else {
                isReqName = issue.getParentReqKey()+"의 하위이슈";
            }
            엑셀데이터_빌더.reqTitle("").reqState("")
                    .isReq(Boolean.FALSE)
                    .isReqName(isReqName)
                    // 생성일 수정일 해결일 삭제일(있다면)
                    .createDate(Optional.ofNullable(issue.getCreated()).map(this::timeFormatting).orElse("생성일 정보 없음"))
                    .updatedDate(Optional.ofNullable(issue.getUpdated()).map(this::timeFormatting).orElse("수정일 정보 없음"))
                    .resolutionDate(Optional.ofNullable(issue.getResolutiondate()).map(this::timeFormatting).orElse(""));
        }

        return 엑셀데이터_빌더.build();
    }

    private String timeFormatting(String inputDate) {
        // 밀리초 있음 + 콜론 없음
        DateTimeFormatter formatterWithMillisNoColon = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZ");
        // 밀리초 있음 + 콜론 있음
        DateTimeFormatter formatterWithMillisWithColon = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSXXX");
        // 밀리초 없음 + 콜론 없음
        DateTimeFormatter formatterWithoutMillisNoColon = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssZ");
        // 밀리초 없음 + 콜론 있음
        DateTimeFormatter formatterWithoutMillisWithColon = DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ssXXX");

        ZonedDateTime zonedDateTime;

        // 정규식으로 포맷 판별
        if (inputDate.matches(".*\\.\\d{3}\\+\\d{4}")) {
            // 밀리초 있음 + 콜론 없음
            zonedDateTime = ZonedDateTime.parse(inputDate, formatterWithMillisNoColon);
        } else if (inputDate.matches(".*\\.\\d{3}\\+\\d{2}:\\d{2}")) {
            // 밀리초 있음 + 콜론 있음
            zonedDateTime = ZonedDateTime.parse(inputDate, formatterWithMillisWithColon);
        } else if (inputDate.matches(".*\\+\\d{4}")) {
            // 밀리초 없음 + 콜론 없음
            zonedDateTime = ZonedDateTime.parse(inputDate, formatterWithoutMillisNoColon);
        } else if (inputDate.matches(".*\\+\\d{2}:\\d{2}")) {
            // 밀리초 없음 + 콜론 있음
            zonedDateTime = ZonedDateTime.parse(inputDate, formatterWithoutMillisWithColon);
        } else {
            log.error("[ FullDataServiceImpl :: timeFormatting ] :: 지원하지 않는 날짜 형식. 들어온 값 => {}", inputDate);
            return inputDate;
        }

        DateTimeFormatter outputFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        String outputDate = zonedDateTime.format(outputFormatter);
        return outputDate;
    }
}
