package com.arms.api.report.mail.service.mail;

import com.arms.api.report.mail.modal.EmailSendResult;
import com.arms.api.report.mail.modal.entity.receiver.ReceiverEntity;
import com.arms.api.report.mail.modal.dto.sender.SenderDTO;
import com.arms.api.report.mail.modal.entity.sender.SenderEntity;
import com.arms.api.report.mail.modal.vo.TopMenuResourceVO;
import com.arms.api.report.mail.modal.vo.TopMenuScopeVO;
import com.arms.api.report.mail.modal.vo.TopMenuTimeVO;
import com.arms.api.report.mail.service.mailreport.MailReportService;
import com.arms.api.report.mail.service.receiver.ReceiverService;
import com.arms.api.report.mail.service.sender.SenderService;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import io.github.bonigarcia.wdm.WebDriverManager;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.openqa.selenium.*;
import org.openqa.selenium.chrome.ChromeDriver;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.support.ui.ExpectedConditions;
import org.openqa.selenium.support.ui.WebDriverWait;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.mail.javamail.MimeMessageHelper;
import org.springframework.stereotype.Service;
import org.thymeleaf.TemplateEngine;
import org.thymeleaf.templateresolver.ClassLoaderTemplateResolver;
import org.thymeleaf.context.Context;

import java.time.DayOfWeek;
import javax.mail.internet.MimeMessage;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.format.DateTimeFormatter;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Properties;
@Slf4j
@AllArgsConstructor
@Service("mailService")
public class MailServiceImpl extends TreeServiceImpl implements MailService{
    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    private final SenderService senderService;

    private final ReceiverService receiverService;

    private MailReportService mailReportService;

    @Override
    public EmailSendResult validateSmtpConfig(SenderDTO senderDTO) throws Exception{
        // JavaMailSenderImpl 객체 생성
        JavaMailSenderImpl mailSender = new JavaMailSenderImpl();

        // SMTP 서버 설정
        mailSender.setHost(senderDTO.getC_smtp_server());
        mailSender.setPort(senderDTO.getC_smtp_port());
        mailSender.setUsername(senderDTO.getC_sender_email());
        mailSender.setPassword(senderDTO.getC_sender_password()); // 앱 비밀번호

        Properties props = mailSender.getJavaMailProperties();
        props.put("mail.smtp.auth", "true");
        props.put("mail.smtp.starttls.enable", "true");
        props.put("mail.smtp.starttls.required", "true"); //TLS 로 강제
        props.put("mail.smtp.ssl.enable", "true"); // SSL 사용
        props.put("mail.smtp.ssl.protocols", "TLSv1.2");
        // SMTP 서버와의 연결 검증
        EmailSendResult result = new EmailSendResult(null, true, null);
        try {
            mailSender.testConnection(); // SMTP 서버에 연결 시도
            result.setSuccess(true);
            return result; // 연결 성공 시 유효한 설정
        } catch (Exception e) {
            logger.error("SMTP 설정 검증 실패: " + e.getMessage());
            result.setSuccess(false);
            result.setErrorMessage(e.getMessage());
            return result; // 연결 실패 시 유효하지 않은 설정
        }
    }
    @Override
    public List<EmailSendResult> sendTestEmailsToAllUsers() throws Exception {
        // smtp 정보 조회
        SenderEntity smtpConfig = getSenderConfig();

        List<ReceiverEntity> receiverList = getReceiverList();

        JavaMailSenderImpl mailSender =createMailSender(smtpConfig);

        List<EmailSendResult> sendResults = new ArrayList<>();

        for (ReceiverEntity receiver : receiverList) {
            EmailSendResult result = new EmailSendResult(receiver.getC_receiver_email(), true, null);
            try {
                SimpleMailMessage message = new SimpleMailMessage();
                message.setTo(receiver.getC_receiver_email());
                message.setFrom(smtpConfig.getC_sender_email());
                message.setSubject("A-RMS 테스트 이메일");
                message.setText("이메일 발송 테스트입니다.");

                mailSender.send(message); // 메일 전송
                logger.info("테스트 메일을 " + receiver.getC_receiver_email() + "에게 전송하였습니다.");
            } catch (Exception e) {
                result.setSuccess(false);
                result.setErrorMessage(e.getMessage());
                logger.error("테스트 메일 전송 실패: " + e.getMessage());
            }

            sendResults.add(result);
        }
        return sendResults; // 모든 수신자의 발송 결과 리스트 반환
    }

    @Override
    public EmailSendResult sendTestEmail(String userEmail) throws Exception{

        SenderEntity smtpConfig = getSenderConfig();

        JavaMailSenderImpl mailSender =createMailSender(smtpConfig);

        EmailSendResult result = new EmailSendResult(userEmail, true, null);
        try {
            SimpleMailMessage message = new SimpleMailMessage();
            message.setTo(userEmail);
            message.setFrom(smtpConfig.getC_sender_email());
            message.setSubject("A-RMS 테스트 이메일");
            String dateTime = LocalDateTime.now().format(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm"));
            message.setText("이메일 발송 테스트입니다.\n\n발송 시각: " + dateTime);
            mailSender.send(message); // 메일 전송
            logger.info("테스트 메일을 " + userEmail + "에게 전송하였습니다.");
        } catch (Exception e) {
            result.setSuccess(false);
            result.setErrorMessage(e.getMessage());
            logger.error("테스트 메일 전송 실패: " + e.getMessage());
        }

        return result;
    }
    // todo 스케줄러 등록 방법 확인 필요
    @Override
    public List<EmailSendResult> sendEmailsToAllUsers() throws Exception {
        // SMTP 정보 조회
        SenderEntity smtpConfig = getSenderConfig();

        LocalTime sendTime = smtpConfig.getC_send_time();
        LocalTime now = LocalTime.now();

        if (!sendTime.equals(now)) {
            log.info(" 현재 시간이 발송 시간과 일치하지 않습니다. 이메일 발송을 중단합니다. ");
            return Collections.emptyList();
        }
        // 수신자 정보 조회
        List<ReceiverEntity> receiverList = getReceiverList();

        // 메일 발송 설정
        JavaMailSenderImpl mailSender =createMailSender(smtpConfig);

        String htmlContent =fetchHtmlData();

        List<EmailSendResult> sendResults = new ArrayList<>();

        for (ReceiverEntity receiver : receiverList) {

            String receiverActivate = receiver.getC_receiver_activate();

            boolean isActivated = receiverActivate != null && Boolean.parseBoolean(receiverActivate);

            if(isActivated){
                sendResults.add(
                        sendHtmlEmail(
                                mailSender,
                                receiver.getC_receiver_email(),
                                smtpConfig.getC_sender_email(),
                                smtpConfig.getC_email_subject(),
                                htmlContent));
            }
        }

        return sendResults;
    }

    // SMTP 설정 조회
    @Override
    public SenderEntity getSenderConfig() throws Exception {
        List<SenderEntity> senderInfo = senderService.getNodesWithoutRoot(new SenderEntity());
        if (senderInfo == null || senderInfo.isEmpty()) {
            throw new RuntimeException("SMTP 설정을 찾을 수 없습니다.");
        }
        return senderInfo.get(0);
    }

    // 수신자 정보 조호
    private List<ReceiverEntity> getReceiverList() throws Exception {
        List<ReceiverEntity> receiverList = receiverService.getNodesWithoutRoot(new ReceiverEntity());
        if (receiverList == null || receiverList.isEmpty()) {
            throw new RuntimeException("수신자 이메일이 존재하지 않습니다.");
        }
        return receiverList;
    }

    // SMTP 설정
    @Override
    public JavaMailSenderImpl createMailSender(SenderEntity smtpConfig) {

        JavaMailSenderImpl mailSender = new JavaMailSenderImpl();

        mailSender.setHost(smtpConfig.getC_smtp_server());
        mailSender.setPort(smtpConfig.getC_smtp_port());
        mailSender.setUsername(smtpConfig.getC_sender_email());
        mailSender.setPassword(smtpConfig.getC_sender_password());

        Properties props = mailSender.getJavaMailProperties();
//        spring.mail.properties.mail.smtp.auth=true
//        spring.mail.properties.mail.smtp.ssl.enable=true
//        spring.mail.properties.mail.smtp.starttls.enable=false
        props.put("mail.smtp.auth", "true");
        props.put("mail.smtp.ssl.enable", "true");
        props.put("mail.smtp.starttls.enable", "false");
        props.put("mail.smtp.ssl.protocols", "TLSv1.2");

        return mailSender;
    }

    // todo 12/19 차트 이미지르 메일 첨부
    //  현재까지 샘플로 static 하위 경로에 있는 로고 이미지를 로드하여 메일에 직접 첨부 하는 방식으로 메일을 발송하였음. (url 형태로 이미지 경로 제공 x)
    //  "차트 이미지 첨부 또는 링크 거는 방법" 중 어떤 방법을 사용할 지 확실히하고 메일에 차트 노츌 가능하도록
    //  /arms/template.html?page=mailReport 참고 -> 메일 리포트 페이지
    private EmailSendResult sendHtmlEmail(JavaMailSenderImpl mailSender, String toEmail, String fromEmail, String subject, String htmlContent) {
        EmailSendResult result = new EmailSendResult(toEmail, true, null);
        try {
            MimeMessage mimeMessage = mailSender.createMimeMessage();
            MimeMessageHelper helper = new MimeMessageHelper(mimeMessage, true, "UTF-8");
            helper.setTo(toEmail);
            helper.setFrom(fromEmail);
            helper.setSubject(subject);
            helper.setText(htmlContent, true);

            // 이미지 첨부
            ClassPathResource imageResource = new ClassPathResource("static/313_logo_large_drop.png");
            helper.addInline("logoImage", imageResource);

            mailSender.send(mimeMessage);
            logger.info("HTML 메일을 " + toEmail + "에게 전송하였습니다.");
        } catch (Exception e) {
            result.setSuccess(false);
            result.setErrorMessage(e.getMessage());
            logger.error("HTML 메일 전송 실패: " + e.getMessage());
        }
        return result;
    }
    // todo 12/19 탑매뉴 해댕하는 차트 데이터 까지만 랜더링 작업 하였음
    //  분석 매뉴 차트
    @Override
    public String fetchHtmlData() throws Exception {
        // todo 12/19 하위 3개의 api는 진행중인 버전을 연산하는 과정이 중복되어있음, 효율을 위해 중복 제거 필요함
        TopMenuTimeVO timeData = mailReportService.getTopMenuTime();
        TopMenuScopeVO scopeData = mailReportService.getTopMenuScope();
        TopMenuResourceVO resourceData = mailReportService.getTopMenuResource();

        LocalDate now = LocalDate.now();

        LocalDate previousMonday = now.minusWeeks(1).with(TemporalAdjusters.previousOrSame(DayOfWeek.MONDAY));
        LocalDate previousSunday = previousMonday.with(TemporalAdjusters.nextOrSame(DayOfWeek.SUNDAY));

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy년 MM월 dd일");
        String formattedStartDate = previousMonday.format(formatter);
        String formattedEndDate = previousSunday.format(formatter);

        ClassLoaderTemplateResolver templateResolver = new ClassLoaderTemplateResolver();
        templateResolver.setPrefix("/templates/");
        templateResolver.setSuffix(".html");
        templateResolver.setTemplateMode("HTML");
        templateResolver.setCharacterEncoding("UTF-8");

        TemplateEngine templateEngine = new TemplateEngine();
        templateEngine.setTemplateResolver(templateResolver);

        Context context = new Context();

        context.setVariable("startDate", formattedStartDate);
        context.setVariable("endDate", formattedEndDate);

        context.setVariable("timeData", timeData);
        context.setVariable("scopeData", scopeData);
        context.setVariable("resourceData", resourceData);

        try {
            return templateEngine.process("mail", context);
        } catch (Exception e) {
            logger.error("mail.html 템플릿 처리 중 오류 발생: " + e.getMessage());
            throw new Exception("mail.html 템플릿 처리 중 오류 발생.", e);
        }
    }
    // todo 12/19 차트별 이미지 변환 로직 (디비에 저장 전)
    //  가상 브라우저로 GET /html/mail/chart.html 을 호출
    //  -> chart.html 랜더링 되면 echart 이미지를 /html/mail/save-chart POST 전달
    //  -> /html/mail/save-chart 에서 전달 받은 차트이미지 /resource/static/saved-chart.png 로 저장
    @Override
    public String generateChartImage() throws Exception {

        WebDriverManager.chromedriver().setup();
        ChromeOptions options = new ChromeOptions();
        options.addArguments("--headless", "--disable-gpu", "--window-size=1920,1080");
        WebDriver driver = new ChromeDriver(options);

        try {
            driver.get("http://localhost:31313/html/mail/chart.html");

            WebDriverWait wait = new WebDriverWait(driver,10); // 최대 10초 까지 기다림
            wait.until(ExpectedConditions.presenceOfElementLocated(By.id("progress-overview-chart"))); // 차트 렌더링 대기
            return "success";
        } catch (Exception e) {
            throw new Exception("Failed to trigger", e);
        } finally {
            driver.quit();
        }
    }

}
