package com.arms.api.report.mail.service.mailreport;

import com.arms.api.product_service.pdservice.model.PdServiceAndVersionListDTO;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.report.fulldata.model.FullDataRequestDTO;
import com.arms.api.report.fulldata.model.FullDataResponseDTO;
import com.arms.api.report.mail.modal.vo.AssigneeIssueCountVO;
import com.arms.api.report.mail.modal.vo.TopMenuResourceVO;
import com.arms.api.report.mail.modal.vo.TopMenuScopeVO;
import com.arms.api.report.mail.modal.vo.TopMenuTimeVO;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.external.response.jira.AlmIssue;
import com.arms.api.util.communicate.internal.InternalService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

@Slf4j
@AllArgsConstructor
@Service("mailReportService")
public class MailReportServiceImpl implements MailReportService{

    private final InternalService internalService;

    private final EngineService engineService;

    @Override
    public TopMenuTimeVO getTopMenuTime() {

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm");
        DateTimeFormatter dateFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        List<PdServiceAndVersionListDTO> pdServices = internalService.getPdServiceEntityAndVersionList().getBody();

        if (pdServices == null) {
            String errorLog = "[MailReportServiceImpl :: getTopMenuTime] :: internalService.PdServiceEntityAndVersionList is null.";
            log.error(errorLog);
            throw new RuntimeException(errorLog);
        }

        int total = 0, ongoing = 0, completed = 0, upcoming = 0;
        LocalDateTime now = LocalDateTime.now();

        LocalDate earliestStartDate = null;
        LocalDate latestEndDate = null;

        for (PdServiceAndVersionListDTO pdService : pdServices) {
            List<PdServiceVersionEntity> versions = pdService.getPdServiceVersionEntityList();

            for (PdServiceVersionEntity version : versions) {
                total++;
                LocalDateTime startDateTime = LocalDateTime.parse(version.getC_pds_version_start_date(), formatter);
                LocalDateTime endDateTime = LocalDateTime.parse(version.getC_pds_version_end_date(), formatter);

                LocalDate startDate = startDateTime.toLocalDate();
                LocalDate endDate = endDateTime.toLocalDate();

                if (now.isBefore(startDateTime)) {
                    upcoming++;
                } else if (now.isAfter(endDateTime)) {
                    completed++;
                } else {
                    ongoing++;

                    if (earliestStartDate == null || startDate.isBefore(earliestStartDate)) {
                        earliestStartDate = startDate;
                    }
                    if (latestEndDate == null || endDate.isAfter(latestEndDate)) {
                        latestEndDate = endDate;
                    }
                }
            }
        }

        long daysElapsed = 0;
        int elapsedPercentage = 0;

        if (earliestStartDate != null && latestEndDate != null) {
            // 경과 일자 계산
            daysElapsed = java.time.Duration.between(earliestStartDate.atStartOfDay(), now).toDays();

            // 경과 퍼센티지 계산 (소수점 제거)
            long totalDuration = java.time.Duration.between(earliestStartDate.atStartOfDay(), latestEndDate.atStartOfDay()).toMillis();
            long elapsedDuration = java.time.Duration.between(earliestStartDate.atStartOfDay(), now).toMillis();
            elapsedPercentage = (int) ((elapsedDuration * 100) / totalDuration);
        }

        String earliestStart = earliestStartDate != null ? earliestStartDate.format(dateFormatter) : null;
        String latestEnd = latestEndDate != null ? latestEndDate.format(dateFormatter) : null;

        return new TopMenuTimeVO(
                total,
                ongoing,
                completed,
                upcoming,
                daysElapsed,
                elapsedPercentage,
                earliestStart,
                latestEnd
        );
    }

    @Override
    public TopMenuScopeVO getTopMenuScope() throws Exception {
        try {
            // 진행 중인 제품 - 버전
            List<PdServiceAndVersionListDTO> services = getOngoingPdServiceAndVersionList();

            FullDataRequestDTO requestDTO = setFullDataRequestDTO(services);

            FullDataResponseDTO responseDTO = engineService.getReqIssueList(requestDTO).getBody();

            if (responseDTO == null) {
                String errorLog = "[MailReportServiceImpl :: getTopMenuScope] :: engineService.getReqIssueList(requestDTO).getBody() is null.";
                log.error(errorLog);
                throw new RuntimeException(errorLog);
            }

            Long reqTotalCount = responseDTO.getTotalHits();
            List<AlmIssue> reqList = responseDTO.getIssueEntityList();
            Map<String, Long> stateMap = countByCReqStateLink(reqList);
            Long createdReqCount = countIssuesCreatedThisWeek(reqList);

            return new TopMenuScopeVO(
                    reqTotalCount,                             // totalReqCount
                    stateMap.getOrDefault("열림", 0L),         // openReqCount
                    stateMap.getOrDefault("진행중", 0L),       // progressReqCount
                    stateMap.getOrDefault("해결됨", 0L),       // resolvedReqCount
                    stateMap.getOrDefault("닫힘", 0L),         // closedReqCount
                    createdReqCount,
                    0
            );
        } catch (Exception e){
            throw new Exception("메일 리포트 탑메뉴 스콥 조회시 오류 발생하였습니다."+e.getMessage());
        }
    }

    @Override
    public TopMenuResourceVO getTopMenuResource() throws Exception {
        try {
            List<PdServiceAndVersionListDTO> services = getOngoingPdServiceAndVersionList();

            FullDataRequestDTO requestDTO = setFullDataRequestDTO(services);

            AssigneeIssueCountVO responseDTO = engineService.getAssigneeAggregationList(requestDTO).getBody();

            if (responseDTO == null) {
                String errorLog = "[MailReportServiceImpl :: getTopMenuResource] :: engineService.getAssigneeAggregationList(requestDTO).getBody() is null.";
                log.error(errorLog);
                throw new RuntimeException(errorLog);
            }

            return new TopMenuResourceVO(
                    responseDTO.getTotalIssueCount(),
                    responseDTO.getTotalAssigneeCount(),
                    responseDTO.getAvgIssuesPerAssignee(),
                    responseDTO.getMaxIssues(),
                    responseDTO.getMinIssues()
            );
        } catch (Exception e){
            throw new Exception("메일 리포트 탑메뉴 리소스 조회시 오류 발생하였습니다."+e.getMessage());
        }
    }
    
    public long countIssuesCreatedThisWeek(List<AlmIssue> issues) {

        // 오늘 날짜 기준으로 이번 주의 시작일(월요일)과 종료일(일요일) 계산
        Calendar calendar = Calendar.getInstance();
        calendar.set(Calendar.DAY_OF_WEEK, Calendar.MONDAY);
        Date weekStartDate = calendar.getTime(); // 이번 주 월요일
        calendar.add(Calendar.DATE, 6);
        Date weekEndDate = calendar.getTime(); // 이번 주 일요일

        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd");

        return issues.stream()
                .filter(issue -> {
                    try {
                        Date createdDate = dateFormat.parse(issue.getCreated());
                        return !createdDate.before(weekStartDate) && !createdDate.after(weekEndDate); //이번 주 범위 안에 있는지 확인
                    } catch (Exception e) {
                        return false;
                    }
                })
                .count();
    }

    private FullDataRequestDTO setFullDataRequestDTO(List<PdServiceAndVersionListDTO> services){

        List<Long> pdServiceCids = services.stream()
                .map(PdServiceAndVersionListDTO::getC_id)
                .collect(Collectors.toList());

        List<Long> pdServiceVersionIds = services.stream()
                .flatMap(service -> service.getPdServiceVersionEntityList().stream())
                .map(PdServiceVersionEntity::getC_id)
                .collect(Collectors.toList());

        FullDataRequestDTO fullDataRequestDTO = new FullDataRequestDTO();
        fullDataRequestDTO.setPdServiceIds(pdServiceCids);
        fullDataRequestDTO.setPdServiceVersionIds(pdServiceVersionIds);
        
        return fullDataRequestDTO;

    }

    public Map<String, Long> countByCReqStateLink(List<AlmIssue> reqList) {
        return reqList.stream()
                .map(AlmIssue::getCReqProperty)
                .filter(Objects::nonNull)
                .map(AlmIssue.암스_요구사항_속성::getCReqStateName)
                .filter(Objects::nonNull)
                .collect(Collectors.groupingBy(stateLink -> stateLink, Collectors.counting()));
    }

    private List<PdServiceAndVersionListDTO> getOngoingPdServiceAndVersionList() {

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm");
        LocalDateTime now = LocalDateTime.now();

        List<PdServiceAndVersionListDTO> pdServices = internalService.getPdServiceEntityAndVersionList().getBody();

        if (pdServices == null) {
            String errorLog = "[MailReportServiceImpl :: getOngoingPdServiceAndVersionList] :: internalService.getPdServiceEntityAndVersionList().getBody() is null.";
            log.error(errorLog);
            throw new RuntimeException(errorLog);
        }

        return pdServices.stream()
                .map(pdService -> {
                    List<PdServiceVersionEntity> ongoingVersions = pdService.getPdServiceVersionEntityList().stream()
                            .filter(version -> {
                                LocalDateTime startDate = LocalDateTime.parse(version.getC_pds_version_start_date(), formatter);
                                LocalDateTime endDate = LocalDateTime.parse(version.getC_pds_version_end_date(), formatter);
                                return now.isAfter(startDate) && now.isBefore(endDate);
                            })
                            .collect(Collectors.toList());

                    if (!ongoingVersions.isEmpty()) {
                        pdService.setPdServiceVersionEntityList(ongoingVersions);
                        return pdService;
                    }
                    return null;
                })
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

}
