package com.arms.api.requirement.reqadd_state_pure.service;

import com.arms.api.product_service.pdservice.model.PdServiceEntity;
import com.arms.api.product_service.pdservice.service.PdService;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.requirement.reqadd.model.dto.ReqAnalysisDTO;
import com.arms.api.requirement.reqadd_state_pure.model.ReportReqAddEntity;
import com.arms.api.requirement.reqadd_state_pure.model.ReqAddStatePureEntity;
import com.arms.api.requirement.reqadd_state_pure.model.vo.*;
import com.arms.api.requirement.reqstate.model.ReqStateEntity;
import com.arms.api.requirement.reqstate.service.ReqState;
import com.arms.api.util.VersionUtil;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import com.arms.egovframework.javaservice.treeframework.remote.Chat;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import com.arms.egovframework.javaservice.treeframework.util.DateUtils;
import lombok.AllArgsConstructor;
import org.apache.commons.lang.StringUtils;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.WeekFields;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;

@AllArgsConstructor
@Service("reqAddStatePure")
public class ReqAddStatePureImpl extends TreeServiceImpl implements ReqAddStatePure {

	private final Logger logger = LoggerFactory.getLogger(this.getClass());

	@Autowired
	private AggregationService aggregationService;

	@Autowired
	private PdService pdService;

	@Autowired
	@Qualifier("reqState")
	private ReqState reqState;

	@Autowired
	protected Chat chat;

	@Override
	public List<ReqAddStatePureEntity> reqProgress(ReqAddStatePureEntity reqAddStatePureEntity, String changeReqTableName,
												   Long pdServiceId, String c_req_pdservice_versionset_link) throws Exception {

		String[] versionStrArr = StringUtils.split(c_req_pdservice_versionset_link, ",");

		SessionUtil.setAttribute("reqProgress", changeReqTableName);

		List<ReqAddStatePureEntity> 전체요구사항_목록;
		if (versionStrArr == null || versionStrArr.length == 0) {
			reqAddStatePureEntity.setOrder(Order.asc("c_position"));
			전체요구사항_목록 = this.getNodesWithoutRoot(reqAddStatePureEntity);
		}
		else {
			Disjunction orCondition = Restrictions.disjunction();
			for (String versionStr : versionStrArr) {
				versionStr = "\\\"" + versionStr + "\\\"";
				orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", versionStr, MatchMode.ANYWHERE));
			}
			orCondition.add(Restrictions.eq("c_type", TreeConstant.Branch_TYPE));
			reqAddStatePureEntity.getCriterions().add(orCondition);
			reqAddStatePureEntity.setOrder(Order.asc("c_position"));

			전체요구사항_목록 = this.getChildNode(reqAddStatePureEntity);
		}

		SessionUtil.removeAttribute("reqProgress");

		List<Long> pdServiceVersionLinks = null;
		if (versionStrArr != null && versionStrArr.length > 0) {
			pdServiceVersionLinks = Arrays.stream(versionStrArr)
					.map(Long::valueOf)
					.collect(Collectors.toList());
		}

		PdServiceAndIsReqDTO pdServiceAndIsReq = new PdServiceAndIsReqDTO();
		pdServiceAndIsReq.setPdServiceLink(pdServiceId);
		pdServiceAndIsReq.setPdServiceVersionLinks(pdServiceVersionLinks);

		ReqAnalysisDTO reqAnalysisDTO = new ReqAnalysisDTO();
		reqAnalysisDTO.setPdServiceAndIsReq(pdServiceAndIsReq);

		// 1.1 resolution 필드에 값 넣는것 없이, 집계 조회
		ResponseEntity<List<ReqProgressVO>> 일반_버전필터_집계 = aggregationService.제품서비스_일반_버전_해결책유무_통계_V2(reqAnalysisDTO, null);
		// 1.2 resolution 에 "resolutiondate" 넣어서 집계 조회
		ResponseEntity<List<ReqProgressVO>> 완료상태 = aggregationService.제품서비스_일반_버전_해결책유무_통계_V2(reqAnalysisDTO, "resolutiondate");

		Map<Long, Map<String, Long>> 진행률계산맵 = new HashMap<>();

		// 제품 아이디, 버전들의 적용된
		List<ReqProgressVO> 일반_버전필터_집계결과목록 = Optional.ofNullable(일반_버전필터_집계.getBody()).orElse(new ArrayList<>());
		집계결과처리(진행률계산맵, 일반_버전필터_집계결과목록, "전체");

		List<ReqProgressVO> 완료상태집계결과목록 = Optional.ofNullable(완료상태.getBody()).orElse(new ArrayList<>());
		집계결과처리(진행률계산맵, 완료상태집계결과목록, "완료");

		Map<Long, ReqStateEntity> 완료상태맵 = reqState.완료상태조회();

		// 실적, 계획 진행퍼센트 처리
		List<ReqAddStatePureEntity> 실적계산_결과목록 = 전체요구사항_목록.stream().map(요구사항_엔티티 -> {

					// 시작일, 종료일 데이터 있을 시 계획 진척율, 실적 진척율 계산
					if (요구사항_엔티티.getC_req_start_date() != null && 요구사항_엔티티.getC_req_end_date() != null) {
						Date 시작일 = 요구사항_엔티티.getC_req_start_date();
						Date 종료일 = 요구사항_엔티티.getC_req_end_date();
						Date 오늘 = new Date();

						// 총 작업량 계산(종료일 - 시작일)
						long 총작업량 = DateUtils.getRoundedDiffDays(시작일, 종료일);

						// 계획 진행률 계산
						long 진행율 = 계획진행률_계산(시작일, 종료일, 오늘);
						long 계획작업량 = 0;
						// 진행율에 따라서 작업량 계산(시작일과 종료일, 오늘 기준으로 계산)
						if (진행율 > 0L && 진행율 < 100L) {
							계획작업량 = DateUtils.getRoundedDiffDays(시작일, 오늘);
						}
						else if (진행율 == 100L) {
							계획작업량 = 총작업량;
						}

						요구사항_엔티티.setC_req_total_resource(총작업량);
						요구사항_엔티티.setC_req_plan_resource(계획작업량);
						요구사항_엔티티.setC_req_plan_progress(진행율);
					}
					else {
						요구사항_엔티티.setC_req_total_resource(0L);
						요구사항_엔티티.setC_req_plan_resource(0L);
						요구사항_엔티티.setC_req_plan_progress(0L);
					}

					// 폴더 타입 요구사항은 실적계산 전 리턴
					if (요구사항_엔티티.getC_type() != null && StringUtils.equals(요구사항_엔티티.getC_type(), TreeConstant.Branch_TYPE)) {
						return 요구사항_엔티티;
					}

					// 요구사항 req state가 완료상태일 경우 실적 100% 처리
					if (요구사항_엔티티.getReqStateEntity() != null && 완료상태맵.get(요구사항_엔티티.getReqStateEntity().getC_id()) != null) {
						요구사항_엔티티.setC_req_performance_progress(100L);
					}
					else {
						Map<String, Long> 전체완료맵 = 진행률계산맵.get(요구사항_엔티티.getC_id());
						if (전체완료맵 != null) {
							Long 전체개수 = 전체완료맵.getOrDefault("전체", 0L);
							Long 완료개수 = 전체완료맵.getOrDefault("완료", 0L);

							Long 진행률 = 실적계산(전체개수, 완료개수);

							요구사항_엔티티.setC_req_performance_progress(진행률);
						}
					}

					return 요구사항_엔티티;
				})
				.filter(Objects::nonNull)
				.collect(Collectors.toList());

		return 실적계산_결과목록;
	}

	private void 집계결과처리(Map<Long, Map<String, Long>> 진행률계산맵, List<ReqProgressVO> reqProgressListByEngine, String 상태) {
		if (reqProgressListByEngine.isEmpty()) {
			logger.error("[ReqAddStatePureImpl ::집계결과처리] :: reqProgressListByEngine is empty");
			return ;
		}
		for (ReqProgressVO reqProgressVO : reqProgressListByEngine) {
			진행률계산맵.computeIfAbsent(reqProgressVO.getCReqLink(), k-> new HashMap<>())
					.put(상태, reqProgressVO.getDocCount());
		}
	}

	private Long 실적계산(Long 전체개수, Long 완료개수) {
		return 전체개수 > 0 ? (완료개수 * 100) / 전체개수 : 0L;
	}

	private long 계획진행률_계산(Date 시작일, Date 종료일, Date 오늘) {
		// 시작일이 종료일과 같거나 이후일 경우 진행율 0 처리
		if (!시작일.before(종료일)) {
			return 0L;
		}

		// 오늘이 시작일 이전인 경우, 진행율 0 처리
		if (오늘.before(시작일)) {
			return 0L;
		}

		// 오늘이 종료일 이후의 경우, 진행율 100 처리
		if (오늘.after(종료일)) {
			return 100L;
		}

		// 프로젝트가 진행 중인 상태
		long 전체일수 = DateUtils.getRoundedDiffDays(시작일, 종료일);
		long 진행일수 = DateUtils.getRoundedDiffDays(시작일, 오늘);
		long 진행율 = 실적계산(전체일수, 진행일수);

		// 진행율 100 최댓값 처리
		return Math.min(진행율, 100L);
	}

	@Override
	public ProgressResultVO calculateProgress(ReqAddStatePureEntity reqAddStatePureEntity, String changeReqTableName,
											  Long pdServiceId, String c_req_pdservice_versionset_link) throws Exception {

		int totalWork = 0;
		int planWork = 0;
		double totalPerformanceProgress = 0.0;
		int reqCount = 0;

		Date today = new Date();

		List<ReqAddStatePureEntity> performanceCalcReqList = this.reqProgress(reqAddStatePureEntity,changeReqTableName,pdServiceId,c_req_pdservice_versionset_link);

		for (ReqAddStatePureEntity cur : performanceCalcReqList) {
			if (!"default".equals(cur.getC_type())) continue;

			totalWork += cur.getC_req_total_resource();
			planWork += cur.getC_req_plan_resource();

			if (cur.getC_req_start_date() != null && cur.getC_req_end_date() != null && cur.getC_req_start_date().before(today)) {
				totalPerformanceProgress += cur.getC_req_performance_progress() != null ? cur.getC_req_performance_progress() : 0;
				reqCount++;
			}
		}

		double performanceProgressRate = reqCount > 0 ? totalPerformanceProgress / reqCount : 0.0;
		double planProgressRate = totalWork > 0 ? ((double) planWork / totalWork) * 100.0 : 0.0;
		double projectProgress = performanceProgressRate - planProgressRate;
		double performanceCapability = ((totalWork * performanceProgressRate) / 100);

		return ProgressResultVO.builder()
				.totalWork(totalWork)
				.planWork(planWork)
				.performanceCapability(performanceCapability)
				.planProgressRate(planProgressRate)
				.performanceProgressRate(performanceProgressRate)
				.projectProgress(projectProgress)
				.build();
	}

	@Override
	public List<ReportReqAddVO> reportReqStatus(ReportReqAddEntity reportReqAddEntity, String changeReqTableName,
												Long pdServiceId, String c_req_pdservice_versionset_link) throws Exception {

		PdServiceEntity pdServiceForSearch = new PdServiceEntity();
		pdServiceForSearch.setC_id(pdServiceId);

		PdServiceEntity pdServiceEntity = pdService.getNode(pdServiceForSearch);

		Set<PdServiceVersionEntity> pdServiceVersionEntities = pdServiceEntity.getPdServiceVersionEntities();
		Map<Long, String> versionIdNameMap = new HashMap<>();
		for (PdServiceVersionEntity entity : pdServiceVersionEntities) {
			versionIdNameMap.put(entity.getC_id(), entity.getC_title());
		}

		String[] versionStrArr = StringUtils.split(c_req_pdservice_versionset_link, ",");

		SessionUtil.setAttribute("reportReqStatus", changeReqTableName);

		List<ReportReqAddEntity> reportReqAddList;
		if (versionStrArr == null || versionStrArr.length == 0) {
			reportReqAddEntity.setOrder(Order.asc("c_position"));
			reportReqAddList = this.getNodesWithoutRoot(reportReqAddEntity);
		}
		else {
			Disjunction orCondition = Restrictions.disjunction();
			for (String versionStr : versionStrArr) {
				versionStr = "\\\"" + versionStr + "\\\"";
				orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", versionStr, MatchMode.ANYWHERE));
			}
			orCondition.add(Restrictions.eq("c_type", TreeConstant.Branch_TYPE));
			reportReqAddEntity.getCriterions().add(orCondition);
			reportReqAddEntity.setOrder(Order.asc("c_position"));

			reportReqAddList = this.getChildNode(reportReqAddEntity);
		}

		SessionUtil.removeAttribute("reportReqStatus");

		List<ReportReqAddVO> reportReqAddVOList = reportReqAddList.stream()
				.filter(Objects::nonNull)
				.map(reqAdd -> {
					return ReportReqAddVO.builder()
							.title(reqAdd.getC_title() == null ? null : reqAdd.getC_id() + ". " + reqAdd.getC_title())
							.versions(getVersionNames(versionIdNameMap, VersionUtil.stringToLongArray(reqAdd.getC_req_pdservice_versionset_link())))
							.status(reqAdd.getReqStateEntity() == null ? null : reqAdd.getReqStateEntity().getC_title())
							.priority(reqAdd.getReqPriorityEntity() == null ? null : reqAdd.getReqPriorityEntity().getC_title())
							.difficulty(reqAdd.getReqDifficultyEntity() == null ? null : reqAdd.getReqDifficultyEntity().getC_title())
							.start_date(reqAdd.getC_req_start_date())
							.end_date(reqAdd.getC_req_end_date())
							.build();
				})
				.collect(Collectors.toList());

		return reportReqAddVOList;
	}

	@Override
	public List<WeeklyReportReqAddVO> weeklyReportReqStatus(ReqAddStatePureEntity reqAddStatePureEntity, String changeReqTableName,
												Long pdServiceId, String c_req_pdservice_versionset_link) throws Exception {

		SessionUtil.setAttribute("weeklyReportReqStatus", changeReqTableName);

		List<ReqAddStatePureEntity> performanceCalcReqList = this.reqProgress(reqAddStatePureEntity, changeReqTableName, pdServiceId, c_req_pdservice_versionset_link);

		SessionUtil.removeAttribute("weeklyReportReqStatus");

		WeekFields weekFields = WeekFields.of(Locale.getDefault());
		Map<String, WeeklyReportReqAddVO> weeklyDataMap = new HashMap<>();

		performanceCalcReqList.stream()
				.filter(Objects::nonNull)
				.forEach(reqAdd -> {
					Date cReqStartDate = reqAdd.getC_req_start_date();
					Date cReqEndDate = reqAdd.getC_req_end_date();

					LocalDate startDate = cReqStartDate.toInstant()
							.atZone(ZoneId.systemDefault())
							.toLocalDate();

					LocalDate endDate = cReqEndDate.toInstant()
							.atZone(ZoneId.systemDefault())
							.toLocalDate();

					DateTimeFormatter formatter = DateTimeFormatter.ofPattern("MM.dd");

					LocalDate weekStart = startDate.with(weekFields.dayOfWeek(), 1);
					LocalDate weekEnd = endDate;

					while (!weekStart.isAfter(weekEnd)) {
						int year = weekStart.get(weekFields.weekBasedYear());
						int week = weekStart.get(weekFields.weekOfWeekBasedYear());

						LocalDate weekEndDate = weekStart.with(weekFields.dayOfWeek(), 7);
						String dateRange = formatter.format(weekStart) + "~" + formatter.format(weekEndDate);

						String key = year + "년 " + week + "주차";
						String weekLabel = year + "년 " + week + "주차(" + dateRange + ")";

						WeeklyReportReqAddVO weeklyReportReqAddVO = weeklyDataMap.getOrDefault(
								key,
								WeeklyReportReqAddVO.builder()
										.key(key)
										.weekLabel(weekLabel)
										.year(year)
										.week(week)
										.plan(new WeeklyReportReqAddVO.PlanAmount(0, 0, 0, 0))
										.actual(new WeeklyReportReqAddVO.ActualAmount(0, 0, 0))
										.build()
						);

						int planCount = weeklyReportReqAddVO.getPlan().getCount() + 1;
						int newPlanTotal = weeklyReportReqAddVO.getPlan().getTotal() + reqAdd.getC_req_total_resource().intValue();
						int newPlanResource = weeklyReportReqAddVO.getPlan().getResource() + reqAdd.getC_req_plan_resource().intValue();

						weeklyReportReqAddVO.getPlan().setCount(planCount);
						weeklyReportReqAddVO.getPlan().setTotal(newPlanTotal);
						weeklyReportReqAddVO.getPlan().setResource(newPlanResource);

						int actualCount = weeklyReportReqAddVO.getActual().getCount() + 1;
						int performanceProgress = reqAdd.getC_req_performance_progress() == null ? 0 : reqAdd.getC_req_performance_progress().intValue();
						int actualResource = weeklyReportReqAddVO.getActual().getResource() + (reqAdd.getC_req_total_resource().intValue() * performanceProgress / 100);

						weeklyReportReqAddVO.getActual().setCount(actualCount);
						weeklyReportReqAddVO.getActual().setResource(actualResource);

						weeklyDataMap.put(key, weeklyReportReqAddVO);

						weekStart = weekStart.plusWeeks(1);
					}
				});

		AtomicInteger planAccumulation = new AtomicInteger();
		AtomicInteger actualAccumulation = new AtomicInteger();

		weeklyDataMap.forEach((s, weeklyReportReqAddVO) -> {
			planAccumulation.addAndGet(weeklyReportReqAddVO.getPlan().getResource());
			actualAccumulation.addAndGet(weeklyReportReqAddVO.getActual().getResource());
		});

		List<WeeklyReportReqAddVO> weeklyReportReqAddVOs = new ArrayList<>(weeklyDataMap.values());
		List<WeeklyReportReqAddVO> weeklyReportReqAddList = weeklyReportReqAddVOs
				.stream()
				.map(weeklyReportReqAddVO -> {
						int planResource = weeklyReportReqAddVO.getPlan().getResource();
						int totalPlan = planAccumulation.intValue();

						if (totalPlan > 0) {
							weeklyReportReqAddVO.getPlan().setProgress((double) planResource * 100 / totalPlan); // 퍼센트 표시 시
						} else {
							weeklyReportReqAddVO.getPlan().setProgress(0);
						}

						int actualResource = weeklyReportReqAddVO.getActual().getResource();

						if (totalPlan > 0) {
							weeklyReportReqAddVO.getActual().setProgress((double) actualResource * 100 / totalPlan); // 퍼센트 표시 시
						} else {
							weeklyReportReqAddVO.getActual().setProgress(0);
						}

						return weeklyReportReqAddVO;
				})
				.collect(Collectors.toList());

		weeklyReportReqAddList.sort(Comparator.comparing(WeeklyReportReqAddVO::getYear).thenComparing(WeeklyReportReqAddVO::getWeek));
		return weeklyReportReqAddList;
	}

	private String getVersionNames(Map<Long, String> versionNameMap, Long[] versionArr) {
		String key =
				Arrays.stream(versionArr)
						.map(version-> versionNameMap.getOrDefault(version,""))
						.filter(name -> !name.isEmpty())  // 빈 문자열 제거
						.collect(Collectors.joining(","));
		return key;
	}
}
