package com.arms.api.util.aspect;

import com.arms.api.report.mail.modal.EmailSendResult;
import com.arms.api.report.mail.modal.entity.sender.SenderEntity;
import com.arms.api.report.mail.service.mail.MailService;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.JoinPoint;
import org.aspectj.lang.annotation.*;
import org.springframework.core.env.Environment;
import org.springframework.mail.SimpleMailMessage;
import org.springframework.mail.javamail.JavaMailSenderImpl;
import org.springframework.stereotype.Component;

import java.io.PrintWriter;
import java.io.StringWriter;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

@Aspect
@Component
@Slf4j
@AllArgsConstructor
public class MailSendAdvice {

    private final MailService mailService;

    private final Environment environment;

    private EmailSendResult sendEmail(String subject,String content) throws Exception{

        SenderEntity senderEntity = mailService.getSenderConfig();

        String cSenderEmail = senderEntity.getC_sender_email();

        JavaMailSenderImpl mailSender = mailService.createMailSender(senderEntity);

        EmailSendResult result = new EmailSendResult(cSenderEmail, true, null);
        try {
            SimpleMailMessage message = new SimpleMailMessage();
            message.setFrom(senderEntity.getC_sender_email());
            message.setTo(senderEntity.getC_sender_email());
            message.setSubject(subject);
            message.setText(content);
            mailSender.send(message); // 메일 전송
        } catch (Exception e) {
            result.setSuccess(false);
            result.setErrorMessage(e.getMessage());
            log.error("메일 전송 실패: " + e.getMessage());
        }

        return result;
    }

    @AfterReturning("@annotation(mailSendAlarm)")
    public void afterReturning(JoinPoint joinPoint, MailSendAlarm mailSendAlarm) {
        try{
            if (isStageOrLiveProfile()) {

                Object[] args = joinPoint.getArgs();
                String prefixSubject = null;
                String prefixContent = null;

                for (Object arg : args) {
                    if(arg == null) continue;

                    for (Field field : getAllFields(arg.getClass())) {
                        field.setAccessible(true);

                        if(field.isAnnotationPresent(MailSubject.class)){
                            Object oSubject = field.get(arg);
                            prefixSubject = Optional.ofNullable(oSubject).map(String::valueOf).orElse(null);
                        }
                        if(field.isAnnotationPresent(MailContent.class)){
                            Object oContent = field.get(arg);
                            prefixContent = Optional.ofNullable(oContent).map(String::valueOf).orElse(null);
                        }
                    }
                    if(prefixSubject != null && prefixContent != null) break;
                }

                String subject = Optional.ofNullable(prefixSubject).map(a-> a+" "+mailSendAlarm.subject()).orElseGet(mailSendAlarm::subject);

                String content = Optional.ofNullable(prefixContent).map(a-> a+" "+mailSendAlarm.content()).orElseGet(mailSendAlarm::content);

                this.sendEmail(subject, content);

            }
        }catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private List<Field> getAllFields(Class<?> clazz) {
        List<Field> fields = new java.util.ArrayList<>();
        while (clazz != null) {
            fields.addAll(Arrays.asList(clazz.getDeclaredFields()));
            clazz = clazz.getSuperclass();
        }
        return fields;
    }

    @AfterThrowing(value = "@annotation(mailSendAlarm)",throwing = "e")
    public void afterThrowing(JoinPoint joinPoint, MailSendAlarm mailSendAlarm, Exception e) {
        try{
            if (isStageOrLiveProfile()) {

                String connectUrl = Optional.ofNullable(System.getenv("ARMS_DETAIL_CONNECT_URL")).orElse("");

                String profile = Optional.of(environment.getActiveProfiles()).
                        filter(a -> a.length > 0).map(a -> a[0]).orElse("");

                StringBuilder sb = new StringBuilder();

                if (!profile.isEmpty()) {
                    sb.append("[").append(profile).append("]");
                }

                if (!connectUrl.isEmpty()) {
                    sb.append("[").append(connectUrl).append("]");
                }

                sb.append(" ").append("에러발생");

                this.sendEmail( String.valueOf(sb) ,this.getStackTraceAsString(e));
            }
        }catch (Exception ex){
            log.error(ex.getMessage());
        }
    }

    private String getStackTraceAsString(Throwable e) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);
        e.printStackTrace(pw);
        return sw.toString();
    }

    private boolean isStageOrLiveProfile() {
        List<String> allowedProfiles = Arrays.asList("stg", "live");
        return Arrays.stream(environment.getActiveProfiles())
                .anyMatch(allowedProfiles::contains);
    }
}
