package com.arms.api.wiki.controller;

import com.arms.api.util.aspect.MailSendAlarm;
import com.arms.api.util.communicate.external.EngineService;
        import com.arms.api.wiki.model.WikiContentsDTO;
import com.arms.api.wiki.model.WikiDTO;
import com.arms.api.wiki.model.WikiEntity;
import com.arms.api.wiki.service.WikiService;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.controller.TreeAbstractController;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import com.arms.egovframework.javaservice.treeframework.model.TreeSearchEntity;
import com.arms.egovframework.javaservice.treeframework.util.ParameterParser;
import com.arms.egovframework.javaservice.treeframework.util.Util_TitleChecker;
import com.arms.egovframework.javaservice.treeframework.validation.group.*;
import com.google.common.collect.Maps;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Criterion;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.springframework.ui.ModelMap;
import org.springframework.util.StringUtils;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.List;
import java.util.Optional;

@Slf4j
@RestController
@RequiredArgsConstructor
@RequestMapping(value = {"/arms/wiki"})
public class WikiController extends TreeAbstractController<WikiService, WikiDTO, WikiEntity> {

    private final WikiService wikiService;

    private final EngineService engineService;

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/getNode.do", method = RequestMethod.GET)
    public ModelAndView getNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, WikiDTO wikiDTO, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);
        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        SessionUtil.setAttribute("getNode", changeReqTableName);

        WikiEntity wiki = wikiService.getNode(wikiEntity);

        SessionUtil.removeAttribute("getNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wiki);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{wikiId}/getWiki.do", method = RequestMethod.GET)
    public ModelAndView getWiki(@PathVariable(value = "wikiId") String wikiId, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getWiki");

        WikiContentsDTO wikiContentsDTO = Optional.ofNullable(engineService.getWiki(wikiId).getBody()).orElse(new CommonResponse.ApiResult<>()).getResponse();

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiContentsDTO);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{wikiId}/{version}/getWiki.do", method = RequestMethod.GET)
    public ModelAndView getWiki(@PathVariable(value = "wikiId") String wikiId, @PathVariable("version") Long version, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getWiki");

        WikiContentsDTO wikiContentsDTO = Optional.ofNullable(engineService.getWiki(wikiId, version).getBody()).orElse(new CommonResponse.ApiResult<>()).getResponse();

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiContentsDTO);
        return modelAndView;
    }

    @ResponseBody
    @GetMapping("/{wikiId}/getWikiVersions.do")
    public ModelAndView getWikiVersions(@PathVariable(value = "wikiId") String wikiId, @RequestParam("size") int size, @RequestParam("page") int page, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getWikiVersions");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", engineService.getWikiVersions(wikiId, size, page));
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/getChildNode.do", method = RequestMethod.GET)
    public ModelAndView getChildNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, WikiDTO wikiDTO, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getChildNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);
        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException("c_id is minus value");
        }

        SessionUtil.setAttribute("getChildNode", changeReqTableName);

        wikiEntity.setWhere("c_parentid", Long.valueOf(parser.get("c_id")));
        wikiEntity.setOrder(Order.asc("c_position"));
        List<WikiEntity> wikiList = wikiService.getChildNode(wikiEntity);

        SessionUtil.removeAttribute("getChildNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiList);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/getNodesWithoutRoot.do", method = RequestMethod.GET)
    public ModelAndView getNodesWithoutRoot(@PathVariable(value = "changeReqTableName") String changeReqTableName, WikiDTO wikiDTO) throws Exception {

        log.info("WikiController :: getNodesWithoutRoot");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        SessionUtil.setAttribute("getNodesWithoutRoot", changeReqTableName);

        wikiEntity.setOrder(Order.asc("c_position"));
        Criterion criterion = Restrictions.not(
                // replace "id" below with property name, depending on what you're filtering against
                Restrictions.in("c_id", TreeConstant.ROOT_CID, TreeConstant.First_Node_CID));
        wikiEntity.getCriterions().add(criterion);

        List<WikiEntity> wikiList = wikiService.getChildNode(wikiEntity);

        SessionUtil.removeAttribute("getNodesWithoutRoot");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        HashMap<String, Object> resultMap = Maps.newHashMap();
        resultMap.put("paginationInfo", wikiEntity.getPaginationInfo());
        resultMap.put("result", wikiList);
        modelAndView.addObject("result", resultMap);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/getPaginatedChildNode.do", method = RequestMethod.GET)
    public ModelAndView getPaginatedChildNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, WikiDTO wikiDTO, ModelMap model, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getPaginatedChildNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        SessionUtil.setAttribute("getPaginatedChildNode", changeReqTableName);

        if (wikiEntity.getC_id() <= 0 || wikiEntity.getPageIndex() <= 0 || wikiEntity.getPageUnit() <= 0 || wikiEntity.getPageSize() <= 0) {
            throw new RuntimeException();
        }
        wikiEntity.setWhere("c_parentid", wikiEntity.getC_id());
        List<WikiEntity> wikiList = wikiService.getPaginatedChildNode(wikiEntity);
        wikiEntity.getPaginationInfo().setTotalRecordCount(wikiList.size());

        SessionUtil.removeAttribute("getPaginatedChildNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        HashMap<String, Object> resultMap = Maps.newHashMap();
        resultMap.put("paginationInfo", wikiEntity.getPaginationInfo());
        resultMap.put("result", wikiList);
        modelAndView.addObject("result", resultMap);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/searchNode.do", method = RequestMethod.GET)
    public ModelAndView searchNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, WikiDTO wikiDTO, ModelMap model, HttpServletRequest request) throws Exception {

        log.info("WikiController :: searchNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);
        ParameterParser parser = new ParameterParser(request);

        if (!StringUtils.hasText(request.getParameter("searchString"))) {
            throw new RuntimeException("searchString is null");
        }

        SessionUtil.setAttribute("searchNode", changeReqTableName);

        wikiEntity.setWhereLike("c_title", parser.get("parser"));
        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiService.searchNode(wikiEntity));

        SessionUtil.removeAttribute("searchNode");

        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/addNode.do", method = RequestMethod.POST)
    @MailSendAlarm(subject = "문서 등록",content = "문서가 등록 되었습니다.")
    public ModelAndView addNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, @Validated(value = AddNode.class) WikiDTO wikiDTO, BindingResult bindingResult, ModelMap model) throws Exception {

        log.info("WikiController :: addNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        SessionUtil.setAttribute("addNode", changeReqTableName);

        wikiEntity.setC_title(Util_TitleChecker.StringReplace(wikiEntity.getC_title()));

        WikiEntity newWiki = wikiService.addNode(wikiEntity);
        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", newWiki);

        String wikiIdPrefix = changeReqTableName.replace("T_ARMS_", "");
        WikiContentsDTO wikiContentsDTO = WikiContentsDTO.builder()
                .wikiId(wikiIdPrefix + "_" + newWiki.getC_id())
                .pdServiceId(wikiIdPrefix.replaceAll("[^0-9]", ""))
                .build();
        engineService.createWiki(wikiContentsDTO);

        SessionUtil.removeAttribute("addNode");

        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/removeNode.do", method = RequestMethod.DELETE)
    public ModelAndView removeNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, @Validated(value = RemoveNode.class) WikiDTO wikiDTO, BindingResult bindingResult, ModelMap model) throws Exception {

        log.info("WikiController :: removeNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        SessionUtil.setAttribute("removeNode", changeReqTableName);

        wikiEntity.setStatus(wikiService.removeNode(wikiEntity));
        setJsonDefaultSetting(wikiEntity);

        SessionUtil.removeAttribute("removeNode");

        String wikiId = changeReqTableName.replace("T_ARMS_", "") + "_" + wikiDTO.getC_id();
        engineService.deleteWiki(wikiId);

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiEntity);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/updateNode.do", method = RequestMethod.PUT)
    public ModelAndView updateNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, @Validated(value = UpdateNode.class) WikiDTO wikiDTO, BindingResult bindingResult, HttpServletRequest request, ModelMap model) throws Exception {

        log.info("WikiController :: updateNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        SessionUtil.setAttribute("updateNode", changeReqTableName);

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiService.updateNode(wikiEntity));

        SessionUtil.removeAttribute("updateNode");

        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/updateWiki.do", method = RequestMethod.PUT)
    @MailSendAlarm(subject = "문서 수정",content = "문서가 수정 되었습니다.")
    public ModelAndView updateWiki(@RequestBody WikiContentsDTO wikiContentsDTO, HttpServletRequest request) throws Exception {

        log.info("WikiController :: updateWiki");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", engineService.updateWiki(wikiContentsDTO));

        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/changeRecent.do", method = RequestMethod.PUT)
    public ModelAndView changeRecent(@RequestBody WikiContentsDTO wikiContentsDTO, HttpServletRequest request) throws Exception {

        log.info("WikiController :: changeRecent");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", engineService.changeRecent(wikiContentsDTO));

        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/alterNode.do", method = RequestMethod.PUT)
    public ModelAndView alterNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, @Validated(value = AlterNode.class) WikiDTO wikiDTO, BindingResult bindingResult, ModelMap model) throws Exception {

        log.info("WikiController :: alterNode");

        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        SessionUtil.setAttribute("alterNode", changeReqTableName);

        wikiEntity.setC_title(Util_TitleChecker.StringReplace(wikiEntity.getC_title()));

        wikiEntity.setStatus(wikiService.alterNode(wikiEntity));
        setJsonDefaultSetting(wikiEntity);

        SessionUtil.removeAttribute("alterNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiEntity);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/alterNodeType.do", method = RequestMethod.PUT)
    public ModelAndView alterNodeType(@PathVariable(value = "changeReqTableName") String changeReqTableName, @Validated(value = AlterNodeType.class) WikiDTO wikiDTO, BindingResult bindingResult, ModelMap model) throws Exception {

        log.info("WikiController :: alterNodeType");

        SessionUtil.setAttribute("alterNodeType", changeReqTableName);
        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        wikiService.alterNodeType(wikiEntity);
        setJsonDefaultSetting(wikiEntity);

        SessionUtil.removeAttribute("alterNodeType");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiEntity);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/moveNode.do", method = RequestMethod.POST)
    public ModelAndView moveNode(@PathVariable(value = "changeReqTableName") String changeReqTableName, @Validated(value = MoveNode.class) WikiDTO wikiDTO, BindingResult bindingResult, ModelMap model, HttpServletRequest request) throws Exception {

        log.info("WikiController :: moveNode");

        SessionUtil.setAttribute("moveNode", changeReqTableName);
        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        wikiService.moveNode(wikiEntity, request);
        setJsonDefaultSetting(wikiEntity);

        SessionUtil.removeAttribute("moveNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiEntity);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/{changeReqTableName}/getMonitor.do", method = RequestMethod.GET)
    public ModelAndView getMonitor(@PathVariable(value = "changeReqTableName") String changeReqTableName, WikiDTO wikiDTO, ModelMap model, HttpServletRequest request) throws Exception {

        log.info("WikiController :: getMonitor");

        SessionUtil.setAttribute("getMonitor", changeReqTableName);
        WikiEntity wikiEntity = modelMapper.map(wikiDTO, WikiEntity.class);

        wikiEntity.setOrder(Order.asc("c_position"));
        List<TreeSearchEntity> wikiList = wikiService.getChildNode(wikiEntity);

        SessionUtil.removeAttribute("getMonitor");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiList);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/imageUpload.do", method = RequestMethod.POST)
    public ModelAndView imageUpload(MultipartHttpServletRequest multiRequest) {
        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", wikiService.imageUpload(multiRequest));
        return modelAndView;
    }

}
