package com.arms.config;


import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.AdminClientConfig;
import org.apache.kafka.clients.consumer.ConsumerConfig;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.common.serialization.StringDeserializer;
import org.apache.kafka.common.serialization.StringSerializer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.kafka.annotation.EnableKafka;
import org.springframework.kafka.config.ConcurrentKafkaListenerContainerFactory;
import org.springframework.kafka.core.ConsumerFactory;
import org.springframework.kafka.core.DefaultKafkaConsumerFactory;
import org.springframework.kafka.core.DefaultKafkaProducerFactory;
import org.springframework.kafka.core.KafkaAdmin;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.core.ProducerFactory;
import org.springframework.kafka.listener.ContainerProperties;
import org.springframework.kafka.listener.SeekToCurrentErrorHandler;
import org.springframework.util.backoff.FixedBackOff;

import java.util.HashMap;
import java.util.Map;

/**
 * Kafka 설정 클래스
 *
 * REQADD 토픽에 대한 순차 처리 보장 설정:
 * 1. 컨슈머 동시성(concurrency) = 1 → 단일 스레드로 메시지 순차 처리
 * 2. MAX_POLL_RECORDS = 1 → 한 번에 1개 메시지만 폴링
 * 3. 수동 커밋 모드 → 메시지 처리 완료 후 명시적 커밋
 *
 * @author 313DEVGRP
 * @since 2023
 */
/*
토픽별 독립적인 처리
현재 설정에서 서로 다른 토픽 2개가 있으면 각 토픽당 컨슈머 1개씩 총 2개의 메시지를 동시에 처리할 수 있습니다.

예시 시나리오
Topic A (REQADD)          Topic B (WIKI)
    ↓                          ↓
Consumer A (Thread 1)      Consumer B (Thread 2)
    ↓                          ↓
메시지 1개 처리 중          메시지 1개 처리 중

동작 방식
// Consumer A - REQADD 토픽
@KafkaListener(
    topics = "REQADD",
    containerFactory = "reqAddKafkaListenerContainerFactory"
)
public void consumeReqAdd(...) {
    // REQADD 토픽에서 1개 메시지 처리
}

// Consumer B - REQUPDATE 토픽
@KafkaListener(
    topics = "REQUPDATE",
    containerFactory = "reqUpdateKafkaListenerContainerFactory"
)
public void consumeReqUpdate(...) {
    // REQUPDATE 토픽에서 1개 메시지 처리
}
```

### 핵심 포인트

✅ **토픽별로 독립적**: 각 토픽은 별도의 컨슈머 스레드를 가짐
- REQADD 토픽: 1개 메시지 처리 중
- REQUPDATE 토픽: 동시에 1개 메시지 처리 중

✅ **토픽 내에서는 순차적**: 같은 토픽 내에서는 1개씩만 처리
- REQADD의 메시지1 처리 중 → REQADD의 메시지2는 대기
- REQUPDATE의 메시지1 처리 중 → REQUPDATE의 메시지2는 대기

### 타임라인 예시
```
시간 →

토픽A: [메시지1 처리중...] → [메시지2 처리중...] → [메시지3 처리중...]
토픽B: [메시지1 처리중...] → [메시지2 처리중...] → [메시지3 처리중...]


결론:

✅ 토픽 2개 = 최대 2개 메시지 동시 처리 가능
✅ 토픽 3개 = 최대 3개 메시지 동시 처리 가능
✅ 각 토픽 내에서는 순차 처리 보장

토픽별로 독립적인 컨슈머가 동작하므로 정확히 이해하신 게 맞습니다! 👍

 */
@Configuration
@RefreshScope
@EnableKafka
public class KafkaConfig {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Value("${spring.kafka.bootstrap-servers:Kafka00Service:9094,Kafka01Service:9094,Kafka02Service:9094}")
    private String bootstrapServers;

    /**
     * Kafka 컨슈머 그룹 ID
     *
     * 그룹이 필요한 이유:
     * 1. 오프셋 관리: 어디까지 읽었는지 Kafka가 추적
     * 2. 재시작 시 이어서 처리: 마지막 처리 위치부터 재개
     * 3. 메시지 손실 방지: 미처리 메시지 자동 처리
     *
     * 환경별 독립적인 그룹 ID 사용 권장
     */
    @Value("${spring.kafka.consumer.group-id:requirement-consumer-group}")
    private String groupId;

    @Bean
    public KafkaAdmin kafkaAdmin() {
        Map<String, Object> configs = new HashMap<>();
        configs.put(AdminClientConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);
        return new KafkaAdmin(configs);
    }

    @Bean
    public AdminClient kafkaAdminClient(KafkaAdmin kafkaAdmin) {
        return AdminClient.create(kafkaAdmin.getConfigurationProperties());
    }

    // ========================================
    // Producer 설정
    // ========================================

    @Bean
    public ProducerFactory<String, String> producerFactory() {
        Map<String, Object> configs = new HashMap<>();
        configs.put(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);
        configs.put(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG, StringSerializer.class);
        configs.put(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG, StringSerializer.class);
        return new DefaultKafkaProducerFactory<>(configs);
    }

    @Bean
    public KafkaTemplate<String, String> reqAddKafkaTemplate() {
        return new KafkaTemplate<>(producerFactory());
    }

    // ========================================
    // Consumer 설정 - REQADD 토픽 순차 처리
    // ========================================

    /**
     * REQADD 토픽 전용 Consumer Factory
     *
     * 순차 처리 보장을 위한 설정:
     * - MAX_POLL_RECORDS = 1: 한 번에 1개 메시지만 가져옴
     * - ENABLE_AUTO_COMMIT = false: 수동 커밋으로 처리 완료 후 커밋
     * - AUTO_OFFSET_RESET = earliest: 컨슈머 그룹 최초 실행 시 처음부터 읽음
     */
    @Bean
    public ConsumerFactory<String, String> consumerFactory() {
        Map<String, Object> configs = new HashMap<>();
        configs.put(ConsumerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);
        configs.put(ConsumerConfig.GROUP_ID_CONFIG, groupId);
        configs.put(ConsumerConfig.KEY_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);
        configs.put(ConsumerConfig.VALUE_DESERIALIZER_CLASS_CONFIG, StringDeserializer.class);

        // 오프셋 관리
        configs.put(ConsumerConfig.AUTO_OFFSET_RESET_CONFIG, "earliest");
        configs.put(ConsumerConfig.ENABLE_AUTO_COMMIT_CONFIG, false); // 수동 커밋 명시

        // 순차 처리를 위한 핵심 설정
        configs.put(ConsumerConfig.MAX_POLL_RECORDS_CONFIG, 1); // 한 번에 1개 메시지만 폴링

        // 타임아웃 설정
        configs.put(ConsumerConfig.SESSION_TIMEOUT_MS_CONFIG, 300000); // 세션 타임아웃 5분
        configs.put(ConsumerConfig.HEARTBEAT_INTERVAL_MS_CONFIG, 10000); // 하트비트 간격 10초
        configs.put(ConsumerConfig.MAX_POLL_INTERVAL_MS_CONFIG, 300000); // 최대 폴 간격 5분

        return new DefaultKafkaConsumerFactory<>(configs);
    }

    /**
     * REQADD 토픽 전용 Listener Container Factory
     *
     * 순차 처리 보장을 위한 설정:
     * - concurrency = 1: 단일 컨슈머 스레드로 순차 처리
     * - AckMode.MANUAL: 수동 커밋으로 메시지 처리 완료를 명시적으로 제어
     * - SeekToCurrentErrorHandler: 에러 발생 시 재시도 및 로깅
     */
    @Bean
    public ConcurrentKafkaListenerContainerFactory<String, String> reqAddKafkaListenerContainerFactory() {
        logger.info("========================================");
        logger.info("Creating reqAddKafkaListenerContainerFactory Bean");
        logger.info("Bootstrap Servers: {}", bootstrapServers);
        logger.info("Consumer Group ID: {}", groupId);
        logger.info("========================================");

        ConcurrentKafkaListenerContainerFactory<String, String> factory = new ConcurrentKafkaListenerContainerFactory<>();
        factory.setConsumerFactory(consumerFactory());

        // 순차 처리를 위한 핵심 설정: 컨슈머 스레드 1개로 제한
        factory.setConcurrency(1);

        // 수동 커밋 모드: 메시지 처리 완료 후 ack.acknowledge() 호출 필요
        factory.getContainerProperties().setAckMode(ContainerProperties.AckMode.MANUAL);

        // 에러 핸들러 설정: 1초 간격으로 3회 재시도
        SeekToCurrentErrorHandler errorHandler = new SeekToCurrentErrorHandler(
                (record, exception) -> {
                    // 모든 재시도 실패 후 실행되는 로직
                    logger.error("=== Kafka Message Processing Failed ===");
                    logger.error("Topic: {}", record.topic());
                    logger.error("Key: {}", record.key());
                    logger.error("Value: {}", record.value());
                    logger.error("Partition: {}, Offset: {}", record.partition(), record.offset());
                    logger.error("Exception: {}", exception.getMessage(), exception);

                    // TODO: 필요 시 Slack 알림 또는 Dead Letter Queue 전송 구현
                    // 예시: slackService.sendAlert("Kafka Error", ...);
                    // 예시: deadLetterQueueService.send(record);
                },
                new FixedBackOff(1000L, 3L)  // 1초 간격, 3회 재시도
        );

        // 재시도하지 않을 예외 타입 지정 (선택사항)
        // errorHandler.addNotRetryableExceptions(
        //     IllegalArgumentException.class,
        //     NullPointerException.class
        // );

        factory.setErrorHandler(errorHandler);

        logger.info("reqAddKafkaListenerContainerFactory Bean created successfully");

        return factory;
    }
}
