package com.arms.config;

import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.ListConsumerGroupOffsetsResult;
import org.apache.kafka.clients.admin.ListOffsetsResult;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.common.TopicPartition;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutionException;

/**
 * Kafka Consumer Lag Monitor
 * 
 * 토픽의 남은 메시지 개수(Consumer Lag)를 실시간으로 조회
 * 
 * 주요 기능:
 * 1. 토픽별 전체 Lag 조회
 * 2. 파티션별 상세 Lag 조회
 * 3. Consumer Group 오프셋 모니터링
 * 
 * @author 313DEVGRP
 * @since 2025
 */
@Component
public class KafkaLagMonitor {

    private final Logger logger = LoggerFactory.getLogger(this.getClass());

    @Autowired(required = false)
    private AdminClient kafkaAdminClient;

    /**
     * KafkaConfig에서 정의된 groupId 사용
     */
    @Value("${spring.kafka.consumer.group-id:requirement-consumer-group}")
    private String groupId;

    /**
     * 특정 토픽의 Consumer Lag 조회
     * 
     * Lag = End Offset (최신 오프셋) - Current Offset (현재 처리한 오프셋)
     * 
     * @param topic 토픽 이름
     * @return 남은 메시지 개수 (모든 파티션 합계), 실패 시 -1
     */
    public long getConsumerLag(String topic) {
        if (kafkaAdminClient == null) {
            logger.warn("AdminClient is not available. Cannot get consumer lag.");
            return -1;
        }

        try {
            logger.debug("Getting consumer lag for topic: {} with group: {}", topic, groupId);
            
            // 1. Consumer Group의 현재 오프셋 조회
            Map<TopicPartition, Long> currentOffsets = getCurrentOffsets(topic);
            
            if (currentOffsets.isEmpty()) {
                logger.warn("No consumer offsets found for topic: {} in group: {}", topic, groupId);
                return 0;
            }
            
            // 2. 토픽의 최신 오프셋 조회 (End Offset)
            Map<TopicPartition, Long> endOffsets = getEndOffsets(topic);
            
            // 3. Lag 계산 (End Offset - Current Offset)
            long totalLag = 0;
            for (Map.Entry<TopicPartition, Long> entry : endOffsets.entrySet()) {
                TopicPartition tp = entry.getKey();
                long endOffset = entry.getValue();
                long currentOffset = currentOffsets.getOrDefault(tp, 0L);
                
                long lag = endOffset - currentOffset;
                totalLag += lag;
                
                logger.debug("[{}] Partition {}: Current={}, End={}, Lag={}", 
                            topic, tp.partition(), currentOffset, endOffset, lag);
            }
            
            logger.info("Topic [{}] Total Lag: {} messages", topic, totalLag);
            return totalLag;
            
        } catch (Exception e) {
            logger.error("Error getting consumer lag for topic: {}", topic, e);
            return -1;
        }
    }

    /**
     * Consumer Group의 현재 오프셋 조회
     * 
     * @param topic 토픽 이름
     * @return TopicPartition별 현재 오프셋
     */
    private Map<TopicPartition, Long> getCurrentOffsets(String topic) 
            throws ExecutionException, InterruptedException {
        
        ListConsumerGroupOffsetsResult result = 
            kafkaAdminClient.listConsumerGroupOffsets(groupId);
        
        Map<TopicPartition, OffsetAndMetadata> offsets = result.partitionsToOffsetAndMetadata().get();
        
        Map<TopicPartition, Long> currentOffsets = new HashMap<>();
        for (Map.Entry<TopicPartition, OffsetAndMetadata> entry : offsets.entrySet()) {
            if (entry.getKey().topic().equals(topic)) {
                currentOffsets.put(entry.getKey(), entry.getValue().offset());
            }
        }
        
        return currentOffsets;
    }

    /**
     * 토픽의 최신 오프셋 조회 (End Offset)
     * 
     * @param topic 토픽 이름
     * @return TopicPartition별 최신 오프셋
     */
    private Map<TopicPartition, Long> getEndOffsets(String topic) 
            throws ExecutionException, InterruptedException {
        
        // 토픽의 모든 파티션 정보 조회
        var topicDescription = kafkaAdminClient.describeTopics(Collections.singleton(topic))
                                                .all().get();
        
        Map<TopicPartition, Long> endOffsets = new HashMap<>();
        
        for (var partitionInfo : topicDescription.get(topic).partitions()) {
            TopicPartition tp = new TopicPartition(topic, partitionInfo.partition());
            
            // 최신 오프셋 조회
            Map<TopicPartition, org.apache.kafka.clients.admin.OffsetSpec> requestMap = 
                Collections.singletonMap(tp, org.apache.kafka.clients.admin.OffsetSpec.latest());
            
            ListOffsetsResult listOffsetsResult = kafkaAdminClient.listOffsets(requestMap);
            long endOffset = listOffsetsResult.all().get().get(tp).offset();
            
            endOffsets.put(tp, endOffset);
        }
        
        return endOffsets;
    }

    /**
     * 파티션별 상세 Lag 정보 조회
     * 
     * @param topic 토픽 이름
     * @return 파티션 번호별 Lag 맵
     */
    public Map<Integer, Long> getDetailedLag(String topic) {
        Map<Integer, Long> partitionLags = new HashMap<>();
        
        if (kafkaAdminClient == null) {
            logger.warn("AdminClient is not available");
            return partitionLags;
        }
        
        try {
            Map<TopicPartition, Long> currentOffsets = getCurrentOffsets(topic);
            Map<TopicPartition, Long> endOffsets = getEndOffsets(topic);
            
            for (Map.Entry<TopicPartition, Long> entry : endOffsets.entrySet()) {
                TopicPartition tp = entry.getKey();
                long endOffset = entry.getValue();
                long currentOffset = currentOffsets.getOrDefault(tp, 0L);
                long lag = endOffset - currentOffset;
                
                partitionLags.put(tp.partition(), lag);
                
                logger.debug("Partition {} - Lag: {}", tp.partition(), lag);
            }
            
        } catch (Exception e) {
            logger.error("Error getting detailed lag for topic: {}", topic, e);
        }
        
        return partitionLags;
    }

    /**
     * Consumer Group의 전체 상태 정보 조회
     * 
     * @param topic 토픽 이름
     * @return 상태 정보 맵
     */
    public Map<String, Object> getConsumerStatus(String topic) {
        Map<String, Object> status = new HashMap<>();
        
        try {
            long totalLag = getConsumerLag(topic);
            Map<Integer, Long> detailedLag = getDetailedLag(topic);
            
            status.put("topic", topic);
            status.put("groupId", groupId);
            status.put("totalLag", totalLag);
            status.put("partitionCount", detailedLag.size());
            status.put("partitionLags", detailedLag);
            status.put("timestamp", System.currentTimeMillis());
            
            // Lag 상태 판단
            String lagStatus;
            if (totalLag == 0) {
                lagStatus = "NO_LAG";
            } else if (totalLag < 100) {
                lagStatus = "LOW";
            } else if (totalLag < 1000) {
                lagStatus = "MEDIUM";
            } else {
                lagStatus = "HIGH";
            }
            status.put("lagStatus", lagStatus);
            
        } catch (Exception e) {
            logger.error("Error getting consumer status for topic: {}", topic, e);
            status.put("error", e.getMessage());
        }
        
        return status;
    }
}
