const PAGE_SIZE = 12; //페이지 개수

function loadFunctionCSS(hrefList){
	hrefList.forEach(function (href) {
		var cssLink = $("<link>").attr({
			type: "text/css",
			rel: "stylesheet",
			href: href
		});
		$("head").append(cssLink);
	});
}

function execDocReady() {
	var pluginGroups = [
		["../reference/light-blue/lib/vendor/jquery.ui.widget.js", "../reference/lightblue4/docs/lib/widgster/widgster.js"],
		["../reference/lightblue4/docs/lib/bootstrap-select/dist/js/bootstrap-select.min.js"]
	];

	loadPluginGroupsParallelAndSequential(pluginGroups)
		.then(function () {
			loadFunctionCSS([
				"/cover/css/function/layout.css",
				"/cover/css/experience/experience-common.css",
				"/cover/css/experience/experience-list.css"
			]);

			$(".widget").widgster();

			$("#sidebar").hide();
			$(".wrap").css("margin-left", 0);
			$("#footer").load("/cover/html/template/landing-footer.html");
		})
		.catch(function (error) {
			console.error("플러그인 로드 중 오류 발생");
			console.error(error);
		});
}

/* ---------- 공통 유틸 ---------- */
function escapeHTML(s) {
	return String(s || "").replace(/[&<>"']/g, m => ({
		"&": "&amp;", "<": "&lt;", ">": "&gt;", "\"": "&quot;", "'": "&#39;"
	}[m]));
}

// 등록 화면 이동
function buildEditorHref() {
	const url = new URL(location.href);
	if (url.searchParams.get('page')) {
		url.searchParams.set('page', 'experienceEditor');
		url.searchParams.delete('id');
		return url.toString();
	}
	return 'experienceEditor.html';
}

$(document).on('click', '#btn-new-experience', function (e) {
	e.preventDefault();
	sessionStorage.removeItem('experienceEditDraft');
	window.location.href = buildEditorHref();
});

// 상세 화면 이동 링크 생성
function buildDetailHref(id) {
	const url = new URL(location.href);
	if (url.searchParams.get('page')) {
		url.searchParams.set('page', 'experienceDetail');
		url.searchParams.set('id', String(id));
		return url.toString();
	}
	return `experienceDetail.html?id=${id}`;
}

/* ---------- 백엔드에서 목록 가져오기 ---------- */
function fetchExperienceList(params = {}) {
	const defaultParams = {
		pageIndex: 1,
		pageUnit: PAGE_SIZE,
		...params
	};

	return $.ajax({
		url: '/auth-anon/api/cover/experience/getExperienceList',
		type: 'GET',
		data: defaultParams,
	}).then(function(res){
		console.log("응답:", res);

		if (res && res.success && res.response) {
			const response = res.response;
			return {
				list: response.data || [],
				pagination: {
					currentPage: response.paginationInfo?.currentPageNo || 1,
					totalPages: response.paginationInfo?.totalPageCount || 0,
					totalCount: response.paginationInfo?.totalRecordCount || 0
				}
			};
		}

		return {
			list: [],
			pagination: { currentPage: 1, totalPages: 0, totalCount: 0 }
		};
	});
}

/* ---------- 목록 렌더 ---------- */
function renderList($container, items) {
	if (!items || !items.length) {
		$container.html('<div class="col-12 text-center text-muted py-5">게시물이 없습니다.</div>');
		return;
	}

	const html = items.map(it => {
		const id    = Number(it.c_id);
		const href  = buildDetailHref(id);
		const title = it.c_clientcase_title; //타이틀
		const cat   = it.c_clientcase_category; //카테고리
		const thumb = it.c_clientcase_thumbnail_image || '/cover/img/logo.png'; //썸네일


		const rawDateObject = it.c_clientcase_created; // 생성일자
		let date = '';

		//생성일자
		if (rawDateObject && typeof rawDateObject === 'object' && rawDateObject.year) {
			// 각 필드를 추출하고 2자리로 패딩 년,월,일 형태
			const year  = rawDateObject.year;
			const month = String(rawDateObject.monthValue).padStart(2, '0');
			const day   = String(rawDateObject.dayOfMonth).padStart(2, '0');

			date = `${year}-${month}-${day}`;
		}
		const sub   = it.c_clientcase_subtitle || '';

		return `
      <div class="col-lg-4 col-md-6 mb-5">
        <a href="${href}" class="card-link" data-id="${id}">
          <div class="card customer-card shadow-sm util-transition">
            <div class="img-wrapper">
              <img src="${escapeHTML(thumb)}" alt="${escapeHTML(title || '')}" class="experience-thumb" />
            </div>
            <div class="card-body">
              <small>
                <span class="text-small-title text-link-only" data-id="${id}">
                  ${escapeHTML(cat || '')}
                </span>
                <span class="text-muted"> | ${escapeHTML(date || '')}</span>
              </small>
              <h5 class="card-title">${escapeHTML(title || '')}</h5>
              <p class="card-text text-muted">${escapeHTML(sub)}</p>
            </div>
            <div class="card-overlay"></div>
          </div>
        </a>
      </div>
    `;
	}).join('');

	$container.html(html);

	// 이미지 로드 후 비율에 따라 object-fit 조정
	adjustImageObjectFit($container);
}

/* ---------- 이미지 비율에 따라 object-fit 조정 ---------- */
function adjustImageObjectFit($container) {
	$container.find('.experience-thumb').each(function() {
		const img = this;

		// 이미지가 이미 로드된 경우
		if (img.complete && img.naturalWidth > 0) {
			applyObjectFit(img);
		} else {
			// 이미지 로드 대기
			$(img).on('load', function() {
				applyObjectFit(this);
			});
		}
	});
}

function applyObjectFit(img) {
	const ratio = img.naturalWidth / img.naturalHeight;

	// 16:10 비율 = 1.6, 허용 범위 1.4 ~ 1.8 (16:9 ≈ 1.78 포함)
	if (ratio >= 1.4 && ratio <= 1.8) {
		img.style.objectFit = 'cover';
	} else {
		img.style.objectFit = 'contain';
	}
}

// 텍스트만 눌러도 상세 이동
$(document).on('click', '.text-link-only', function (e) {
	e.preventDefault();
	e.stopPropagation();
	const $a = $(this).closest('a.card-link');
	const id = $a.data('id') || $(this).data('id');
	if (!id) return;
	window.location.href = buildDetailHref(id);
});

/* ---------- 페이지네이션 렌더링 ---------- */
function renderPagination($container, pagination, currentFilters) {
	const { currentPage, totalPages, totalCount } = pagination;

	if (totalPages <= 1) {
		$container.html('');
		return;
	}

	let html = '<nav aria-label="Page navigation"><ul class="pagination justify-content-center">';

	// 이전 버튼
	if (currentPage > 1) {
		html += `<li class="page-item">
            <a class="page-link" href="#" data-page="${currentPage - 1}">이전</a>
        </li>`;
	} else {
		html += `<li class="page-item disabled">
            <span class="page-link">이전</span>
        </li>`;
	}

	// 페이지 번호
	const startPage = Math.max(1, currentPage - 2);
	const endPage = Math.min(totalPages, currentPage + 2);

	if (startPage > 1) {
		html += `<li class="page-item">
            <a class="page-link" href="#" data-page="1">1</a>
        </li>`;
		if (startPage > 2) {
			html += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
		}
	}

	for (let i = startPage; i <= endPage; i++) {
		if (i === currentPage) {
			html += `<li class="page-item active">
                <span class="page-link">${i}</span>
            </li>`;
		} else {
			html += `<li class="page-item">
                <a class="page-link" href="#" data-page="${i}">${i}</a>
            </li>`;
		}
	}

	if (endPage < totalPages) {
		if (endPage < totalPages - 1) {
			html += `<li class="page-item disabled"><span class="page-link">...</span></li>`;
		}
		html += `<li class="page-item">
            <a class="page-link" href="#" data-page="${totalPages}">${totalPages}</a>
        </li>`;
	}

	// 다음 버튼
	if (currentPage < totalPages) {
		html += `<li class="page-item">
            <a class="page-link" href="#" data-page="${currentPage + 1}">다음</a>
        </li>`;
	} else {
		html += `<li class="page-item disabled">
            <span class="page-link">다음</span>
        </li>`;
	}

	html += '</ul></nav>';

	// 총 개수 표시
	html += `<div class="text-center text-muted mt-2">
        총 ${totalCount}개의 게시물 (${currentPage} / ${totalPages} 페이지)
    </div>`;

	$container.html(html);

	// 페이지 클릭 이벤트
	$container.off('click', 'a.page-link').on('click', 'a.page-link', function(e) {
		e.preventDefault();
		const page = $(this).data('page');
		if (page) {
			loadPage(page, currentFilters);
		}
	});
}

/* ---------- 페이지 로드 함수 ---------- */
function loadPage(pageIndex, filters = {}) {
	const $listContainer = $('#experience-list');
	let $paginationContainer = $('#experience-pagination');

	// 페이지네이션 컨테이너가 없으면 동적 생성
	if (!$paginationContainer.length) {
		$paginationContainer = $('<div id="experience-pagination"></div>');
		$listContainer.after($paginationContainer);
	}

	const params = {
		pageIndex: pageIndex,
		pageUnit: PAGE_SIZE,
		...filters
	};

	fetchExperienceList(params)
		.then(data => {
			renderList($listContainer, data.list);
			renderPagination($paginationContainer, data.pagination, filters);

			// 페이지 상단으로 스크롤
			$('html, body').animate({ scrollTop: 0 }, 300);
		})
		.catch(err => {
			console.error(err);
			$listContainer.html('<div class="col-12 text-center text-danger py-5">목록을 불러오지 못했습니다.</div>');
		});
}

/* ---------- 권한 체크 ---------- */
function experienceAuthAdminCheck() {
	$.ajax({
		url: "/auth-user/me",
		type: "GET",
		timeout: 7313,
		global: false,
		statusCode: {
			200: function (json) {
				console.log("[ experience :: authUserCheck ] userName = " + json.preferred_username);
				console.log("[ experience :: authUserCheck ] roles = " + json.realm_access.roles);

				const hasAdminPermission = json.realm_access.roles.includes("ROLE_ADMIN");

				if (hasAdminPermission) {
					showAdminButtons();
				} else {
					hideAdminButtons();
				}
			},
			401: function () {
				// 비로그인 사용자
				console.log("비로그인 사용자 - 등록 버튼 숨김");
				hideAdminButtons();
			}
		}
	});
}

function showAdminButtons() {
	$("#btn-new-experience").show();
}

function hideAdminButtons() {
	$("#btn-new-experience").hide();
}

/* ---------- 자동 마운트 ---------- */
$(function () {
	const $listContainer = $('#experience-list');

	if (!$listContainer.length) return;

	// 권한 체크
	experienceAuthAdminCheck();

	// 초기 필터 값
	const initialFilters = {
		category: $('#filter-category').val() || undefined,
		visibility: $('#filter-visibility').val() || undefined
	};

	// 첫 페이지 로드
	loadPage(1, initialFilters);

	// 필터 변경 시 첫 페이지부터 다시 로드
	$(document).on('change', '#filter-category, #filter-visibility', function(){
		const filters = {
			category: $('#filter-category').val() || undefined,
			visibility: $('#filter-visibility').val() || undefined
		};
		loadPage(1, filters);
	});
});