package com.arms.api.keycloak.admin.controller;

import com.arms.api.keycloak.admin.model.CreateUserVO;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.keycloak.admin.client.Keycloak;
import org.keycloak.admin.client.resource.UsersResource;
import org.keycloak.admin.client.token.TokenManager;
import org.keycloak.representations.idm.*;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import javax.ws.rs.core.Response;
import java.util.*;
import java.util.stream.Collectors;

@RestController
@AllArgsConstructor
@Slf4j
public class KeycloakAdminController {

    private final Keycloak keycloak;

    @GetMapping("/auth-admin/token")
    @ResponseBody
    public Mono<TokenManager> getAuthToken(ServerWebExchange exchange) {
        return Mono.just(keycloak.tokenManager());
    }

    @GetMapping("/auth-admin/realms/")
    @ResponseBody
    public Mono<List<RealmRepresentation>> getRealms(
            ServerWebExchange exchange
    ) {
        return Mono.just(keycloak.realms().findAll());
    }

    @GetMapping("/auth-admin/realms/{realm}")
    @ResponseBody
    public Mono<RealmRepresentation> getRealms(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {
        return Mono.just(
                keycloak.realm(realm).toRepresentation()
        );
    }

    @PutMapping("/auth-admin/realms/{realm}")
    @ResponseBody
    public Mono<Void> updateRealm(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @RequestBody RealmRepresentation realmRepresentation
    ) {
        keycloak.realm(realm).update(realmRepresentation);
        return Mono.empty();
    }

    @PostMapping("/auth-admin/realms/{realm}")
    @ResponseBody
    public Mono<Void> testSMTPConnection(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @RequestBody RealmRepresentation realmRepresentation
    ) {
        keycloak.realm(realm).testSMTPConnection(realmRepresentation.getSmtpServer());
        return Mono.empty();
    }

    @PostMapping("/auth-admin/realms/smtp/{realm}")
    @ResponseBody
    public Mono<Void> configureSmtp(
            ServerWebExchange exchange,
            @PathVariable("realm") String realmName
    ) {
        try {
            RealmRepresentation realm = keycloak.realm(realmName).toRepresentation();

            Map<String, String> smtpConfig = new HashMap<>();
            smtpConfig.put("host", "smtp.smtp.net");
            smtpConfig.put("port", "465");
            smtpConfig.put("from", "mail@a-rms.net");
            smtpConfig.put("auth", "true");
            smtpConfig.put("ssl", "true");
            smtpConfig.put("user", "mail@a-rms.net");
            smtpConfig.put("password", ""); // 앱 비밀번호 사용 권장
            smtpConfig.put("starttls", "true");
            smtpConfig.put("replyToDisplayName", "[ A-RMS ]");
            smtpConfig.put("replyTo", "mail@a-rms.net");
            smtpConfig.put("envelopeFrom", "mail@a-rms.net");
            smtpConfig.put("fromDisplayName", "[ A-RMS ]"); // **필수 추가**

            realm.setSmtpServer(smtpConfig);

            ObjectMapper mapper = new ObjectMapper();
            mapper.enable(SerializationFeature.INDENT_OUTPUT); // 보기 좋게 출력 (pretty print)

            String realmJson = mapper.writeValueAsString(realm);
            log.info("Realm 전체 설정 정보:\n{}", realmJson);

            smtpConfig.forEach((key, value) ->
                    log.info("smtpServer 설정 - {} = {}", key, value));

            keycloak.realm(realmName).update(realm);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }


        return Mono.empty();
    }

    @GetMapping("/auth-admin/realms/{realm}/clients")
    @ResponseBody
    public Mono<List<ClientRepresentation>> getClients(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {
        return Mono.just(
                keycloak.realm(realm)
                        .clients()
                        .findAll()
        );
    }

    @GetMapping("/auth-admin/realms/{realm}/client-id/{clientId}")
    @ResponseBody
    public Mono<List<ClientRepresentation>> getClient(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @PathVariable("clientId") String clientId
    ) {
        return Mono.just(
                keycloak.realm(realm)
                        .clients()
                        .findByClientId(clientId)
        );
    }

    @GetMapping("/auth-admin/realms/{realm}/groups")
    @ResponseBody
    public Mono<List<GroupRepresentation>> getGroups(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {

        return Mono.just(
                keycloak.realm(realm).groups().groups()
        );
    }

    @GetMapping("/auth-admin/realms/{realm}/groups/{group-id}")
    @ResponseBody
    public Mono<GroupRepresentation> getGroup(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId // group-id
    ) {

        return Mono.just(
                keycloak.realm(realm)
                        .groups()
                        .group(groupId).toRepresentation()
        );
    }

    @PutMapping("/auth-admin/realms/{realm}/groups/{group-id}")
    @ResponseBody
    public void editGroup(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId, // group-id
            @RequestBody GroupRepresentation groupRepresentation
    ) {
        keycloak.realm(realm).groups().group(groupId).update(groupRepresentation);
    }

    // delete Group including subgroup
    @DeleteMapping("/auth-admin/realms/{realm}/groups/{group-id}")
    public void deleteGroup(ServerWebExchange exchange,
                            @PathVariable("realm") String realm,
                            @PathVariable("group-id") String groupId
    ) {
        keycloak.realm(realm).groups().group(groupId).remove();
    }

    // create or add a top level realm groupSet or create child.
    @PostMapping("/auth-admin/realms/{realms}/groups")
    public void createGroup(ServerWebExchange exchange,
                            @PathVariable("realms") String realm,
                            @RequestBody GroupRepresentation groupRepresentation
    ) {
        keycloak.realm(realm).groups().add(groupRepresentation);
    }

    // Set or create child.
    @PostMapping("/auth-admin/realms/{realms}/groups/{group-id}/children")
    public void createChildGroup(ServerWebExchange exchange,
                                 @PathVariable("realms") String realm,
                                 @PathVariable("group-id") String groupId,
                                 @RequestBody GroupRepresentation groupRepresentation
    ) {
        keycloak.realm(realm).groups().group(groupId).subGroup(groupRepresentation);
    }


    // group - Role Mappings :: Composite
    @GetMapping("/auth-admin/realms/{realm}/groups/{group-id}/role-mappings/realm/composite")
    @ResponseBody
    public Mono<List<RoleRepresentation>> getComposite(
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId,
            ServerWebExchange exchange
    ) {
        return Mono.just(keycloak.realm(realm).groups().group(groupId).roles().realmLevel().listEffective());
    }

    // group - Role Mappings :: Available Roles
    @GetMapping("/auth-admin/realms/{realm}/groups/{group-id}/role-mappings/realm/available")
    @ResponseBody
    public Mono<List<RoleRepresentation>> getAvailableRoles(
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId,
            ServerWebExchange exchange
    ) {
        return Mono.just(keycloak.realm(realm).groups().group(groupId).roles().realmLevel().listAvailable());
    }

    // group - Role Mappings :: Assigned Roles
    @GetMapping("/auth-admin/realms/{realm}/groups/{group-id}/role-mappings/realm")
    @ResponseBody
    public Mono<List<RoleRepresentation>> getAssignedRoles(
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId,
            ServerWebExchange exchange
    ) {
        return Mono.just(keycloak.realm(realm).groups().group(groupId).roles().realmLevel().listAll());
    }

    @PostMapping("/auth-admin/realms/{realm}/groups/{group-id}/role-mappings/realm")
    public void addGroupRoles(
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId,
            @RequestBody List<RoleRepresentation> roleRepresentations,
            ServerWebExchange exchange
    ) {
        keycloak.realm(realm).groups().group(groupId).roles().realmLevel().add(roleRepresentations);
    }

    @DeleteMapping("/auth-admin/realms/{realm}/groups/{group-id}/role-mappings/realm")
    public void deleteGroupRoles(
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId,
            @RequestBody List<RoleRepresentation> roleRepresentations,
            ServerWebExchange exchange
    ) {
        keycloak.realm(realm).groups().group(groupId).roles().realmLevel().remove(roleRepresentations);
    }

    // group - Members
    @GetMapping("/auth-admin/realms/{realm}/groups/{group-id}/members")
    @ResponseBody
    public Mono<List<UserRepresentation>> getGroupMembers(
            @PathVariable("realm") String realm,
            @PathVariable("group-id") String groupId,
            ServerWebExchange exchange
    ) {
        return Mono.just(keycloak.realm(realm).groups().group(groupId).members());
    }

    @PostMapping("/auth-admin/realms/{realm}/role")
    public void createRole(@PathVariable("realm") String realm,
                           @RequestBody RoleRepresentation roleRepresentation) {
        keycloak.realm(realm).roles().create(roleRepresentation);
    }

    @GetMapping("/auth-admin/realms/{realm}/roles")
    @ResponseBody
    public Mono<List<RoleRepresentation>> getRoles(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {

        return Mono.just(
                keycloak.realm(realm)
                        .roles().list(false)
        );
    }

    @GetMapping("/auth-admin/realms/{realm}/role/{role}")
    @ResponseBody
    public Mono<RoleRepresentation> getRole(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @PathVariable("role") String role
    ) {

        return Mono.just(
                keycloak.realm(realm)
                        .roles()
                        .get(role).toRepresentation()
        );
    }

    @GetMapping("/auth-admin/realms/{realm}/role/{role}/users")
    @ResponseBody
    public Mono<Set<UserRepresentation>> getUsersInRole(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @PathVariable("role") String role
    ) {

        return Mono.just(
                keycloak.realm(realm)
                        .roles()
                        .get(role).getRoleUserMembers()
        );
    }

    @PutMapping("/auth-admin/realms/{realm}/role/{role}")
    public void updateRole(@PathVariable("realm") String realm,
                           @PathVariable("role") String role,
                           @RequestBody RoleRepresentation roleRepresentation) {
        keycloak.realm(realm).roles().get(role).update(roleRepresentation);
    }

    @DeleteMapping("/auth-admin/realms/{realm}/role/{role}")
    public void removeRole(@PathVariable("realm") String realm,
                           @PathVariable("role") String role) {
        keycloak.realm(realm).roles().get(role).remove();
    }

    @PostMapping("/auth-admin/realms/{realm}/user")
    @ResponseBody
    public Mono<Void> createUser(@PathVariable("realm") String realm,
                                 @RequestBody CreateUserVO createUserVO) {
        UserRepresentation userRepresentation = createUserVO.getUserRepresentation();
        CredentialRepresentation credentialRepresentation = createUserVO.getCredentialRepresentation();
        List<RoleRepresentation> roleRepresentation = Collections.singletonList(
                keycloak.realm(realm).roles().get("ROLE_USER").toRepresentation());

        userRepresentation.setEnabled(true);

        try (Response newUser = keycloak.realm(realm).users().create(userRepresentation);) {
            String userId = newUser.getLocation().getPath().replaceAll(".*/([^/]+)$", "$1");

            resetPassword(realm, userId, credentialRepresentation);
            addMappingRoles(realm, userId, roleRepresentation);
        }

        return Mono.empty();
    }

    @PutMapping("/auth-admin/realms/{realm}/user/{user}/reset-password")
    public Mono<Void> resetPassword(
            @PathVariable("realm") String realm,
            @PathVariable("user") String user,
            @RequestBody CredentialRepresentation credentialRepresentation) {
        credentialRepresentation.setType(CredentialRepresentation.PASSWORD);
        keycloak.realm(realm).users().get(user).resetPassword(credentialRepresentation);

        return Mono.empty();
    }

    @GetMapping("/auth-admin/realms/{realm}/users")
    @ResponseBody
    public Mono<List<UserRepresentation>> getUsers(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {

        return Mono.just(
                keycloak.realm(realm)
                        .users().list()
        );
    }

    @GetMapping("/auth-admin/realms/{realm}/user/{user}")
    @ResponseBody
    public Mono<UserRepresentation> getUser(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm,
            @PathVariable("user") String user
    ) {

        return Mono.just(
                keycloak.realm(realm)
                        .users()
                        .get(user).toRepresentation()
        );
    }

    @PutMapping("/auth-admin/realms/{realm}/user/{user}")
    public void updateUser(@PathVariable("realm") String realm,
                           @PathVariable("user") String user, @RequestBody UserRepresentation userRepresentation) {
        keycloak.realm(realm).users().get(user).update(userRepresentation);
    }

    @DeleteMapping("/auth-admin/realms/{realm}/user/{user}")
    public void removeUser(@PathVariable("realm") String realm,
                           @PathVariable("user") String user) {
        keycloak.realm(realm).users().get(user).remove();
    }

    @GetMapping("/auth-admin/realms/{realm}/user/{user}/role-mappings/realm")
    @ResponseBody
    public Mono<List<RoleRepresentation>> getRoleMappings(@PathVariable("realm") String realm,
                                                          @PathVariable("user") String user) {
        return Mono.just(keycloak.realm(realm).users().get(user).roles().getAll().getRealmMappings());
    }

    @PostMapping("/auth-admin/realms/{realm}/user/{user}/role-mappings/realm")
    public Mono<Void> addMappingRoles(@PathVariable("realm") String realm,
                                      @PathVariable("user") String user,
                                      @RequestBody List<RoleRepresentation> rolesToAdd) {
        keycloak.realm(realm).users().get(user).roles().realmLevel().add(rolesToAdd);
        return Mono.empty();
    }

    @DeleteMapping("/auth-admin/realms/{realm}/user/{user}/role-mappings/realm")
    public Mono<Void> deleteMappingRoles(@PathVariable("realm") String realm,
                                         @PathVariable("user") String user,
                                         @RequestBody List<RoleRepresentation> rolesToRemove) {
        keycloak.realm(realm).users().get(user).roles().realmLevel().remove(rolesToRemove);
        return Mono.empty();
    }

    @GetMapping("/auth-admin/realms/{realm}/user/{user}/groups")
    public Mono<List<GroupRepresentation>> getGroupMappings(@PathVariable("realm") String realm,
                                                            @PathVariable("user") String user) {
        return Mono.just(keycloak.realm(realm).users().get(user).groups());
    }

    @PutMapping("/auth-admin/realms/{realm}/user/{user}/groups/{group-id}")
    public Mono<Void> addMappingGroup(@PathVariable("realm") String realm,
                                      @PathVariable("user") String user,
                                      @PathVariable("group-id") String groupId) {
        keycloak.realm(realm).users().get(user).joinGroup(groupId);
        return Mono.empty();
    }

    @DeleteMapping("/auth-admin/realms/{realm}/user/{user}/groups/{group-id}")
    public Mono<Void> deleteMappingGroup(@PathVariable("realm") String realm,
                                         @PathVariable("user") String user,
                                         @PathVariable("group-id") String groupId) {
        keycloak.realm(realm).users().get(user).leaveGroup(groupId);
        return Mono.empty();
    }

    // Access Control
    @GetMapping("/auth-admin/realms/{realm}/access-control/users")
    @ResponseBody
    public Mono<List<UserRepresentation>> getAccessControlUsers(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {
        List<UserRepresentation> users = keycloak.realm(realm).users().list()
                .stream().map(user -> keycloak.realm(realm)
                        .users().get(user.getId()).toRepresentation())
                .collect(Collectors.toList());

        return Mono.just(keycloak.realm(realm).users().list());
    }

    public List<GroupRepresentation> getFlattenGroup(
            List<GroupRepresentation> groupRepresentationList, GroupRepresentation group) {
        groupRepresentationList.add(group);

        if (group.getSubGroups() != null && !group.getSubGroups().isEmpty()) {
            group.getSubGroups().forEach(subGroup -> {
                getFlattenGroup(groupRepresentationList, subGroup);
            });
        }

        return groupRepresentationList;
    }

    @GetMapping("/auth-admin/realms/{realm}/access-control/groups")
    @ResponseBody
    public Mono<List<GroupRepresentation>> getAccessControlGroups(
            ServerWebExchange exchange,
            @PathVariable("realm") String realm
    ) {
        List<GroupRepresentation> groups = new ArrayList<>();

        keycloak.realm(realm).groups().groups().forEach(
                tempGroup -> {
                    GroupRepresentation group = keycloak.realm(realm)
                            .groups()
                            .group(tempGroup.getId()).toRepresentation();

                    getFlattenGroup(groups, group);
                }
        );

        return Mono.just(groups);
    }
}
