(function ($) {
	"use strict";

	function SessionManager(userInfo, client) {
		this.userInfo = userInfo;
		this.client = client;

		this.documentId;
		this.sessionId;

		this.onContentsChange = function () {};
		this.unsubscribeState = function () {};
		this.unsubscribeContentsChange = function () {};
	}

	SessionManager.prototype.createSession = function () {
		return $.ajax({
			url: "/api/sessions/create",
			type: "POST",
			contentType: "application/json;charset=utf-8",
			data: JSON.stringify({ creatorName: this.documentId })
		});
	};

	SessionManager.prototype.setDocumentId = function (content) {
		return $.ajax({
			url: "/api/sessions/" + this.sessionId + "/set-document",
			type: "POST",
			contentType: "application/json;charset=utf-8",
			data: JSON.stringify({ documentId: this.documentId, content: content })
		});
	};

	SessionManager.prototype.remoteUserUpdate = function (selection, keyEvent) {
		// console.log(selection.getStartElement());

		if (this.client && this.client.connected) {
			var payload = {
				documentId: this.documentId,
				sessionId: this.sessionId,
				userInfo: {
					id: this.userInfo.userId,
					name: this.userInfo.userName.trim(),
					color: this.userInfo.userColor,
					// cursorPosition: {
					// 	// lineNumber: [...editor.getSelection().root.$.querySelectorAll("p")].indexOf(selection.getStartElement().$),
					// 	lineNumber: selection.getStartElement().getPositionedAncestor(),
					// 	column: selection.getRanges()[0].startOffset
					// },
					selection: { message: selection }
				}
			};
			console.log(payload);

			this.client.publish({ destination: "/app/selection", body: JSON.stringify(payload) });
		} else {
			this.client.deactivate();
			this.client.activate();
		}
	};

	SessionManager.prototype.joinRoom = function () {
		if (this.client && this.client.connected) {
			var joinPayload = $.extend(
				{},
				{
					sessionId: this.sessionId,
					documentId: this.documentId
				},
				this.userInfo
			);
			this.client.publish({ destination: "app/join", body: JSON.stringify(joinPayload) });
		} else {
			this.client.deactivate();
			this.client.activate();
		}
	};

	SessionManager.prototype.subscribeState = function () {
		if (this.client && this.client.connected) {
			var TOPIC_URL = "/topic/sessions/" + this.sessionId + "/state/document/" + this.documentId;
			var subscription = this.client.subscribe(TOPIC_URL, function (message) {
				var response = JSON.parse(message.body);
				console.log(response);
			});

			this.unsubscribeState = subscription.unsubscribe;
		} else {
			this.client.deactivate();
			this.client.activate();
		}
	};

	SessionManager.prototype.subscribeContentsChange = function () {
		if (this.client && this.client.connected) {
			var TOPIC_URL = "/topic/sessions/" + this.sessionId + "/selections/document/" + this.documentId;
			var subscription = this.client.subscribe(
				TOPIC_URL,
				function (message) {
					var response = JSON.parse(message.body);

					if (response.userInfo.id !== this.userInfo.userId) {
						console.log(response);

						this.onContentsChange(response.userInfo);
					}
				}.bind(this)
			);

			this.unsubscribeContentsChange = subscription.unsubscribe;
		} else {
			this.client.deactivate();
			this.client.activate();
		}
	};

	SessionManager.prototype.setRoom = function (documentId, content) {
		var deferred = $.Deferred();

		this.documentId = documentId;
		this.createSession(documentId).done(
			function (sessionInfo) {
				this.sessionId = sessionInfo.sessionId;
				this.setDocumentId(content).done(function () {
					deferred.resolve();
				});
			}.bind(this)
		);

		return deferred.promise();
	};

	SessionManager.prototype.closeRoom = function () {
		if (this.client && this.client.connected) {
			this.unsubscribeState();
			this.unsubscribeContentsChange();
		}
	};

	SessionManager.prototype.openWebsocket = function () {
		this.closeRoom();
		this.joinRoom();
		this.subscribeState();
		this.subscribeContentsChange();
	};

	var sessionManager = function (Client, SockJS, userInfo) {
		var SOCKET_URL = "/ws";
		var client = new Client({
			brokerURL: SOCKET_URL,
			webSocketFactory: function () {
				return new SockJS(SOCKET_URL);
			},
			debug: function (str) {
				console.log(str);
			},
			onConnect: function (frame) {
				console.log("Connected: " + frame);
			},
			onStompError: function (frame) {
				console.log("Broker reported error: " + frame.headers["message"]);
				console.log("Additional details: " + frame.body);
			},
			onWebSocketError: function (event) {
				console.error("Web Socket Error: ", event);
			},

			onDisconnect: function () {
				console.log("Disconnected");
			},
			reconnectDelay: 5000
		});

		client.activate();

		var _sessionManager = new SessionManager(userInfo, client);

		return _sessionManager;
	};

	$.sessionManager = sessionManager;
})(window.jQuery);
