package com.arms.api.kafka.reqadd.service;

import com.arms.api.kafka.reqadd.model.ReqAddDTO;
import com.arms.api.kafka.reqadd.model.ReqAddResponseVO;
import com.arms.api.util.redisrepo.CustomRedisTemplate;
import com.arms.util.DataSerializer;
import lombok.extern.slf4j.Slf4j;
import org.apache.kafka.clients.admin.AdminClient;
import org.apache.kafka.clients.admin.NewTopic;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Profile;
import org.springframework.kafka.core.KafkaTemplate;
import org.springframework.kafka.support.SendResult;
import org.springframework.stereotype.Service;
import org.springframework.util.concurrent.ListenableFuture;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

@Service
@Slf4j
public class ReqAddService {

    private static final String ALM_ISSUE_TOPIC = "REQADD";

    private static final String REQUIREMENT_KEY_FORMAT = "requirement:%s:%s:%s";

    private final KafkaTemplate<String, String> kafkaTemplate;

    private final AdminClient adminClient;

    public ReqAddService(
            @Qualifier("customRedisTemplateImpl") CustomRedisTemplate customRedisTemplate,
            KafkaTemplate<String, String> kafkaTemplate,
            AdminClient adminClient
    ) {
        this.kafkaTemplate = kafkaTemplate;
        this.adminClient = adminClient;
    }

    private String generateRequirementKey(String tableName, String type, String uuid) {
        return String.format(REQUIREMENT_KEY_FORMAT, tableName, type, uuid);
    }

    public List<ReqAddResponseVO> read() {

        log.info("Reading requirements");

        List<ReqAddResponseVO> responses = new ArrayList<>();

        //요구사항 읽기를 왜 ???? 카프카에서 읽어가려 하는거지?
        //레디스는 왜 ????

        return responses;
    }

    public void create(String tableName, ReqAddDTO reqAddDTO) {

        hasTopic();

        String uuid = UUID.randomUUID().toString();

        String requirementKey = generateRequirementKey(tableName, "CREATE", uuid);

        String jsonString = DataSerializer.serialize(reqAddDTO);

        Map<String, String> value = new HashMap<>();
        value.put("payload", jsonString );
        value.put("status", "READY");
        value.put("createdAt", Instant.now().toString());

        ListenableFuture<SendResult<String, String>> future = kafkaTemplate.send(ALM_ISSUE_TOPIC, requirementKey, jsonString);

        future.addCallback(
                result -> {
                    log.info("Successfully sent message");
                },
                ex -> {
                    log.error("Failed to send message", ex);
                }
        );

    }

    public boolean update(String tableName, String payload) {

        hasTopic();

        String uuid = UUID.randomUUID().toString();

        String requirementKey = generateRequirementKey(tableName, "UPDATE", uuid);

        kafkaTemplate.send(ALM_ISSUE_TOPIC, requirementKey, payload);

        return true;

    }

    public boolean delete(String tableName, String payload) {

        hasTopic();

        String uuid = UUID.randomUUID().toString();

        String requirementKey = generateRequirementKey(tableName, "DELETE", uuid);

        kafkaTemplate.send(ALM_ISSUE_TOPIC, requirementKey, payload);

        return true;

    }



    private void hasTopic() {
        if (!topicExists(ALM_ISSUE_TOPIC)) {
            createTopic(ALM_ISSUE_TOPIC);
            log.info("Created new topic: {}", ALM_ISSUE_TOPIC);
        }
    }

    private boolean topicExists(String topicName) {
        try {
            Set<String> topics = adminClient.listTopics().names().get();
            return topics.contains(topicName);
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.error("Thread interrupted while checking topic existence: {}", topicName, e);
            return false;
        } catch (Exception e) {
            log.error("Failed to check topic existence: {}", topicName, e);
            return false;
        }
    }

    private void createTopic(String topicName) {
        try {
            NewTopic newTopic = new NewTopic(topicName, 1, (short) 1);
            adminClient.createTopics(Collections.singleton(newTopic)).all().get();
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            log.error("Thread interrupted while creating topic: {}", topicName, e);
        } catch (Exception e) {
            log.error("Failed to create topic: {}", topicName, e);
        }
    }
}
