package com.arms.api.languageconfig.controller;

import com.arms.api.languageconfig.model.vo.LanguagePackFileVO;
import com.arms.api.languageconfig.model.vo.SingleLanguagePackVO;
import com.arms.api.languageconfig.service.LanguageConfigService;
import com.arms.egovframework.javaservice.gcframework.model.RepoType;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiParam;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.io.IOException;
import java.util.Calendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@RestController
@AllArgsConstructor
@Slf4j
@RequestMapping("/auth-sche/language-config")
public class LanguageConfigController {

    private final LanguageConfigService languageConfigService;

    /**
     * Gitea 저장소의 언어팩 파일 목록 조회
     *
     * @return 언어팩 파일들의 상세 정보 (파일명, 언어코드, SHA, 크기, 수정일시, 키 개수 등)
     */
    @ApiOperation(value = "언어팩 파일 목록 조회", notes = "저장소에 있는 모든 언어팩 파일의 목록과 상세 정보를 조회합니다.")
    @GetMapping("/packs/files")
    public ResponseEntity<List<LanguagePackFileVO>> getLanguagePackFilesNew(
            @ApiParam(value = "저장소 타입", defaultValue = "GITEA", required = true) 
            @RequestParam(defaultValue = "GITEA") RepoType repoType) {
        log.info("[ LanguageConfigController :: getLanguagePackFilesNew ] 동작 : {}", Calendar.getInstance().getTime());
        return ResponseEntity.ok(languageConfigService.getLanguagePackFiles(repoType));
    }

    /**
     * 특정 언어의 언어팩 조회
     *
     * @param language 언어 코드 (ko, en, ja, cn 등)
     * @return 해당 언어의 모든 key-value 쌍
     */
    @ApiOperation(value = "특정 언어팩 조회", notes = "지정된 언어 코드에 해당하는 언어팩의 모든 key-value 쌍을 조회합니다.")
    @GetMapping("/packs/language/{language}")
    public ResponseEntity<SingleLanguagePackVO> getSingleLanguagePack(
            @ApiParam(value = "언어 코드 (ko, en, ja, cn 등)", required = true, example = "ko")
            @PathVariable String language) {
        log.info("[ LanguageConfigController :: getSingleLanguagePack ] language: {}, 동작 : {}",
                language, Calendar.getInstance().getTime());
        return ResponseEntity.ok(languageConfigService.getSingleLanguagePack(language));
    }

    /**
     * 특정 언어의 언어팩 수정(및 언어팩 추가)
     *
     * @param language 언어 코드 (ko, en, ja, cn 등)
     * @return 해당 언어의 모든 key-value 쌍
     */
    @ApiOperation(value = "특정 언어팩 수정", notes = "지정된 언어 코드에 해당하는 언어팩을 수정하거나 새로운 언어팩을 추가합니다.")
    @PutMapping("/packs/language/{language}")
    public ResponseEntity<String> updateSingleLanguagePack(
                        @ApiParam(value = "언어 코드 (ko, en, ja, cn 등)", required = true, example = "ko")
                        @PathVariable String language,
                        @RequestBody List<Map<String, String>> data) throws IOException {
        log.info("[ LanguageConfigController :: updateSingleLanguagePack ] language: {}, 동작 : {}",
                language, Calendar.getInstance().getTime());
        // Convert List<Map<String, String>> to a single Map<String, String>
        Map<String, String> resultMap = new HashMap<>();
        for (Map<String, String> item : data) {
            resultMap.putAll(item);
        }

        return ResponseEntity.ok(languageConfigService.updateSingleLanguagePack(language, resultMap));
    }

    @GetMapping("/packs/language/{language}/refresh")
    public ResponseEntity<SingleLanguagePackVO> refreshSingleLanguagePack(@PathVariable String language) {
        log.info("[ LanguageConfigController :: refreshSingleLanguagePack] :: refresh single language => {}", language);

        return ResponseEntity.ok(languageConfigService.refreshSingleLanguagePack(language));
    }


    /**
     * 언어팩 삭제
     *
     * @param language 언어 코드 (ko, en, ja, cn 등)
     * @return 삭제 완료 응답
     */
    @ApiOperation(value = "특정 언어팩 삭제", notes = "지정된 언어 코드에 해당하는 언어팩을 삭제합니다.")
    @DeleteMapping("/packs/language/{language}")
    public ResponseEntity<String> deleteSingleLanguagePack(@PathVariable String language) {
        log.info("[ LanguageConfigController :: deleteSingleLanguagePack ] language: {}, 동작 : {}",
                language, Calendar.getInstance().getTime());
        return ResponseEntity.ok(languageConfigService.deleteSingleLanguagePack(language));
    }

    /**
     * 모든 언어팩의 전체 내용 조회
     * 각 언어팩별로 key-value 쌍을 목록(List)으로 반환합니다.
     *
     * @return SingleLanguagePackVO 목록(language, languagePack, totalKeys, lastModified)
     */
    @ApiOperation(value = "모든 언어팩 조회", notes = "모든 언어의 언어팩 내용을 조회합니다. 각 언어팩별로 key-value 쌍을 목록으로 반환합니다.")
    @GetMapping("/packs/all-contents")
    public ResponseEntity<List<SingleLanguagePackVO>> getAllLanguagePackContents() {
        log.info("[ LanguageConfigController :: getAllLanguagePackContents ] 동작 : {}", Calendar.getInstance().getTime());
        return ResponseEntity.ok(languageConfigService.getAllLanguagePackContents());
    }

    @ApiOperation(value = "모든 언어팩 갱신 및 조회", notes = "모든 언어의 언어팩 내용을 갱신합니다. 각 언어팩별로 key-value 쌍을 목록으로 반환합니다.")
    @GetMapping("/packs/all-contents/refresh")
    public ResponseEntity<List<SingleLanguagePackVO>> refreshAllLanguagePackContents() {
        log.info("[ LanguageConfigController :: refreshAllLanguagePackContents ] 동작 : {}", Calendar.getInstance().getTime());
        return ResponseEntity.ok(languageConfigService.refreshAllLanguagePackContents());
    }

    @ApiOperation(value = "KEY 가 이미 존재하는지 확인 및 반환", notes = "추가할 키(KEY)가 기존 언어팩에 있는지 확인합니다.")
    @GetMapping("/packs/language/{language}/keys/exists")
    public ResponseEntity<List<String>> checkLanguageKeyExists(@PathVariable String language, @RequestParam("keys") List<String> keys) {
        log.info("[ LanguageConfigController :: checkLanguageKeyExists ] 동작 : {}, NO. of keys : {}", Calendar.getInstance().getTime(), keys.size());
        return ResponseEntity.ok(languageConfigService.checkLanguageKeyExists(language, keys));
    }

}
