package com.arms.api.analysis.cost.service;

import com.arms.api.analysis.cost.model.dto.CalculationCostDTO;
import com.arms.api.analysis.cost.model.dto.CostDTO;
import com.arms.api.analysis.cost.model.dto.AssigneeSalaryDTO;
import com.arms.api.analysis.cost.model.dto.SalaryLogJdbcDTO;
import com.arms.api.analysis.cost.model.vo.*;
import com.arms.api.analysis.cost.model.vo.CostVO;
import com.arms.api.product_service.pdserviceversion.model.PdServiceVersionEntity;
import com.arms.api.product_service.pdserviceversion.service.PdServiceVersion;
import com.arms.api.requirement.reqadd.model.entity.ReqAddCostEntity;
import com.arms.api.requirement.reqadd.service.ReqAdd;
import com.arms.api.requirement.reqdifficulty.model.ReqDifficultyEntity;
import com.arms.api.requirement.reqpriority.model.ReqPriorityEntity;
import com.arms.api.requirement.reqstate.model.ReqStateEntity;
import com.arms.api.requirement.reqstate.service.ReqState;
import com.arms.api.requirement.reqstatus.model.ReqStatusDTO;
import com.arms.api.requirement.reqstatus.model.ReqStatusEntity;
import com.arms.api.requirement.reqstatus.service.ReqStatus;
import com.arms.api.util.VersionUtil;
import com.arms.api.util.communicate.external.AggregationService;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.api.util.communicate.internal.InternalService;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import com.arms.egovframework.javaservice.treeframework.remote.Chat;
import com.arms.egovframework.javaservice.treeframework.util.StringUtils;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.math.NumberUtils;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Restrictions;
import org.modelmapper.ModelMapper;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import java.text.DecimalFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

@Service
@RequiredArgsConstructor
@Slf4j
public class CostServiceImpl implements CostService {
    
    private static final String DATE_FORMAT = "yyyy-MM-dd";

    private final AggregationService aggregationService;

    private final PdServiceVersion pdServiceVersion;

    protected final Chat chat;

    private final SalaryLog salaryLog;

    private final InternalService internalCommunicator;

    private final ReqAdd reqAdd;

    private final ReqStatus reqStatus;

    private final ReqState reqStateService;

    private final SalaryService salaryService;

    protected final ModelMapper modelMapper;
    private final EngineService engineService;

    @Override
    public AllAssigneeVO getAssigneeList(CostDTO costDTO) throws Exception {

        Map<String, SalaryEntity> salaryEntityMap = salaryService.getAllSalariesMap();

        if (salaryEntityMap.isEmpty()) {
            log.info(" [ " + this.getClass().getName() + " :: 버전별_요구사항별_담당자가져오기 ] :: 디비에서 연봉 정보를 조회하는 데 실패했습니다.");
            chat.sendMessageByEngine("등록 된 연봉 정보가 없습니다.");
            return new AllAssigneeVO(new ArrayList<>());
        }

        ResponseEntity<List<CostVO>> responseEntity = aggregationService.getAssigneeList(costDTO);

        List<CostVO> costVOs = new ArrayList<>();
        costVOs.addAll(responseEntity.getBody());

        List<AssigneeSalaryVO> allAssigneeList = new ArrayList<>();

        for (CostVO costVO : costVOs) {
            String assigneeAccountId = costVO.getAssigneeKey();
            String assigneeDisplayName = costVO.getDisplayNameKey();

            Long salary = Optional.ofNullable(salaryEntityMap)
                    .flatMap(map -> Optional.ofNullable(map.get(assigneeAccountId)))
                    .map(SalaryEntity::getC_annual_income)
                    .filter(s -> !s.trim().isEmpty())
                    .map(NumberUtils::toLong)
                    .orElse(0L);

            AssigneeSalaryVO assigneeSalaryVO = AssigneeSalaryVO.builder()
                    .id(assigneeAccountId)
                    .name(assigneeDisplayName)
                    .salary(salary)
                    .build();

            allAssigneeList.add(assigneeSalaryVO);
        }

        return new AllAssigneeVO(allAssigneeList);
    }

    @Override
    public RequirementDifficultyAndPriorityVO getRequirementListStats(CostDTO costDTO) throws Exception {

        PdServiceAndIsReqDTO pdServiceAndIsReq = costDTO.getPdServiceAndIsReq();

        List<Long> pdServiceVersionLinks = pdServiceAndIsReq.getPdServiceVersionLinks();

        if (pdServiceVersionLinks == null || pdServiceVersionLinks.isEmpty()) {
            return null;
        }

        String[] versionStrArr = pdServiceVersionLinks.stream().map(Object::toString).toArray(String[]::new);

        RequirementDifficultyAndPriorityVO response = new RequirementDifficultyAndPriorityVO();

        Disjunction orCondition = Restrictions.disjunction();
        for (String versionStr : versionStrArr) {
            versionStr = "\\\"" + versionStr + "\\\"";
            orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", versionStr, MatchMode.ANYWHERE));
        }

        ReqAddCostEntity reqAddCostEntity = new ReqAddCostEntity();

        reqAddCostEntity.getCriterions().add(orCondition);

        List<ReqAddCostEntity> reqAddCostEntities = reqAdd.getChildNode(reqAddCostEntity);

        String startDate = costDTO.getStartDate();
        if (StringUtils.isNullCheck(startDate)) {
            Date searchStartDate = parseDate(startDate);
            reqAddCostEntities = reqAddCostEntities.stream()
                    .filter(srr -> {
                        Date reqStartDate = srr.getC_req_start_date();
                        return reqStartDate != null && !reqStartDate.before(searchStartDate);
                    })
                    .collect(Collectors.toList());
        }

        String endDate = costDTO.getEndDate();
        if (StringUtils.isNullCheck(endDate)) {
            Date searchEndDate = parseDate(endDate);
            reqAddCostEntities = reqAddCostEntities.stream()
                    .filter(srr -> {
                        Date reqEndDate = srr.getC_req_end_date();
                        return reqEndDate != null && !reqEndDate.after(searchEndDate);
                    })
                    .collect(Collectors.toList());
        }

        List<ReqAddCostVO> reqAddCostVOs = reqAddCostEntities.stream()
                .map(this::convertEntityToReqAddCostVO)
                .collect(Collectors.toList());

        response.setRequirement(reqAddCostVOs);

        Map<String, Long> difficultyMap = new HashMap<>();
        Map<String, Long> priorityMap = new HashMap<>();

        reqAddCostEntities.forEach(reqAddCost -> {
            ReqDifficultyEntity reqDifficultyEntity = reqAddCost.getReqDifficultyEntity();
            ReqPriorityEntity reqPriorityEntity = reqAddCost.getReqPriorityEntity();

            if (reqDifficultyEntity != null) {
                difficultyMap.merge(reqDifficultyEntity.getC_title(), 1L, Long::sum);
            }

            if (reqPriorityEntity != null) {
                priorityMap.merge(reqPriorityEntity.getC_title(), 1L, Long::sum);
            }
        });

        if (!priorityMap.isEmpty()) {
            response.setPriority(priorityMap);
        }

        if (!difficultyMap.isEmpty()) {
            response.setDifficulty(difficultyMap);
        }

        return response;
    }

    private ReqAddCostVO convertEntityToReqAddCostVO(ReqAddCostEntity reqAddCostEntity) {
        String stateName =  reqAddCostEntity.getReqStatePureEntity() != null
                ? reqAddCostEntity.getReqStatePureEntity().getC_title() : null;

        return ReqAddCostVO.builder()
                .c_id(reqAddCostEntity.getC_id())
                .c_title(reqAddCostEntity.getC_title())
                .c_req_start_date(reqAddCostEntity.getC_req_start_date())
                .c_req_end_date(reqAddCostEntity.getC_req_end_date())
                .c_req_pdservice_versionset_link(reqAddCostEntity.getC_req_pdservice_versionset_link())
                .c_req_state_title(stateName)
                .reqAmount(0L)
                .build();
    }

    private Date parseDate(String dateStr) {
        if (StringUtils.isNullCheck(dateStr)) {
            SimpleDateFormat dateFormat = new SimpleDateFormat(DATE_FORMAT);
            try {
                return dateFormat.parse(dateStr);
            } catch (ParseException e) {
                log.error("잘못된 날짜 형식: {} (yyyy-MM-dd 형식 필요)", dateStr);
            }
        }
        return null;
    }

    @Override
    public LinkedJiraIssueVO getLinkedJiraIssuesByVersionAndRequirement(CostDTO costDTO) throws Exception {

        PdServiceAndIsReqDTO pdServiceAndIsReq = costDTO.getPdServiceAndIsReq();

        Long pdServiceLink = pdServiceAndIsReq.getPdServiceLink();
        List<Long> pdServiceVersionLinks = pdServiceAndIsReq.getPdServiceVersionLinks();

        List<ReqStatusEntity> reqStatusEntities = this.getReqStatusListByVersionAndDateFilter(pdServiceLink, pdServiceVersionLinks, costDTO.getStartDate(), costDTO.getEndDate());

        LinkedJiraIssueVO linkedJiraIssueVO = new LinkedJiraIssueVO();
        // key1 - version id, key2 - reqLink
        Map<String, Map<Long, List<LinkedJiraIssueVO.RequirementData>>> groupingMap =
                reqStatusEntities.stream()
                        .map(LinkedJiraIssueVO::createRequirementData)
                        .filter(데이터 -> 데이터.getC_req_pdservice_versionset_link() != null && !데이터.getC_req_pdservice_versionset_link().isEmpty())
                        .flatMap(데이터 ->
                                Arrays.stream(데이터.getC_req_pdservice_versionset_link().split("[\\[\\],\"]"))
                                        .filter(s -> !s.isEmpty())
                                        .map(버전데이터 -> new AbstractMap.SimpleImmutableEntry<>(버전데이터, 데이터)))
                        .collect(Collectors.groupingBy(Map.Entry::getKey,
                                Collectors.groupingBy(entry -> entry.getValue().getC_req_link(),
                                        Collectors.mapping(Map.Entry::getValue, Collectors.toList()))));

        linkedJiraIssueVO.setLinkedJiraIssueData(groupingMap);

        return linkedJiraIssueVO;
    }

    private List<ReqStatusEntity> getReqStatusListByVersionAndDateFilter(Long 제품및서비스, List<Long> 버전, String startDate, String endDate) throws Exception {

        ReqStatusEntity reqStatusEntity = new ReqStatusEntity();

        String 조회대상_지라이슈상태_테이블 = "T_ARMS_REQSTATUS_" + 제품및서비스;

        log.info("조회 대상 테이블 searchTable :" + 조회대상_지라이슈상태_테이블);

        String[] versionStrArr = 버전.stream()
                .map(Object::toString)
                .toArray(String[]::new);

        Disjunction orCondition = Restrictions.disjunction();
        for (String versionStr : versionStrArr) {
            versionStr = "\\\"" + versionStr + "\\\"";
            orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", versionStr, MatchMode.ANYWHERE));
        }

        reqStatusEntity.getCriterions().add(orCondition);

        List<ReqStatusEntity> searchResultReq = reqStatus.getChildNode(reqStatusEntity);

        if (StringUtils.isNullCheck(startDate)) {
            Date searchStartDate = parseDate(startDate);
            searchResultReq = searchResultReq.stream()
                    .filter(srr -> {
                        Date reqStartDate = srr.getC_req_start_date();
                        return reqStartDate != null && !reqStartDate.before(searchStartDate);
                    })
                    .collect(Collectors.toList());
        }

        if (StringUtils.isNullCheck(endDate)) {
            Date searchEndDate = parseDate(endDate);
            searchResultReq = searchResultReq.stream()
                    .filter(srr -> {
                        Date reqEndDate = srr.getC_req_end_date();
                        return reqEndDate != null && !reqEndDate.after(searchEndDate);
                    })
                    .collect(Collectors.toList());
        }

        return searchResultReq;
    }

    @Override
    public ProductCostResponseVO candleStickChartAPI(CostDTO costDTO) throws Exception {
        PdServiceAndIsReqDTO pdServiceAndIsReqDTO = costDTO.getPdServiceAndIsReq();
        Long pdServiceLink = pdServiceAndIsReqDTO.getPdServiceLink();
        List<Long> pdServiceVersionLinks = pdServiceAndIsReqDTO.getPdServiceVersionLinks();

        // 1. 해결 된 이슈를 찾기 위해 해결 상태값을 조회함. (ReqState)
        Map<Long, ReqStateEntity> 완료상태맵 = reqStateService.완료상태조회();
        List<Long> filteredReqStateId = filterResolvedStateIds(완료상태맵);

        // 2. ReqStatus(요구사항)을 조회함
        List<ReqStatusEntity> reqStatusEntities = internalCommunicator.제품별_요구사항_상황_조회("T_ARMS_REQSTATUS_" + pdServiceLink, new ReqStatusDTO());

        // 3. 요구사항의 ReqStateLink 값을 가지고 필터링함. ReqState 값이 완료 키워드인 ReqStatus 만 가져옴
        List<ReqStatusEntity> filteredReqStatusEntities = filterResolvedReqStatusEntities(reqStatusEntities, filteredReqStateId);

        // 4. 엔진 통신 cReqLink 기준 집계 및 필터링
        List<Long> cReqLinks = filteredReqStatusEntities.stream().map(ReqStatusEntity::getC_req_link).distinct().collect(Collectors.toList());

        List<CostVO> engineResponse = Optional.ofNullable(aggregationService.aggregationByReqLinkAndAssigneeAccountId(costDTO).getBody()).orElseGet(ArrayList::new);

        List<CostVO> groupByCReqLink = engineResponse.stream().filter(response -> response.getReqLinkKey() != null && response.getAssigneeKey() != null)
                .filter(response -> cReqLinks.contains(Long.parseLong(response.getReqLinkKey())))
                .collect(Collectors.toList());

        // 5. 제품 버전을 기준으로 x 축에 해당하는 시작일, 종료일 구하기
        List<PdServiceVersionEntity> pdServiceVersionEntities = pdServiceVersion.getNodesWithoutRoot(new PdServiceVersionEntity())
                .stream().filter(pdServiceVersionEntity -> pdServiceVersionLinks.contains(pdServiceVersionEntity.getC_id())).collect(Collectors.toList());

        String startDateOrNull = pdServiceVersionEntities.stream()
                .filter(pdServiceVersionEntity -> !pdServiceVersionEntity.getC_pds_version_start_date().equals("start"))
                .map(PdServiceVersionEntity::getC_pds_version_start_date)
                .min(String::compareTo).orElse(null);

        String endDateOrNull = pdServiceVersionEntities.stream()
                .filter(pdServiceVersionEntity -> !pdServiceVersionEntity.getC_pds_version_end_date().equals("end"))
                .map(PdServiceVersionEntity::getC_pds_version_end_date)
                .max(String::compareTo).orElse(null);

        if (startDateOrNull == null || endDateOrNull == null) {
            chat.sendMessageByEngine("제품 버전의 시작일과 종료일이 없습니다.");
            return new ProductCostResponseVO(new TreeMap<>(), new TreeMap<>(), new TreeMap<>());
        }

        String formattedStartDate = convertDateTimeFormat(startDateOrNull);
        String formattedEndDate = convertDateTimeFormat(endDateOrNull);

        LocalDate versionStartDate = LocalDate.parse(formattedStartDate);
        LocalDate versionEndDate = LocalDate.parse(formattedEndDate);

        // 6. 작업자 별 최초 연봉 데이터 추가 시 쌓인 "create" log 조회
        Set<String> getAssignees = getAssignees(pdServiceLink, pdServiceVersionLinks);
        Map<String, SalaryLogJdbcDTO> salaryCreateLogs = salaryLog.findAllLogsToMap("create", endDateOrNull);
        Map<String, SalaryLogJdbcDTO> filteredSalaryCreateLogs = salaryCreateLogs.entrySet().stream()
                .filter(entry -> getAssignees.contains(entry.getKey()))
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));

        if (filteredSalaryCreateLogs.isEmpty()) {
            chat.sendMessageByEngine("버전의 기간 이후 연봉을 등록한 경우, 해당 버전은 비용 산정이 되지 않습니다.");
            SortedMap<String, Integer> dailyCost = generateDailyCostsMap(formattedStartDate, formattedEndDate, 0);
            return new ProductCostResponseVO(dailyCost, dailyCost, generateDailyCostsCandleStick(versionStartDate, versionEndDate));
        }

        // 7. 작업자 별 최초 연봉 수정 시 쌓인 "update" log 조회
        List<SalaryLogJdbcDTO> salaryUpdateLogs = salaryLog.findAllLogs("update", formattedStartDate, formattedEndDate);

        // 8. 같은 날 연봉 데이터를 여러번 수정한 경우, 가장 마지막에 등록한 연봉 데이터 1개만 꺼내온다.
        List<SalaryLogJdbcDTO> filteredLogs = getLatestSalaryUpdates(salaryUpdateLogs, getAssignees);

        filteredLogs.sort(Comparator.comparing(SalaryLogJdbcDTO::getFormatted_date));

        // 9. 담당자 별 연봉 캘린더 생성
        Map<String, SortedMap<String, Integer>> allAssigneeSalaries = assigneeCostCalendar(filteredSalaryCreateLogs, filteredLogs, versionStartDate, versionEndDate);

        // 10. 완료 된 요구사항에 대한 비용 캘린더 생성. 기본값으로 0을 세팅
        SortedMap<String, Integer> barCost = generateDailyCostsMap(formattedStartDate, formattedEndDate, 0);

        // 10-1. 완료 된 요구사항으로 루프를 돌면서, 각 요구사항의 시작일과 종료일에 맞는 담당자의 연봉 데이터를 기반으로 성과 비용을 책정한다.
        calculateBarCost(filteredReqStatusEntities, versionEndDate, groupByCReqLink, allAssigneeSalaries, barCost);

        // 11. 성과 기준선을 책정하기 위한 변수 세팅
        SortedMap<String, Integer> lineCost = getLineCost(allAssigneeSalaries);

        // 12. 총 연봉 비용의 변동 추이를 보여주기 위한 캔들스틱 차트 데이터 세팅
        // 12-1. 업데이트 로그를 담당자, 날짜 별로 그루핑
        Map<String, Map<String, List<SalaryLogJdbcDTO>>> updatesGroupedByDateAndKey = salaryUpdateLogs.stream()
                .collect(Collectors.groupingBy(SalaryLogJdbcDTO::getC_key,
                        Collectors.groupingBy(SalaryLogJdbcDTO::getFormatted_date)));

        // 12-2. 담당자 별 연봉 캘린더를 활용하여 캔들스틱 차트 데이터 생성. 연봉 캘린더의 금액으로 시가, 종가를 알 수 있다.
        Map<String, SortedMap<String, CandleStickVO>> allAssigneeCandleSticks = getAllAssigneeCandleSticks(allAssigneeSalaries, updatesGroupedByDateAndKey);

        SortedMap<String, List<Integer>> candleStickCost = getCandleStickCost(allAssigneeCandleSticks);

        return new ProductCostResponseVO(lineCost, barCost, candleStickCost);
    }

    private void calculateBarCost(
            List<ReqStatusEntity> filteredReqStatusEntities,
            LocalDate versionEndDate,
            List<CostVO> groupByCReqLink,
            Map<String, SortedMap<String, Integer>> allAssigneeSalaries,
            SortedMap<String, Integer> barCost
    ) {
        // 10-1. 완료 된 요구사항으로 루프를 돌면서, 각 요구사항의 시작일과 종료일에 맞는 담당자의 연봉 데이터를 기반으로 성과 비용을 책정한다.
        for (ReqStatusEntity filteredReqStatusEntity : filteredReqStatusEntities) {
            LocalDate 요구사항시작일 = filteredReqStatusEntity.getC_req_start_date().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            LocalDate 요구사항종료일 = filteredReqStatusEntity.getC_req_end_date().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
            LocalDate adjustedEndDate = 요구사항종료일.isAfter(versionEndDate) ? versionEndDate : 요구사항종료일;

            List<CostVO> costVOList = groupByCReqLink.stream().filter(a -> Long.parseLong(a.getReqLinkKey()) == filteredReqStatusEntity.getC_req_link()).collect(Collectors.toList());

            costVOList.forEach(assignee -> {
                Optional.ofNullable(allAssigneeSalaries.get(assignee.getAssigneeKey())).ifPresent(assigneeSalaries -> {
                    assigneeSalaries.entrySet().stream().filter(entry -> {
                        LocalDate date = LocalDate.parse(entry.getKey());
                        return (date.isAfter(요구사항시작일) || date.isEqual(요구사항시작일)) && (date.isBefore(adjustedEndDate) || date.isEqual(adjustedEndDate));
                    }).forEach(entry -> {
                        barCost.merge(entry.getKey(), entry.getValue(), Integer::sum);
                    });
                });
            });
        }

        // 10-2. 요구사항의 연봉 캘린더는 만원 단위이기 때문에, 원 단위로 환산하고, 365로 나누어 일 단위로 변환.
        barCost.replaceAll((k, v) -> v / 365);

        // 10-3. 요구사항의 연봉 캘린더를 일 별 누적시킨다.
        int barSum = 0;
        for (Map.Entry<String, Integer> entry : barCost.entrySet()) {
            barSum += entry.getValue();
            barCost.put(entry.getKey(), barSum);
        }
    }

    private SortedMap<String, List<Integer>> getCandleStickCost(Map<String, SortedMap<String, CandleStickVO>> allAssigneeCandleSticks) {
        SortedMap<String, List<Integer>> candleStickCost = new TreeMap<>();

        // 12-5. 모든 담당자의 캔들스틱 데이터를 하나로 합친다. 최종 연봉 비용의 변동 추이를 보여주기 위함.
        for (SortedMap<String, CandleStickVO> assigneeCandleSticks : allAssigneeCandleSticks.values()) {
            for (Map.Entry<String, CandleStickVO> entry : assigneeCandleSticks.entrySet()) {
                String date = entry.getKey();
                CandleStickVO candleStickVO = entry.getValue();
                List<Integer> sums = candleStickCost.getOrDefault(date, Arrays.asList(0, 0, 0, 0));

                sums.set(0, sums.get(0) + (candleStickVO.get시가()));
                sums.set(1, sums.get(1) + (candleStickVO.get종가()));
                sums.set(2, sums.get(2) + (candleStickVO.get최저가()));
                sums.set(3, sums.get(3) + (candleStickVO.get최고가()));

                candleStickCost.put(date, sums);
            }
        }
        return candleStickCost;
    }

    private Map<String, SortedMap<String, CandleStickVO>> getAllAssigneeCandleSticks(Map<String, SortedMap<String, Integer>> allAssigneeSalaries, Map<String, Map<String, List<SalaryLogJdbcDTO>>> updatesGroupedByDateAndKey) {
        Map<String, SortedMap<String, CandleStickVO>> allAssigneeCandleSticks = new HashMap<>();
        for (Map.Entry<String, SortedMap<String, Integer>> assigneeEntry : allAssigneeSalaries.entrySet()) {
            String assignee = assigneeEntry.getKey();
            SortedMap<String, Integer> salaryMap = assigneeEntry.getValue();
            Map<String, List<SalaryLogJdbcDTO>> updateLogsByAssignee = updatesGroupedByDateAndKey.get(assignee);
            SortedMap<String, CandleStickVO> candleStickMap = new TreeMap<>();
            String previousDate = null;
            for (Map.Entry<String, Integer> salaryEntry : salaryMap.entrySet()) {
                String date = salaryEntry.getKey();
                Integer salary = salaryEntry.getValue();
                int min = 0;
                int max = 0;
                int open = 0;
                if (previousDate != null) {
                    CandleStickVO previousCandleStickVO = candleStickMap.get(previousDate);
                    open = previousCandleStickVO.get종가(); // 이전 날짜의 종가를 현재 날짜의 시가로 설정
                }
                // 12-3. 시가, 종가, 최저가, 최고가 기본 세팅
                min = open > salary ? salary : open;
                max = open > salary ? open : salary;

                // 12-4. 업데이트 로그가 있는 날짜엔 해당 날짜의 최소값과 최대값을 찾아서 min, max 를 업데이트한다.
                if (updateLogsByAssignee != null && updateLogsByAssignee.get(date) != null) {
                    List<SalaryLogJdbcDTO> salaryLogJdbcDTOList = updateLogsByAssignee.get(date).stream().sorted(Comparator.comparing(SalaryLogJdbcDTO::getC_annual_income)).collect(Collectors.toList());
                    if (!salaryLogJdbcDTOList.isEmpty()) {
                        if (salaryLogJdbcDTOList.size() == 1) {
                            Integer updateLogSalary = salaryLogJdbcDTOList.get(0).getC_annual_income();
                            min = min > updateLogSalary ? updateLogSalary : min;
                            max = max > updateLogSalary ? max : updateLogSalary;
                        } else {
                            // 해당 담당자의 해당 날짜의 모든 업데이트 로그를 정렬하여 최소값(인덱스 0)과 최대값(인덱스 size - 1)을 찾는다.
                            Integer updateLogMinSalary = salaryLogJdbcDTOList.get(0).getC_annual_income();
                            Integer updateLogMaxSalary = salaryLogJdbcDTOList.get(salaryLogJdbcDTOList.size() - 1).getC_annual_income();
                            min = min > updateLogMinSalary ? updateLogMinSalary : min;
                            max = max > updateLogMaxSalary ? max : updateLogMaxSalary;
                        }
                    }
                }

                CandleStickVO candleStickVO = new CandleStickVO(open, salary, min, max);
                candleStickMap.put(date, candleStickVO);

                previousDate = date;
            }

            allAssigneeCandleSticks.put(assignee, candleStickMap);
        }
        return allAssigneeCandleSticks;
    }

    private SortedMap<String, Integer> getLineCost(Map<String, SortedMap<String, Integer>> allAssigneeSalaries) {
        SortedMap<String, Integer> lineCost = new TreeMap<>();

        // 11-1. 담당자 별 연봉 캘린더를 활용하여 각 날짜에 해당하는 연봉 데이터를 가져와서 합산한다.
        // 담당자를 별도로 구분하지 않고, 모든 성과를 각 날짜 별로 합치는 과정임에 주의한다.
        for (SortedMap<String, Integer> assigneeSalaries : allAssigneeSalaries.values()) {
            for (Map.Entry<String, Integer> entry : assigneeSalaries.entrySet()) {
                lineCost.merge(entry.getKey(), entry.getValue(), Integer::sum);
            }
        }

        // 11-2. 원 단위로 환산하고, 365로 나누어 일 단위로 변환.
        lineCost.replaceAll((k, v) -> v / 365);

        // 11-3. 성과 기준선의 비용을 누적시킨다.
        int lineSum = 0;
        for (Map.Entry<String, Integer> entry : lineCost.entrySet()) {
            lineSum += entry.getValue();
            lineCost.put(entry.getKey(), lineSum);
        }
        return lineCost;
    }

    private Map<String, SortedMap<String, Integer>> assigneeCostCalendar(Map<String, SalaryLogJdbcDTO> salaryCreateLogs, List<SalaryLogJdbcDTO> filteredLogs, LocalDate versionStartDate, LocalDate versionEndDate) {
        Map<String, SortedMap<String, Integer>> allAssigneeSalaries = new HashMap<>();
        for (Map.Entry<String, SalaryLogJdbcDTO> salaryCreateLog : salaryCreateLogs.entrySet()) {
            String assigneeKey = salaryCreateLog.getKey();
            SalaryLogJdbcDTO salaryCreate = salaryCreateLog.getValue();
            LocalDate salaryCreateDate = LocalDate.parse(salaryCreate.getFormatted_date());
            int createdSalary = salaryCreate.getC_annual_income();
            List<SalaryLogJdbcDTO> salaryUpdateLogsByAssignee = filteredLogs.stream().filter(sle -> sle.getC_key().equals(assigneeKey)).collect(Collectors.toList());
            int updateLogSize = salaryUpdateLogsByAssignee.size();

            // 1-1. 정상적인 케이스. 버전 먼저 등록하고, 이후에 연봉 데이터를 입력한 경우.
            if (versionStartDate.isBefore(salaryCreateDate)) {
                addSalaryForPeriod(versionStartDate, salaryCreateDate.minusDays(1), 0, assigneeKey, allAssigneeSalaries);

                if (hasUpdateLog(salaryUpdateLogsByAssignee)) {
                    updateSalaryForSection(salaryCreateDate, versionEndDate, updateLogSize, salaryUpdateLogsByAssignee, assigneeKey, allAssigneeSalaries, createdSalary);
                }
                if (!hasUpdateLog(salaryUpdateLogsByAssignee)) {
                    addSalaryForPeriod(salaryCreateDate, versionEndDate, createdSalary, assigneeKey, allAssigneeSalaries);
                }
            }
            // 1-2. 정상적인 케이스. 제품 버전 시작일과 연봉 데이터 입력일이 같은 경우.
            if (versionStartDate.isEqual(salaryCreateDate)) {
                if (hasUpdateLog(salaryUpdateLogsByAssignee)) {
                    updateSalaryForSection(versionStartDate, versionEndDate, updateLogSize, salaryUpdateLogsByAssignee, assigneeKey, allAssigneeSalaries, createdSalary);
                }
                if (!hasUpdateLog(salaryUpdateLogsByAssignee)) {
                    addSalaryForPeriod(versionStartDate, versionEndDate, createdSalary, assigneeKey, allAssigneeSalaries);
                }
            }
            // 1-3. 비정상적인 케이스. 버전 생성 전 연봉 데이터를 먼저 넣은 경우.
            if (versionStartDate.isAfter(salaryCreateDate)) {
                if (hasUpdateLog(salaryUpdateLogsByAssignee)) {
                    updateSalaryForSection(versionStartDate, versionEndDate, updateLogSize, salaryUpdateLogsByAssignee, assigneeKey, allAssigneeSalaries, createdSalary);
                }
                if (!hasUpdateLog(salaryUpdateLogsByAssignee)) {
                    addSalaryForPeriod(versionStartDate, versionEndDate, createdSalary, assigneeKey, allAssigneeSalaries);
                }
            }
        }
        return allAssigneeSalaries;
    }

    private void updateSalaryForSection(LocalDate startDate, LocalDate endDate, int updateLogSize, List<SalaryLogJdbcDTO> salaryUpdateLogsByAssignee, String assigneeKey, Map<String, SortedMap<String, Integer>> allAssigneeSalaries, int createdSalary) {
        for (int i = 0; i < updateLogSize; i++) {
            if (i == 0) {
                updateSalaryForFirstLog(i, salaryUpdateLogsByAssignee, startDate, assigneeKey, allAssigneeSalaries, createdSalary);
            } else {
                updateSalaryForMiddleLog(i, salaryUpdateLogsByAssignee, assigneeKey, allAssigneeSalaries);
            }
            updateSalaryForLastLog(i, updateLogSize, salaryUpdateLogsByAssignee, endDate, assigneeKey, allAssigneeSalaries);
        }
    }

    private List<Long> filterResolvedStateIds(Map<Long, ReqStateEntity> 완료상태맵) {
        return 완료상태맵.keySet().stream().collect(Collectors.toList());
    }

    private List<ReqStatusEntity> filterResolvedReqStatusEntities(List<ReqStatusEntity> reqStatusEntities, List<Long> filteredReqStateId) {
        Map<Long, ReqStatusEntity> uniqueMap = reqStatusEntities.stream()
                .filter(reqStatusEntity -> reqStatusEntity.getC_req_start_date() != null)
                .filter(reqStatusEntity -> reqStatusEntity.getC_req_end_date() != null)
                .filter(reqStatusEntity -> filteredReqStateId.contains(reqStatusEntity.getC_req_state_link()))
                .collect(Collectors.toMap(ReqStatusEntity::getC_req_link, reqStatusEntity -> reqStatusEntity, (existing, replacement) -> existing));

        return new ArrayList<>(uniqueMap.values());
    }

    private void addSalaryForPeriod(LocalDate startDate, LocalDate endDate, int salary, String assigneeKey, Map<String, SortedMap<String, Integer>> allAssigneeSalaries) {
        SortedMap<String, Integer> assigneeSalaries = allAssigneeSalaries.getOrDefault(assigneeKey, new TreeMap<>());
        for (LocalDate date = startDate; !date.isAfter(endDate); date = date.plusDays(1)) {
            assigneeSalaries.put(date.format(DateTimeFormatter.ofPattern(DATE_FORMAT)), salary);
        }
        allAssigneeSalaries.put(assigneeKey, assigneeSalaries);
    }

    private void updateSalaryDataForPeriod(LocalDate startDate, LocalDate endDate, String assigneeKey, Map<String, SortedMap<String, Integer>> allAssigneeSalaries, int updatedSalary) {
        SortedMap<String, Integer> assigneeSalaries = allAssigneeSalaries.getOrDefault(assigneeKey, new TreeMap<>());
        for (LocalDate date = startDate; !date.isAfter(endDate); date = date.plusDays(1)) {
            String dateString = date.format(DateTimeFormatter.ofPattern(DATE_FORMAT));
            assigneeSalaries.put(dateString, updatedSalary);
        }
        allAssigneeSalaries.put(assigneeKey, assigneeSalaries);
    }

    public boolean hasUpdateLog(List<SalaryLogJdbcDTO> salaryUpdateLogsByAssignee) {
        if (salaryUpdateLogsByAssignee.isEmpty()) {
            return false;
        }
        return true;
    }

    private void updateSalaryForFirstLog(
            int i,
            List<SalaryLogJdbcDTO> salaryUpdateLogsByAssignee,
            LocalDate startDate,
            String assigneeKey,
            Map<String, SortedMap<String, Integer>> allAssigneeSalaries, int createdSalary
    ) {
        LocalDate endDate = LocalDate.parse(salaryUpdateLogsByAssignee.get(i).getFormatted_date()).minusDays(1);
        updateSalaryDataForPeriod(startDate, endDate, assigneeKey, allAssigneeSalaries, createdSalary);
    }

    private void updateSalaryForMiddleLog(
            int i,
            List<SalaryLogJdbcDTO> salaryUpdateLogsByAssignee,
            String assigneeKey,
            Map<String, SortedMap<String, Integer>> allAssigneeSalaries
    ) {
        LocalDate startDate = LocalDate.parse(salaryUpdateLogsByAssignee.get(i - 1).getFormatted_date());
        LocalDate endDate = LocalDate.parse(salaryUpdateLogsByAssignee.get(i).getFormatted_date()).minusDays(1);
        int updatedSalary = salaryUpdateLogsByAssignee.get(i - 1).getC_annual_income();
        updateSalaryDataForPeriod(startDate, endDate, assigneeKey, allAssigneeSalaries, updatedSalary);
    }

    private void updateSalaryForLastLog(
            int i,
            int updateLogSize,
            List<SalaryLogJdbcDTO> salaryUpdateLogsByAssignee,
            LocalDate versionEndDate,
            String assigneeKey,
            Map<String, SortedMap<String, Integer>> allAssigneeSalaries
    ) {
        if (isLastLog(updateLogSize, i)) {
            int currentSalary = salaryUpdateLogsByAssignee.get(i).getC_annual_income();
            LocalDate currentStart = LocalDate.parse(salaryUpdateLogsByAssignee.get(i).getFormatted_date());
            updateSalaryDataForPeriod(currentStart, versionEndDate, assigneeKey, allAssigneeSalaries, currentSalary);
        }
    }

    private boolean isLastLog(int updateLogSize, int i) {
        if (updateLogSize == i + 1) {
            return true;
        }
        return false;
    }

    private List<SalaryLogJdbcDTO> getLatestSalaryUpdates(List<SalaryLogJdbcDTO> salaryUpdateLogs, Set<String> getAssignees) {

        List<SalaryLogJdbcDTO> filteredUpdateLogs = salaryUpdateLogs.stream()
                .filter(salaryLogJdbcDTO -> getAssignees.contains(salaryLogJdbcDTO.getC_key()))
                .collect(Collectors.toList());

        Map<String, Map<String, List<SalaryLogJdbcDTO>>> updatesGroupedByDateAndKey = filteredUpdateLogs.stream()
                .collect(Collectors.groupingBy(SalaryLogJdbcDTO::getFormatted_date,
                        Collectors.groupingBy(SalaryLogJdbcDTO::getC_key)));

        return updatesGroupedByDateAndKey.values().stream()
                .flatMap(dateGroup -> dateGroup.values().stream())
                .map(this::getLatestLogFromGroup)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    private SalaryLogJdbcDTO getLatestLogFromGroup(List<SalaryLogJdbcDTO> logs) {
        return logs.stream()
                .max(Comparator.comparing(SalaryLogJdbcDTO::getC_date))
                .orElse(null);
    }

    private String convertDateTimeFormat(String localDate) {

        DateTimeFormatter inputFormatter = DateTimeFormatter.ofPattern("yyyy/MM/dd HH:mm");

        DateTimeFormatter outputFormatter = DateTimeFormatter.ofPattern(DATE_FORMAT);

        LocalDateTime parse = LocalDateTime.parse(localDate, inputFormatter);

        return parse.format(outputFormatter);
    }

    private SortedMap<String, Integer> generateDailyCostsMap(String startDateStr, String endDateStr, Integer dailyCost) {

        SortedMap<String, Integer> dailySalaryCosts = new TreeMap<>();

        DateTimeFormatter formatter = DateTimeFormatter.ofPattern(DATE_FORMAT);

        LocalDate startDate = LocalDate.parse(startDateStr, formatter);
        LocalDate endDate = LocalDate.parse(endDateStr, formatter);

        for (LocalDate date = startDate; !date.isAfter(endDate); date = date.plusDays(1)) {
            dailySalaryCosts.put(date.format(formatter), dailyCost);
        }

        return dailySalaryCosts;
    }

    private SortedMap<String, List<Integer>> generateDailyCostsCandleStick(LocalDate versionStartDate, LocalDate versionEndDate) {
        SortedMap<String, List<Integer>> candleStick = new TreeMap<>();
        for (LocalDate date = versionStartDate; !date.isAfter(versionEndDate); date = date.plusDays(1)) {
            candleStick.put(date.toString(), Arrays.asList(0, 0, 0, 0));
        }

        return candleStick;
    }

    @Override
    public Set<String> getAssignees(Long pdServiceLink, List<Long> pdServiceVersionLinks) {
        PdServiceAndIsReqDTO pdServiceAndIsReqDTO = PdServiceAndIsReqDTO.builder()
                .pdServiceLink(pdServiceLink)
                .pdServiceVersionLinks(pdServiceVersionLinks)
                .build();
        CostDTO costDTO = new CostDTO(pdServiceAndIsReqDTO);
        ResponseEntity<List<CostVO>> response = aggregationService.aggregationByAssigneeAccountId(costDTO);
        List<CostVO> body = response.getBody();
        if (body == null) {
            return Collections.emptySet();
        }
        return body.stream().map(CostVO::getAssigneeKey).collect(Collectors.toSet());
    }

    // 완료상태 조회(완료키워드)
    private Set<String> getCompleteKeyword() throws Exception {
        Set<String> completeStateKeywordSet = new HashSet<>();
        Map<Long, ReqStateEntity> completeStateEntityMap = reqStateService.완료상태조회();
        completeStateEntityMap.forEach((key, value) -> {
            completeStateKeywordSet.add(value.getC_title());
        });

        return completeStateKeywordSet;
    }

    @Override
    public List<AssigneeTimeDiffVO> getEstimatedPeriodCostByAccountId(CostDTO costDTO) throws Exception {

        Map<String, SalaryEntity> salaryEntityMap = salaryService.getAllSalariesMap();

        if (salaryEntityMap.isEmpty()) {
            log.info(" [ CostServiceImpl :: getEstimatedPeriodCostByAccountId ] :: 디비에서 연봉 정보를 조회하는 데 실패했습니다.");
            chat.sendMessageByEngine("등록 된 연봉 정보가 없습니다.");
        }

        List<AssigneeTimeDiffVO> returnList = aggregationService.calculateWorkdayByAccountId(costDTO);
        if (returnList.isEmpty()) {
            log.info(" [ CostServiceImpl :: getEstimatedPeriodCostByAccountId ] :: calculateWorkdayByAccountId.list is empty");
            return Collections.emptyList();
        }

        for (AssigneeTimeDiffVO vo : returnList) {
            String accountId = vo.getAccountId();
            vo.setSalary(
                    Optional.ofNullable(salaryEntityMap.get(accountId))
                            .map(SalaryEntity::getC_annual_income)
                            .orElse("0")
            );
            double salaryDoubleType = 0.0;
            double dailyAverageCost = 0.0;
            double hourAverageCost = 0.0;
            double costForDaysPeriod = 0.0;
            double costForHoursPeriod = 0.0;
            if (!vo.getSalary().equals("0")) {
                salaryDoubleType = Double.parseDouble(extractNumbers(vo.getSalary()));
                dailyAverageCost = calculateDailyAverageCost(salaryDoubleType);
                hourAverageCost = calculateHourAverageCost(salaryDoubleType);

                if (vo.getDaysDiff() != null && vo.getDaysDiff() > 0) {
                    costForDaysPeriod = vo.getDaysDiff() * dailyAverageCost;
                }
                if (vo.getHoursDiff() != null && vo.getHoursDiff() > 0) {
                    costForHoursPeriod = vo.getHoursDiff() * hourAverageCost;
                }
            } else {
                if (vo.getDaysDiff() != null && vo.getDaysDiff() > 0) {
                    costForDaysPeriod = vo.getDaysDiff() * (-1.0);
                }
                if (vo.getHoursDiff() != null && vo.getHoursDiff() > 0) {
                    costForHoursPeriod = vo.getHoursDiff() * (-1.0);
                }
            }
            vo.setEstimatedCostForDaysPeriod(formatToTwoDecimalPlacesString(costForDaysPeriod));
            vo.setEstimatedCostForHoursPeriod(formatToTwoDecimalPlacesString(costForHoursPeriod));
        }

        return returnList;
    }

    private String formatToTwoDecimalPlacesString(double value) {
        // "#.00"은 정수부는 필요에 따라 표시하고, 소수점 이하 두 자리는 항상 표시 (0이라도).
        // "##.##"은 소수점 이하 0은 표시하지 않음 (예: 10.50 -> 10.5)
        DecimalFormat df = new DecimalFormat("#.##");
        return df.format(value);
    }

    private String extractNumbers(String input) {
        if (input == null || input.isEmpty()) {
            return ""; // Or throw an IllegalArgumentException, depending on your needs
        }
        // Regular expression to keep only digits (0-9)
        // \\D matches any non-digit character
        // "" replaces all non-digit characters with an empty string
        return input.replaceAll("\\D", "");
    }

    /**
     * 일일 평균 비용을 계산합니다.
     * @param totalCost 총 비용 (double)
     * @return 일일 평균 비용
     */
    private double calculateDailyAverageCost(double totalCost) {
        if (totalCost < 0) {
            // 비용이 음수일 경우에 대한 처리 (예: 0 반환 또는 예외 발생)
            return 0.0;
        }
        return totalCost / 365.0;
    }

    /**
     * 시간당 평균 비용을 계산합니다.
     * (1년 = 365일 * 24시간/일)
     * @param totalCost 총 비용 (double)
     * @return 시간당 평균 비용
     */
    private double calculateHourAverageCost(double totalCost) {
        if (totalCost < 0) {
            // 비용이 음수일 경우에 대한 처리
            return 0.0;
        }
        // 365일 * 24시간 = 8760H
        final double MINUTES_IN_A_YEAR = 365.0 * 24.0;
        return totalCost / MINUTES_IN_A_YEAR;
    }

    /**
     * 분당 평균 비용을 계산합니다.
     * (1년 = 365일 * 24시간/일 * 60분/시간)
     * @param totalCost 총 비용 (double)
     * @return 분당 평균 비용
     */
    private double calculateMinuteAverageCost(double totalCost) {
        if (totalCost < 0) {
            // 비용이 음수일 경우에 대한 처리
            return 0.0;
        }
        // 365일 * 24시간 * 60분 = 525600분
        final double MINUTES_IN_A_YEAR = 365.0 * 24.0 * 60.0;
        return totalCost / MINUTES_IN_A_YEAR;
    }

    @Override
    public CostCalculationVO calculateCostAll(CostDTO costDTO) throws Exception {

        Map<Long, PdServiceVersionCostVO> versionMap = this.buildVersionMap(costDTO);
        Set<String> completeKeywords = this.getCompleteKeyword();

        Long pdServiceId = costDTO.pdServiceLink();
        Map<String, Map<Long, List<LinkedJiraIssueVO.RequirementData>>> linkedJiraIssueData = this.getLinkedJiraIssues(costDTO, pdServiceId);

        VersionRequirementAssigneeVO versionRequirementAssigneeVO = this.getVersionRequirementAssignee(costDTO);
        Map<String, Map<String, Map<String, AssigneeSalaryVO>>> versionRequirementAssignee = versionRequirementAssigneeVO.getVersionRequirementAssignee();
        Map<String, AssigneeSalaryVO> allAssignees = versionRequirementAssigneeVO.getAllAssignees();

        RequirementDifficultyAndPriorityVO requirementDifficultyAndPriorityVO = this.getRequirementListAndDifficultyAndPriority(costDTO, pdServiceId);
        List<ReqAddCostVO> requirements = requirementDifficultyAndPriorityVO.getRequirement();

        requirements.forEach(reqAddCostVO ->
                processCalculateCost(
                    reqAddCostVO, versionMap, linkedJiraIssueData, versionRequirementAssignee, completeKeywords
                )
        );

        List<AssigneeTimeDiffVO> estimatedPeriodCostByAccountId = this.getEstimatedPeriodCostByAccountId(costDTO);
        this.updateEstimatedPeriodCostByAccountId(estimatedPeriodCostByAccountId, allAssignees);

        return CostCalculationVO.builder()
                .versionCostMap(versionMap)
                .requirement(requirements)
                .difficulty(requirementDifficultyAndPriorityVO.getDifficulty())
                .priority(requirementDifficultyAndPriorityVO.getPriority())
                .versionRequirementAssignee(versionRequirementAssignee)
                .assigneeTimeDiffVOs(estimatedPeriodCostByAccountId)
                .build();
    }

    private void processCalculateCost(
            ReqAddCostVO reqAddCostVO,
            Map<Long, PdServiceVersionCostVO> versionMap,
            Map<String, Map<Long, List<LinkedJiraIssueVO.RequirementData>>> linkedJiraIssueData,
            Map<String, Map<String, Map<String, AssigneeSalaryVO>>> versionRequirementAssignee,
            Set<String> completeKeywords
    ) {
        extractReqVersionList(reqAddCostVO).forEach(versionId -> {

            PdServiceVersionCostVO versionCostVO = versionMap.get(versionId);
            if (versionCostVO == null) return;

            Map<Long, List<LinkedJiraIssueVO.RequirementData>> versionRequirementMap = linkedJiraIssueData.getOrDefault(versionId.toString(), Collections.emptyMap());
            List<LinkedJiraIssueVO.RequirementData> requirementDataList = versionRequirementMap.getOrDefault(reqAddCostVO.getC_id(), Collections.emptyList());

            Map<String, Map<String, AssigneeSalaryVO>> requirementAssigneeMap = versionRequirementAssignee.getOrDefault(versionId.toString(), Collections.emptyMap());

            requirementDataList.forEach(requirementData -> {
                Map<String, AssigneeSalaryVO> assigneeSalaryVOMap = requirementAssigneeMap.getOrDefault(requirementData.getC_issue_key(), Collections.emptyMap());
                assigneeSalaryVOMap.forEach((assigneeKey, salaryVO) -> {
                    try {
                        LocalDate[] dates = getValidStartEndDate(reqAddCostVO, completeKeywords);
                        if (dates == null) return;

                        LocalDate startDate = dates[0];
                        LocalDate endDate = dates[1];

                        long cost = calculateCostByAssignee(startDate, endDate, salaryVO.getSalary());

                        salaryVO.setCompletePerformance(salaryVO.getCompletePerformance() + cost);
                        salaryVO.setConsumedCostByVersionAssignee(salaryVO.getConsumedCostByVersionAssignee() + cost);
                        reqAddCostVO.setReqAmount(reqAddCostVO.getReqAmount() + cost);
                    }
                    catch (Exception e) {
                        log.warn("비용 계산 중 오류 발생 - assigneeKey: {}, error: {}", assigneeKey, e.getMessage());
                    }
                });
            });
        });
    }

    private void updateEstimatedPeriodCostByAccountId(
            List<AssigneeTimeDiffVO> estimatedPeriodCostByAccountId,
            Map<String, AssigneeSalaryVO> allAssignees
    ) {

        estimatedPeriodCostByAccountId.forEach(assigneeTimeDiffVO -> {
            AssigneeSalaryVO salaryVO = allAssignees.get(assigneeTimeDiffVO.getAccountId());
            assigneeTimeDiffVO.setEstimatedCostForDaysPeriod(String.valueOf(salaryVO.getCompletePerformance()));
        });
    }

    private List<Long> extractReqVersionList(ReqAddCostVO req) {
        return Optional.ofNullable(req.getC_req_pdservice_versionset_link())
                .map(VersionUtil::stringToLongArray)
                .map(Arrays::asList)
                .orElse(Collections.emptyList());
    }

    private LocalDate[] getValidStartEndDate(ReqAddCostVO reqAddCostVO, Set<String> completeKeywords) {
        LocalDate startDate = convertToLocalDate(reqAddCostVO.getC_req_start_date());
        LocalDate endDate = reqAddCostVO.getC_req_end_date() != null
                ? convertToLocalDate(reqAddCostVO.getC_req_end_date())
                : LocalDate.now();

        if (startDate == null || endDate == null || startDate.isAfter(endDate)) {
            return null;
        }

        if (reqAddCostVO.getC_req_state_title() != null && !completeKeywords.contains(reqAddCostVO.getC_req_state_title())) {
            endDate = LocalDate.now();
        }

        return new LocalDate[]{startDate, endDate};
    }

    private Map<Long, PdServiceVersionCostVO> buildVersionMap(CostDTO costDTO) throws Exception {
        List<PdServiceVersionEntity> pdServiceVersionEntities = pdServiceVersion.getVersionListByCids(costDTO.pdServiceVersionLinks());
        return pdServiceVersionEntities.stream()
                .collect(Collectors.toMap(
                        PdServiceVersionEntity::getC_id,
                        entity -> PdServiceVersionCostVO.builder()
                                .id(entity.getC_id())
                                .title(entity.getC_title())
                                .build()
                ));
    }

    private Map<String, Map<Long, List<LinkedJiraIssueVO.RequirementData>>> getLinkedJiraIssues(CostDTO costDTO, Long pdServiceId) throws Exception {

        SessionUtil.setAttribute("cost-calculation", "T_ARMS_REQSTATUS_" + pdServiceId);
        LinkedJiraIssueVO linkedJiraIssueVO = this.getLinkedJiraIssuesByVersionAndRequirement(costDTO);
        SessionUtil.removeAttribute("cost-calculation");

        return linkedJiraIssueVO.getLinkedJiraIssueData();
    }

    private RequirementDifficultyAndPriorityVO getRequirementListAndDifficultyAndPriority(CostDTO costDTO, Long pdServiceId) throws Exception {
        SessionUtil.setAttribute("cost-calculation", "T_ARMS_REQADD_" + pdServiceId);
        RequirementDifficultyAndPriorityVO requirementListStats = this.getRequirementListStats(costDTO);
        SessionUtil.removeAttribute("cost-calculation");
        return requirementListStats;
    }

    private LocalDate convertToLocalDate(Date date) {
        if (date == null) return null;
        return Instant.ofEpochMilli(date.getTime()).atZone(ZoneId.systemDefault()).toLocalDate();
    }

    private long calculateCostByAssignee(LocalDate startDate, LocalDate endDate, long salary) {

        // TODO: 시작일과 종료일 사이의 일수 계산하면서 연휴를 뺄 수 있도록 처리 필요

        long days = ChronoUnit.DAYS.between(startDate, endDate) + 1;
        long dailyPay = Math.round((double) salary / 365);
        return days * dailyPay;
    }

    private VersionRequirementAssigneeVO getVersionRequirementAssignee(CostDTO costDTO) throws Exception {

        Map<String, SalaryEntity> salaryEntityMap = salaryService.getAllSalariesMap();

        if (salaryEntityMap.isEmpty()) {
            log.info(" [ " + this.getClass().getName() + " :: 버전별_요구사항별_담당자가져오기 ] :: 디비에서 연봉 정보를 조회하는 데 실패했습니다.");
            chat.sendMessageByEngine("등록 된 연봉 정보가 없습니다.");
            return new VersionRequirementAssigneeVO(new HashMap<>(), new HashMap<>());
        }

        PdServiceAndIsReqDTO pdServiceAndIsReq = costDTO.getPdServiceAndIsReq();
        pdServiceAndIsReq.setIsReq(true);
        ResponseEntity<List<CostVO>> reqReqList = aggregationService.getAssigneeListByProductVersionAndRequirement(costDTO);

        pdServiceAndIsReq.setIsReq(false);
        ResponseEntity<List<CostVO>> subReqList = aggregationService.getAssigneeListByProductVersionAndRequirement(costDTO);

        List<CostVO> allResult = new ArrayList<>();
        allResult.addAll(reqReqList.getBody());
        allResult.addAll(subReqList.getBody());

        Map<String, Map<String, Map<String, AssigneeSalaryVO>>> versionRequirementAssignee = new HashMap<>();

        Map<String, AssigneeSalaryVO> allAssigneeMap = new HashMap<>();

        for (CostVO costVO : allResult) {
            String assigneeAccountId = costVO.getAssigneeKey();
            String requirementId = costVO.getReqKey();
            String subReqId = costVO.getSubReqKey();
            String versionId = costVO.getVersionKey();
            String assigneeDisplayName = costVO.getDisplayNameKey();

            Long salary = Optional.ofNullable(salaryEntityMap)
                    .flatMap(map -> Optional.ofNullable(map.get(assigneeAccountId)))
                    .map(SalaryEntity::getC_annual_income)
                    .filter(s -> !s.trim().isEmpty())
                    .map(NumberUtils::toLong)
                    .orElse(0L);

            AssigneeSalaryVO assigneeSalaryVO = AssigneeSalaryVO.builder()
                    .id(assigneeAccountId)
                    .name(assigneeDisplayName)
                    .salary(salary)
                    .consumedCostByVersionAssignee(0L)
                    .completePerformance(0L)
                    .build();

            allAssigneeMap.put(assigneeAccountId, assigneeSalaryVO);

            if (requirementId != null && !requirementId.trim().isEmpty()) {
                versionRequirementAssignee
                        .computeIfAbsent(versionId, k -> new HashMap<>())
                        .computeIfAbsent(requirementId, k -> new HashMap<>())
                        .put(assigneeAccountId, assigneeSalaryVO);
            }

            if (subReqId != null && !subReqId.trim().isEmpty()) {
                versionRequirementAssignee
                        .computeIfAbsent(versionId, k -> new HashMap<>())
                        .computeIfAbsent(subReqId, k -> new HashMap<>())
                        .put(assigneeAccountId, assigneeSalaryVO);
            }
        }

        return new VersionRequirementAssigneeVO(versionRequirementAssignee, allAssigneeMap);
    }

    @Override
    public CostCalculationVO calculateVersionCost(CostDTO costDTO) throws Exception {

        CostCalculationVO costCalculationVO = this.calculateCostAll(costDTO);

        return CostCalculationVO.builder()
                .versionCostMap(costCalculationVO.getVersionCostMap())
                .versionRequirementAssignee(costCalculationVO.getVersionRequirementAssignee())
                .build();

    }

    @Override
    public CostCalculationVO calculateRequirementCost(CostDTO costDTO) throws Exception {

        CostCalculationVO costCalculationVO = this.calculateCostAll(costDTO);

        return CostCalculationVO.builder()
                .requirement(costCalculationVO.getRequirement())
                .difficulty(costCalculationVO.getDifficulty())
                .priority(costCalculationVO.getPriority())
                .build();
    }

    @Override
    public CostCalculationVO calculateAssigneeCost(CostDTO costDTO) throws Exception {

        CostCalculationVO costCalculationVO = this.calculateCostAll(costDTO);

        return CostCalculationVO.builder()
                .assigneeTimeDiffVOs(costCalculationVO.getAssigneeTimeDiffVOs())
                .build();

    }

    @Override
    public CostCalculationVO toBeCalculateCost(CalculationCostDTO calculationCostDTO) throws Exception {

        Map<String, SalaryEntity> allSalariesMap = salaryService.getAllSalariesMap();

        if (allSalariesMap.isEmpty()) {
            log.info(" [ " + this.getClass().getName() + " :: toBeCalculateCost ] :: 디비에서 연봉 정보를 조회하는 데 실패했습니다.");
            chat.sendMessageByEngine("등록 된 연봉 정보가 없습니다.");
            return null;
        }

        Map<String, AssigneeSalaryDTO> salaryDTOMap = allSalariesMap.values()
                .stream()
                .map(salaryEntity -> {
                    AssigneeSalaryDTO assigneeSalaryDTO = new AssigneeSalaryDTO();
                    assigneeSalaryDTO.setAssigneeId(salaryEntity.getC_key());
                    assigneeSalaryDTO.setName(salaryEntity.getC_name());
                    assigneeSalaryDTO.setSalary(salaryEntity.getC_annual_income() != null ? Long.parseLong(salaryEntity.getC_annual_income()) : 0L);

                    return assigneeSalaryDTO;
                })
                .collect(Collectors.toMap(
                    AssigneeSalaryDTO::getAssigneeId,
                    Function.identity()
                ));

        calculationCostDTO.setSalaryDTO(salaryDTOMap);

        ResponseEntity<?> response = engineService.calculationCost(calculationCostDTO);

        return null;
    }

}