package com.arms.api.report.export_service.config;

import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.io.Resource;
import org.springframework.core.io.ResourceLoader;

import javax.annotation.PostConstruct;
import java.util.HashMap;
import java.util.Map;

/**
 * 리포트 내보내기 관련 설정을 관리하는 Configuration 클래스
 *
 * <p>application.yml의 'report' 프리픽스 아래 설정값들을 바인딩합니다.
 *
 * <p>설정 예시:
 * <pre>
 * report:
 *   template-path: classpath:templates/report/
 *   font-path: classpath:fonts/
 *   default-font: NanumGothic.ttf
 * </pre>
 *
 * @author HS.Yang
 * @since 2025.12.01
 */
@Slf4j
@Configuration
@ConfigurationProperties(prefix = "report")
@Getter
@Setter
@RequiredArgsConstructor
public class ReportConfig {

    /**
     * Spring의 ResourceLoader - 리소스 파일 로딩에 사용
     */
    private final ResourceLoader resourceLoader;

    // ========================================
    // 기본 경로 설정
    // ========================================

    /**
     * 템플릿 파일이 저장된 경로
     * 기본값: classpath:templates/report/
     */
    private String templatePath = "classpath:templates/report/";

    /**
     * 폰트 파일이 저장된 경로
     * 기본값: classpath:fonts/
     */
    private String fontPath = "classpath:fonts/";

    /**
     * 기본 폰트 파일명 (한글 지원 폰트)
     */
    private String defaultFont = "NanumGothic.ttf";

    /**
     * 굵은 폰트 파일명
     */
    private String boldFont = "NanumGothicBold.ttf";

    /**
     * 리포트에 표시될 회사명
     */
    private String companyName = "313DEVGRP";

    /**
     * 회사 로고 이미지 경로
     */
    private String companyLogo = "classpath:static/313_logo_large_drop.png";

    // ========================================
    // PPT 관련 설정
    // ========================================

    /**
     * PPT 생성 관련 상세 설정
     */
    private PptConfig ppt = new PptConfig();

    // ========================================
    // PDF 관련 설정
    // ========================================

    /**
     * PDF 생성 관련 상세 설정
     */
    private PdfConfig pdf = new PdfConfig();

    /**
     * PPT 설정을 담는 내부 클래스
     */
    @Getter
    @Setter
    public static class PptConfig {
        /** 기본 템플릿 파일명 */
        private String defaultTemplate = "default-template.pptx";

        /** 제목 폰트 크기 (pt) */
        private int titleFontSize = 32;

        /** 본문 폰트 크기 (pt) */
        private int bodyFontSize = 14;

        /** 테이블 폰트 크기 (pt) */
        private int tableFontSize = 11;
    }

    /**
     * PDF 설정을 담는 내부 클래스
     */
    @Getter
    @Setter
    public static class PdfConfig {
        /** 상단 여백 (pt) */
        private float marginTop = 50f;

        /** 하단 여백 (pt) */
        private float marginBottom = 50f;

        /** 좌측 여백 (pt) */
        private float marginLeft = 50f;

        /** 우측 여백 (pt) */
        private float marginRight = 50f;

        /** 제목 폰트 크기 (pt) */
        private int titleFontSize = 20;

        /** 헤더 폰트 크기 (pt) */
        private int headerFontSize = 14;

        /** 본문 폰트 크기 (pt) */
        private int bodyFontSize = 11;
    }

    // ========================================
    // 템플릿 매핑
    // ========================================

    /**
     * 템플릿 이름과 파일명 매핑
     * key: 템플릿 논리명, value: 실제 파일명
     */
    private Map<String, String> templates = new HashMap<>();

    /**
     * Bean 초기화 후 기본 템플릿 등록
     *
     * <p>application.yml에 템플릿 매핑이 없으면 기본값 사용
     */
    @PostConstruct
    public void init() {
        log.info("Initializing ReportConfig...");

        // 기본 템플릿 등록 (설정이 없는 경우)
        if (templates.isEmpty()) {
            templates.put("default", "default-template.pptx");
            templates.put("analysis", "analysis-template.pptx");
            templates.put("summary", "summary-template.pptx");
            log.info("Default templates registered: {}", templates.keySet());
        }

        log.info("ReportConfig initialized - templatePath: {}, fontPath: {}",
                templatePath, fontPath);
    }

    /**
     * 템플릿 이름으로 Resource 객체 반환
     *
     * @param templateName 템플릿 논리명 (예: "default", "analysis")
     * @return 템플릿 파일의 Resource 객체
     */
    public Resource getTemplateResource(String templateName) {
        // 매핑된 파일명 조회, 없으면 기본 템플릿 사용
        String templateFile = templates.getOrDefault(templateName, ppt.getDefaultTemplate());
        String fullPath = templatePath + templateFile;

        log.debug("Loading template: {} -> {}", templateName, fullPath);
        return resourceLoader.getResource(fullPath);
    }

    /**
     * 폰트 이름으로 Resource 객체 반환
     *
     * @param fontName 폰트 파일명
     * @return 폰트 파일의 Resource 객체
     */
    public Resource getFontResource(String fontName) {
        return resourceLoader.getResource(fontPath + fontName);
    }

    /**
     * 기본 폰트의 Resource 객체 반환
     *
     * @return 기본 폰트 파일의 Resource 객체
     */
    public Resource getDefaultFontResource() {
        return resourceLoader.getResource(fontPath + defaultFont);
    }

    /**
     * 굵은 폰트의 Resource 객체 반환
     *
     * @return 굵은 폰트 파일의 Resource 객체
     */
    public Resource getBoldFontResource() {
        return resourceLoader.getResource(fontPath + boldFont);
    }
}
