package com.arms.api.issue.almapi.strategy;

import com.arms.api.issue.almapi.model.dto.*;
import com.arms.api.issue.almapi.model.entity.AlmIssueEntity;
import com.arms.api.issue.almapi.model.vo.*;
import com.arms.api.issue.priority.model.IssuePriorityDTO;
import com.arms.api.issue.status.dto.IssueStatusDTO;
import com.arms.api.issue.type.dto.IssueTypeDTO;
import com.arms.api.project.dto.ProjectDTO;
import com.arms.api.serverinfo.model.ServerInfo;
import com.arms.api.serverinfo.service.ServerInfoService;
import com.arms.api.util.LRUMap;
import com.arms.api.util.alm.RedmineApi;
import com.arms.api.util.alm.RedmineUtil;
import com.arms.api.util.errors.ErrorCode;
import com.arms.api.util.errors.ErrorLogUtil;
import com.arms.api.issue.almapi.service.CategoryMappingService;
import com.arms.egovframework.javaservice.esframework.esquery.SimpleQuery;
import com.arms.egovframework.javaservice.esframework.repository.common.EsCommonRepositoryWrapper;
import com.taskadapter.redmineapi.*;
import com.taskadapter.redmineapi.bean.*;

import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.elasticsearch.core.SearchHit;
import org.springframework.stereotype.Component;

import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.*;


@Slf4j
@Component("REDMINE_ON_PREMISS")
@AllArgsConstructor
public class OnPremiseRedmineIssueStrategy implements IssueStrategy {

    private final RedmineUtil redmineUtil;

    private final RedmineApi redmineApi;

    private final CategoryMappingService categoryMappingService;

    private final EsCommonRepositoryWrapper<AlmIssueEntity> esCommonRepositoryWrapper;

    private final ServerInfoService serverInfoService;

    private final IssueSaveTemplate issueSaveTemplate;

    @Override
    public AlmIssueVO createIssue(AlmIssueDTO almIssueDTO) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueDTO.getServerId());

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(serverInfo.getUri(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        IssueCreationFieldsDTO fieldsDTO = almIssueDTO.getFields();
        if (fieldsDTO == null) {
            String errorMessage = String.format("%s[%s] :: 이슈 생성 필드 데이터가 존재 하지 않습니다.",
                serverInfo.getType(), serverInfo.getUri());
            log.error(errorMessage);
            throw new IllegalArgumentException(ErrorCode.REQUEST_BODY_ERROR_CHECK.getErrorMsg() + " :: " + errorMessage);
        }

        String projectKeyOrId = null;
        String issueTypeId = null;

        if (fieldsDTO.getProject()!= null && StringUtils.isNotEmpty(fieldsDTO.getProject().getId())) {
            projectKeyOrId = fieldsDTO.getProject().getId();
        }

        if (fieldsDTO.getIssuetype() != null
                && StringUtils.isNotEmpty(fieldsDTO.getIssuetype().getId())) {
            issueTypeId = fieldsDTO.getIssuetype().getId();
        }

        if (StringUtils.isEmpty(projectKeyOrId) || StringUtils.isEmpty(issueTypeId)) {
            String errorMessage = String.format("%s :: %s 이슈 생성 필드 확인에 필요한 프로젝트 아이디, 이슈유형 아이디가 존재하지 않습니다.",
                serverInfo.getType(), serverInfo.getUri());
            throw new IllegalArgumentException(errorMessage);
        }

        Tracker issueType = new Tracker()
                .setId(Integer.parseInt(issueTypeId));

        Issue issue = new Issue(redmineManager.getTransport(), Integer.parseInt(projectKeyOrId))
                .setTracker(issueType);

        if (StringUtils.isNotEmpty(fieldsDTO.getSummary())) {
            issue.setSubject(fieldsDTO.getSummary());
        }

        if (StringUtils.isNotEmpty(fieldsDTO.getDescription())) {
            issue.setDescription(fieldsDTO.getDescription());
        }

        if (fieldsDTO.getPriority() != null && StringUtils.isNotEmpty(fieldsDTO.getPriority().getId())) {
            issue.setPriorityId(Integer.parseInt(fieldsDTO.getPriority().getId()));
        }

        if (fieldsDTO.getStartDate() != null) {
            issue.setStartDate(fieldsDTO.getStartDate());
        }

        if (fieldsDTO.getDueDate() != null) {
            issue.setDueDate(fieldsDTO.getDueDate());
        }

        try {
            issue = issue.create();
        }
        catch (RedmineException e) {
            String errorMessage = ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                String.format("%s[%s],  프로젝트[%s], 이슈유형[%s], 생성 필드 :: 이슈 생성하기 중 오류",
                    serverInfo.getType(), serverInfo.getUri(), projectKeyOrId, issueTypeId, issue.toString()));
            throw new IllegalArgumentException(ErrorCode.ISSUE_CREATION_ERROR.getErrorMsg() + " :: " + errorMessage);
        }

        User user = userList(new HashSet<>(), issue.getAssigneeId(), redmineManager);

        OnPremissRedmineIssueVO onPremissRedmineIssueVO = OnPremissRedmineIssueVO.builder()
                .issue(issue)
                .user(user)
                .serverInfo(serverInfo).build();

        return convertOnRedmineIssueToAlmIssue(onPremissRedmineIssueVO);
    }

    @Override
    public Map<String, Object> updateIssue(AlmIssueDTO almIssueDTO) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueDTO.getServerId());

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(serverInfo.getUri(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        Map<String, Object> result = new HashMap<>();
        IssueCreationFieldsDTO fieldsDTO = almIssueDTO.getFields();
        String summary = fieldsDTO.getSummary();
        String description = fieldsDTO.getDescription();
        IssueStatusDTO issueStatusDTO = fieldsDTO.getStatus();

        try {
            Issue updatedIssue = redmineManager.getIssueManager().getIssueById(Integer.parseInt(almIssueDTO.getIssueKeyOrId()));

            if (summary != null && !summary.isEmpty()) {
                updatedIssue.setSubject(summary);
            }

            if (description != null) {
                updatedIssue.setDescription(description);
            }

            updatedIssue.update();
            if (issueStatusDTO != null && !StringUtils.isEmpty(issueStatusDTO.getId())) {
                this.updateIssueStatus(almIssueDTO);
            }

            result.put("success", true);
            result.put("message", "이슈 수정 성공");
        }
        catch (RedmineException e) {
            String errorMessage = ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                "레드마인_온프레미스 ["+ serverInfo.getUri() +"] :: 이슈_키_또는_아이디 :: "
                    + almIssueDTO.getIssueKeyOrId()+ " :: 수정데이터 :: " + fieldsDTO.toString() + "이슈_수정하기 오류");
            log.error(errorMessage);

            result.put("success", false);
            result.put("message", errorMessage);
        }

        return result;
    }

    @Override
    public Map<String, Object> updateIssueStatus(AlmIssueDTO almIssueDTO) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueDTO.getServerId());

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(serverInfo.getUri(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        Map<String, Object> result = new HashMap<>();
        String issueKeyOrId = almIssueDTO.getIssueKeyOrId();
        String statusId = almIssueDTO.getStatusId();

        try {

            Issue updatedIssue = redmineManager.getIssueManager().getIssueById(Integer.parseInt(issueKeyOrId));

            updatedIssue.setStatusId(Integer.parseInt(statusId));
            updatedIssue.update();

            boolean checkStatusChange = verifyIssueStatus(almIssueDTO);

            if (checkStatusChange) {
                result.put("success", true);
                result.put("message", "이슈 상태 변경 성공");
            }
            else {
                String errorMessage = "레드마인_온프레미스 [" + serverInfo.getUri() + "] :: 이슈 키[" + issueKeyOrId + "] :: 상태 아이디[" + statusId + "] :: 해당 업무 흐름으로 변경이 불가능 합니다.";
                log.error(errorMessage);

                result.put("success", false);
                result.put("message", "변경할 이슈 상태가 존재하지 않습니다.");
            }

        }
        catch (RedmineException e) {
            String errorMessage = ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                "레드마인_온프레미스 [" + serverInfo.getUri() + "] :: 이슈_키_또는_아이디 :: "
                    + issueKeyOrId + " :: 수정데이터 :: " + statusId + "이슈_상태_변경하기 오류");
            log.error(errorMessage);

            result.put("success", false);
            result.put("message", errorMessage);
        }

        return result;
    }

    public boolean verifyIssueStatus(AlmIssueDTO almIssueDTO) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueDTO.getServerId());

        String issueKeyOrId = almIssueDTO.getIssueKeyOrId();

        String statusId = almIssueDTO.getStatusId();

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(serverInfo.getUri(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        try {
            Issue updatedIssue = redmineManager.getIssueManager().getIssueById(Integer.parseInt(issueKeyOrId));

            return Optional.ofNullable(updatedIssue)
                .map(Issue::getStatusId)
                .map(currentStatus -> StringUtils.equals(statusId, String.valueOf(currentStatus)))
                .orElse(false);
        }
        catch (RedmineException e) {
            ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                "레드마인_온프레미스 ["+ serverInfo.getUri() +"] :: 이슈_키_또는_아이디 :: "
                    + issueKeyOrId + " :: 이슈_상태_검증하기 오류");
            return false;
        }
    }

    @Override
    public Map<String, Object> deleteIssue(AlmIssueDTO almIssueDTO) {

        Map<String, Object> result = new HashMap<>();

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueDTO.getServerId());

        String issueKeyOrId = almIssueDTO.getIssueKeyOrId();

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(serverInfo.getUri(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        try {
            redmineManager.getIssueManager().deleteIssue(Integer.valueOf(issueKeyOrId));

            result.put("success", true);
            result.put("message", "이슈 삭제 성공");
        }
        catch (RedmineException e) {
            String errorMessage = ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                "레드마인_온프레미스 ["+ serverInfo.getUri() +"] :: 이슈_키_또는_아이디 :: "
                    + issueKeyOrId+ " :: 이슈_삭제하기 오류");
            log.error(errorMessage);

            result.put("success", false);
            result.put("message", errorMessage);
        }

        return result;
    }

    @Override
    public boolean isExistIssue(AlmIssueDTO almIssueDTO)  {
        try {
            AlmIssueVO issue = this.getIssueVO(almIssueDTO, false);
            return issue != null && issue.getKey().equals(almIssueDTO.getIssueKeyOrId());
        } catch (Exception e) {
            return false;
        }
    }

    @Override
    public AlmIssueVO getIssueVO(AlmIssueDTO almIssueDTO)  {
        return this.getIssueVO(almIssueDTO, true);
    }

    private AlmIssueVO getIssueVO(AlmIssueDTO almIssueDTO, boolean convertFlag) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueDTO.getServerId());

        String issueKeyOrId = almIssueDTO.getIssueKeyOrId();

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(serverInfo.getUri(), serverInfoService.getDecryptPasswordOrToken(serverInfo));

        try {
            Issue issue = redmineManager.getIssueManager().getIssueById(Integer.parseInt(issueKeyOrId));

            OnPremissRedmineIssueVO onPremissRedmineIssueVO = OnPremissRedmineIssueVO.builder()
                    .issue(issue)
                    .serverInfo(serverInfo).build();

            if(convertFlag){
                return convertOnRedmineIssueToAlmIssue(onPremissRedmineIssueVO);
            }

            return AlmIssueVO.builder().key(String.valueOf(issue.getId())).build();
        }
        catch (Exception e) {
            ErrorLogUtil.exceptionLogging(e, this.getClass().getName(),
                    "온프레미스 지라(" + serverInfo.getUri() + ") :: 이슈 키(" + issueKeyOrId + ") :: 이슈_상세정보_가져오기에 실패하였습니다.");
            throw new IllegalArgumentException("이슈_상세정보_가져오기에 실패하였습니다.");
        }
    }

    @Override
    public List<AlmIssueEntity> discoveryIssueAndGetReqEntities(AlmIssueIncrementDTO almIssueIncrementDTO) {

        Map<String,Issue> lruMap = new LRUMap<>(10000);

        return issueSaveTemplate.discoveryIncrementALmIssueAndGetReqAlmIssueEntities(
                almIssueIncrementDTO, (dto)-> this.discoveryIssueAndGetReqEntities(dto, lruMap));
    }

    private AlmIssueVOCollection discoveryIssueAndGetReqEntities(AlmIssueIncrementDTO almIssueIncrementDTO, Map<String,Issue> lruMap) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueIncrementDTO.getServerId());

        RedmineManager redmineManager = redmineUtil.createRedmineOnPremiseCommunicator(
                serverInfo.getUri()+"?v[updated_on][]="+almIssueIncrementDTO.getStartDate()
                , serverInfoService.getDecryptPasswordOrToken(serverInfo));

        try {

            List<Issue> issuesFromRedmine = fetchIssuesFromDate(almIssueIncrementDTO, redmineManager);

            issuesFromRedmine.forEach(a->{
                lruMap.put(String.valueOf(a.getId()),a);
            });

            List<OnPremissRedmineIssueVO> onPremissRedmineIssueVOS = issuesFromRedmine.stream()
                    .flatMap(cloudJiraIssueRawDataVO
                            -> incrementTraceArmsIssueVOS(lruMap, cloudJiraIssueRawDataVO, serverInfo, redmineManager).stream())
                    .toList();

            OnPremissRedmineVOCollection onPremissRedmineVOCollection = new OnPremissRedmineVOCollection(
                    onPremissRedmineIssueVOS
                            .stream()
                            .map(onPremissRedmineIssueVO->{
                                if(!issuesFromRedmine.contains(onPremissRedmineIssueVO.getIssue())){
                                    return onPremissRedmineIssueVO.markAsExcludedFromSave();
                                }else{
                                    return onPremissRedmineIssueVO;
                                }
                            }).toList());


            List<OnPremissRedmineIssueVO> onPremissRedmineIssueVOs = onPremissRedmineVOCollection.appliedLinkedIssuePdServiceVO();

            deleteUnrelatedEntities(onPremissRedmineIssueVOs, issuesFromRedmine);

            return this.convertOnRedmineIssueToAlmIssue(onPremissRedmineIssueVOs);

        } catch (Exception e) {
            log.error("증분 데이터 수집에 실패 하였습니다.",e);

            ErrorLogUtil.exceptionLoggingAndReturn(e, this.getClass().getName(),
                    "클라우드 지라(" + serverInfo.getUri() + ") 증분 이슈 수집 중 오류");

            return new AlmIssueVOCollection(new ArrayList<>());

        }
    }

    private void deleteUnrelatedEntities(List<OnPremissRedmineIssueVO> onPremissRedmineIssueVOs, List<Issue> results) {

        results
            .stream()
            .filter(a->!onPremissRedmineIssueVOs.contains(OnPremissRedmineIssueVO.builder().issue(a).build()))
            .forEach(a->{
                onPremissRedmineIssueVOs.stream().findFirst().ifPresent(b->{
                    AlmIssueEntity almIssueEntity = esCommonRepositoryWrapper.findRecentDocByRecentId(b.recentId(String.valueOf(a.getId())));
                    if(almIssueEntity.getId()!=null){
                        esCommonRepositoryWrapper.deleteRecentDocById(almIssueEntity.getId());
                    }
                });
             });
    }

    private AlmIssueVOCollection convertOnRedmineIssueToAlmIssue(List<OnPremissRedmineIssueVO> onPremissRedmineIssueVOs){
        return new AlmIssueVOCollection(onPremissRedmineIssueVOs
                    .stream()
                    .map(this::convertOnRedmineIssueToAlmIssue)
                    .toList());
    }

    private List<Issue> fetchIssuesFromDate(AlmIssueIncrementDTO almIssueIncrementDTO, RedmineManager redmineManager) {

        String endDate = almIssueIncrementDTO.getEndDate();

        String projectKey = almIssueIncrementDTO.getProjectKey();

        int page = 1;

        int maxResults = 25;

        try {

            boolean hasMore = true;

            List<Issue> issueResult = new ArrayList<>();

            while (hasMore) {

                Params params = new Params()
                        .add("f[]", "updated_on")
                        .add("v[updated_on][]", endDate)
                        .add("op[updated_on]", "><")
                        .add("page", String.valueOf(page))
                        .add("project_id", projectKey);

                List<Issue> results = redmineManager.getIssueManager()
                        .getIssues(params)
                        .getResults();

                List<Issue> resultsById = results.stream().map(a -> {
                    try {
                        return redmineManager.getIssueManager()
                                .getIssueById(a.getId(), Include.relations, Include.children, Include.journals);
                    } catch (RedmineException e) {
                        throw new RuntimeException(e);
                    }
                }).toList();

                issueResult.addAll(resultsById);

                modifyIssueAlmToEs(almIssueIncrementDTO, resultsById);

                int fetchCount = results.size();

                if (fetchCount < maxResults) {
                    hasMore = false;
                } else {
                    page++;
                }
            }

            return issueResult;

        } catch (RedmineException e) {
            throw new RuntimeException(e);
        }
    }

    private void modifyIssueAlmToEs(AlmIssueIncrementDTO almIssueIncrementDTO, List<Issue> results) {

        ServerInfo serverInfo = serverInfoService.verifyServerInfo(almIssueIncrementDTO.getServerId());


        for (Issue issue : results) {

            List<SearchHit<AlmIssueEntity>> hitDocs = esCommonRepositoryWrapper.findRecentHits(
                SimpleQuery
                    .termQueryFilter("jira_server_id", serverInfo.getConnectId())
                    .andTermQueryFilter("key", issue.getId())
            ).toHitDocs();

            List<AlmIssueEntity> almIssueRecentFalseList = new ArrayList<>();

            for (SearchHit<AlmIssueEntity> hitDoc : hitDocs) {

                AlmIssueEntity almIssueEntity = hitDoc.getContent();

                if (!almIssueEntity.getKey().equals(String.valueOf(issue.getId()))
                        || !almIssueEntity.getProject().getKey().equals(String.valueOf(issue.getProjectId()))
                ) {
                    almIssueEntity.setRecent(false);
                    esCommonRepositoryWrapper.modifyWithIndexName(almIssueEntity,hitDoc.getIndex());
                    almIssueRecentFalseList.add(almIssueEntity);
                }
            }

            almIssueRecentFalseList
                    .stream()
                    .filter(a -> a.getRecentId() != null)
                    .findFirst()
                    .ifPresent(almIssueEntity -> {
                        almIssueEntity.modifyProjectKeyAndKey(String.valueOf(issue.getProjectId()), String.valueOf(issue.getId()));
                        esCommonRepositoryWrapper.save(almIssueEntity);
                    });
        }
    }

    private List<OnPremissRedmineIssueVO> incrementTraceArmsIssueVOS(
                                 Map<String,Issue> lruMap, Issue currentIssue,
                                 ServerInfo serverInfo, RedmineManager redmineManager) {

        Queue<IssueDTO> queue = new LinkedList<>();

        queue.add(createIssueDTO(currentIssue));

        lruMap.put(String.valueOf(currentIssue.getId()),currentIssue);

        Set<OnPremissRedmineIssueVO> onPremissRedmineIssueVOSet = new HashSet<>();

        Set<String> visited = new HashSet<>();

        while (!queue.isEmpty()) {
            try {

                IssueDTO issueDTO = queue.poll();

                Issue issue = issueList(lruMap, issueDTO, redmineManager);

                if (issue.getId() != null) {
                    lruMap.put(String.valueOf(issue.getId()),issue);
                }

                User assigneeUser = null;
                Integer assigneeId = issue.getAssigneeId();
                if (assigneeId != null) {
                    assigneeUser = redmineManager.getUserManager().getUserById(assigneeId);
                }

                onPremissRedmineIssueVOSet.add(
                    OnPremissRedmineIssueVO.builder()
                            .issue(issue)
                            .user(assigneeUser)
                            .serverInfo(serverInfo).build()
                );

                if (!visited.contains(String.valueOf(issue.getId()))) {
                    queue.add(createIssueParentTraceDTO(issue));

                    List<Issue> childList = issue.getChildren().stream().toList();

                    for (Issue childIssue : childList) {
                        queue.add(this.createIssueDTO(childIssue));
                    }

                    processIssueRelations(issue, queue, visited);

                    visited.add(issueDTO.getKey());
                }


            } catch (RedmineException e) {
                throw new RuntimeException(e);
            }
        }

        return onPremissRedmineIssueVOSet
                .stream()
                .map(a -> {
                    AlmIssueEntity almIssueEntity = esCommonRepositoryWrapper.findRecentDocByRecentId(a.recentId());
                    if (almIssueEntity.izReqTrue()) {
                        return OnPremissRedmineIssueVO.builder()
                                .issue(currentIssue)
                                .almIssueWithRequirementDTO(new AlmIssueWithRequirementDTO(almIssueEntity))
                                .serverInfo(serverInfo)
                                .build();

                    }
                    return null;
                }).filter(Objects::nonNull).toList();
    }

    private void processIssueRelations(Issue issue, Queue<IssueDTO> queue, Set<String> visited) {

        List<IssueRelation> issueRelations = new ArrayList<>(issue.getRelations());

        for (IssueRelation relation : issueRelations) {
            String issueIdKey = String.valueOf(relation.getIssueId());
            if (!visited.contains(issueIdKey)) {
                queue.add(IssueDTO.builder()
                        .key(issueIdKey)
                        .fromKey(String.valueOf(relation.getIssueToId()))
                        .build());
            }
            String issueToIdKey = String.valueOf(relation.getIssueToId());
            if (!visited.contains(issueToIdKey)) {
                queue.add(IssueDTO.builder()
                        .key(issueToIdKey)
                        .fromKey(String.valueOf(relation.getIssueId()))
                        .build());
            }
        }
    }

    private IssueDTO createIssueDTO(Issue issue) {
        return IssueDTO.builder()
                .key(String.valueOf(issue.getId()))
                .fromKey(String.valueOf(issue.getId()))
                .build();
    }

    private IssueDTO createIssueParentTraceDTO(Issue issue) {
        return IssueDTO.builder()
                .key(String.valueOf(Optional.ofNullable(issue.getParentId()).orElseGet(issue::getId)))
                .fromKey(String.valueOf(issue.getId()))
                .build();
    }

    private Issue issueList(Map<String,Issue> lruMap, IssueDTO issueDTO, RedmineManager redmineManager) throws RedmineException {

        log.info("key cache:key=={},{}",issueDTO.getKey(),lruMap.get(issueDTO.getKey())!=null);

        Issue issue = lruMap.get(issueDTO.getKey());

        if(issue !=null){
            return issue;
        }else{
            return redmineManager.getIssueManager()
                    .getIssueById(Integer.parseInt(Objects.requireNonNull(issueDTO.getKey())), Include.relations, Include.children);
        }

    }

    private User userList(Set<User> userSet, Integer assigneeId, RedmineManager redmineManager) {

        Optional<User> optionalUser = userSet.stream()
                .filter(a -> a.getId().equals(assigneeId))
                .findFirst();

        if (optionalUser.isPresent()) {
            return optionalUser.get();
        } else {
            return Optional.ofNullable(assigneeId)
                    .map(a -> redmineUtil.getUserInfo(redmineManager, String.valueOf(a))).orElse(null);
        }

    }

    private AlmIssueVO convertOnRedmineIssueToAlmIssue(OnPremissRedmineIssueVO onPremissRedmineIssueVO) {

        ServerInfo serverInfo = onPremissRedmineIssueVO.getServerInfo();

        Issue issue = onPremissRedmineIssueVO.getIssue();

        String defaultPath = redmineUtil.checkServerInfoPath(serverInfo.getUri());

        AlmIssueVO.AlmIssueVOBuilder almIssueVOBuilder = AlmIssueVO.builder();

        almIssueVOBuilder.excludeFromSave(onPremissRedmineIssueVO.isExcludeFromSave());

        Optional.ofNullable(onPremissRedmineIssueVO.relationRecentIds())
                .ifPresent(almIssueVOBuilder::linkedIssue);

        almIssueVOBuilder.almIssueWithRequirementDTO(onPremissRedmineIssueVO.getAlmIssueWithRequirementDTO());

        Optional.ofNullable(issue.getId())
            .map(String::valueOf)
            .ifPresent(아이디 -> {
                almIssueVOBuilder
                    .id(아이디)
                    .key(아이디)
                    .self(defaultPath + redmineApi.replaceID(redmineApi.getEndpoint().getIssue(), 아이디));
            });


        Optional.ofNullable(issue.getParentId())
            .ifPresent(a->{
                almIssueVOBuilder.upperKey(String.valueOf(a));
            });

        Optional<String> rawData = redmineUtil.toJson(issue);
        rawData.ifPresent(almIssueVOBuilder::rawData);

        IssueFieldData IssueFieldData = 레드마인이슈_지라이슈필드_데이터_매핑(issue, defaultPath);

        User user = onPremissRedmineIssueVO.getUser();

        if (user != null) {
            if (!ObjectUtils.isEmpty(issue.getAssigneeId())
                    || !ObjectUtils.isEmpty(issue.getAssigneeName())
                    || !ObjectUtils.isEmpty(user.getMail())) {
                IssueFieldData.setAssignee(
                        UserData.builder()
                        .accountId(Optional.ofNullable(issue.getAssigneeId()).map(String::valueOf).orElse(null))
                        .emailAddress(Optional.ofNullable(user.getMail()).map(String::valueOf).orElse(null))
                        .displayName(issue.getAssigneeName())
                                .build());
            }
        }

        almIssueVOBuilder
            .fields(IssueFieldData)
            .armsStateCategory(getMappingCategory(serverInfo, IssueFieldData))
            .linkedIssuePdServiceIds(onPremissRedmineIssueVO.getLinkedIssuePdServiceIds())
            .linkedIssuePdServiceVersions(onPremissRedmineIssueVO.getLinkedIssuePdServiceVersions());

        return almIssueVOBuilder.build();

    }

    private IssueFieldData 레드마인이슈_지라이슈필드_데이터_매핑(Issue 이슈, String 기본경로) {
        IssueFieldData IssueFieldData = new IssueFieldData();

        Optional.ofNullable(이슈.getProjectId())
            .map(아이디 -> String.valueOf(아이디))
            .ifPresent(아이디 -> {
                String 프로젝트_경로 = 기본경로 + redmineApi.replaceID(redmineApi.getEndpoint().getProject(), 아이디);
                IssueFieldData.setProject(new ProjectDTO(프로젝트_경로, 아이디, 아이디, 이슈.getProjectName()));
            });

        Optional.ofNullable(이슈.getTracker().getId())
            .map(아이디 -> String.valueOf(아이디))
            .ifPresent(아이디 -> {
                String 이슈유형_경로 = 기본경로 + redmineApi.replaceID(redmineApi.getEndpoint().getIssuetype(), 아이디);
                IssueFieldData.setIssuetype(new IssueTypeDTO(이슈유형_경로, 아이디, 이슈.getTracker().getName()));
            });

        Optional.ofNullable(이슈.getPriorityId())
            .map(아이디 -> String.valueOf(아이디))
            .ifPresent(아이디 -> {
                String 우선순위_경로 = 기본경로 + redmineApi.replaceID(redmineApi.getEndpoint().getPriority(), 아이디);
                IssueFieldData.setPriority(new IssuePriorityDTO(우선순위_경로, 아이디, 이슈.getPriorityText()));
            });

        Optional.ofNullable(이슈.getStatusId())
            .map(아이디 -> String.valueOf(아이디))
            .ifPresent(아이디 -> {
                String 이슈상태_경로 = 기본경로 + redmineApi.replaceID(redmineApi.getEndpoint().getIssuestatus(), 아이디);
                IssueFieldData.setStatus(new IssueStatusDTO(이슈상태_경로, 아이디, 이슈.getStatusName()));
            });

        Optional.ofNullable(이슈.getAuthorId())
            .map(아이디 -> String.valueOf(아이디))
            .ifPresent(아이디 -> {
                UserData 사용자_데이터 =
                        UserData.builder()
                                .accountId(아이디)
                                .displayName(이슈.getAuthorName())
                                .build();
                IssueFieldData.setCreator(사용자_데이터);
                IssueFieldData.setReporter(사용자_데이터);
            });

        Optional.ofNullable(이슈.getCreatedOn())
            .map(생성일 -> 날짜변환(생성일))
            .ifPresent(IssueFieldData::setCreated);

        Optional.ofNullable(이슈.getUpdatedOn())
            .map(업데이트일 -> 날짜변환(업데이트일))
            .ifPresent(IssueFieldData::setUpdated);

        Optional.ofNullable(이슈.getClosedOn())
            .map(해결책 -> 날짜변환(해결책))
            .ifPresent(IssueFieldData::setResolutiondate);

        IssueFieldData.setSummary(이슈.getSubject());

        return IssueFieldData;
    }

    private String 날짜변환(Date 원본_날짜) {

        // 원본 형식
        DateTimeFormatter originalFormatter = DateTimeFormatter.ofPattern("EEE MMM dd HH:mm:ss zzz yyyy", Locale.US);

        // ZonedDateTime 객체로 파싱
        ZonedDateTime 날짜시간 = ZonedDateTime.parse(String.valueOf(원본_날짜), originalFormatter);

        // ISO 8601 형식으로 변환
        String 변환된_날짜시간 = 날짜시간.format(DateTimeFormatter.ISO_OFFSET_DATE_TIME);

        return 변환된_날짜시간;
    }

    private String getMappingCategory(ServerInfo serverInfo, IssueFieldData IssueFieldData) {
        return categoryMappingService.getMappingCategory(serverInfo, IssueFieldData.getAlmStatusId());
    }
}
