package com.arms.api.issue.type.service;

import com.arms.api.issue.type.vo.IssueTypeVO;
import com.arms.api.issue.type.strategy.*;
import com.arms.api.serverinfo.model.enums.ServerType;
import com.arms.api.serverinfo.model.ServerInfo;
import com.arms.api.serverinfo.service.ServerInfoService;
import com.arms.api.util.errors.ErrorCode;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.List;

@Slf4j
@Service
@AllArgsConstructor
public class IssueTypeService {

    private final ServerInfoService ServerInfoService;
    private final CloudJiraIssueTypeStrategy cloudJiraIssueType;
    private final OnpremiseJiraIssueTypeStrategy onpremiseJiraIssueType;
    private final OnpremiseRedmineIssueTypeStrategy onpremiseRedmineIssueType;

    private IssueTypeStrategyFactory settingIssueTypeStrategy(ServerInfo serverInfo) {

        if (serverInfo == null || StringUtils.isEmpty(serverInfo.getType())) {
            log.error("settingIssueTypeStrategy Error: serverInfo 유형 {}", ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
            throw new IllegalArgumentException("settingIssueTypeStrategy Error: serverInfo 유형 " + ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
        }

        ServerType serverType = ServerType.typeValueOf(serverInfo.getType());

        IssueTypeStrategy issueTypeStrategy;
        if (serverType == ServerType.CLOUD) {
            issueTypeStrategy = this.cloudJiraIssueType;
        }
        else if (serverType == ServerType.ON_PREMISS) {
            issueTypeStrategy = this.onpremiseJiraIssueType;
        }
        else if (serverType == ServerType.REDMINE_ON_PREMISS) {
            issueTypeStrategy = this.onpremiseRedmineIssueType;
        }
        else {
            throw new IllegalArgumentException("settingIssueTypeStrategy Error: 허용하지 않는 serverInfo 유형입니다. :: " + serverType+ " :: " + ErrorCode.SERVER_TYPE_INFO_ERROR.getErrorMsg());
        }

        IssueTypeStrategyFactory issueTypeStrategyFactory = new IssueTypeStrategyFactory();
        issueTypeStrategyFactory.registerIssueTypeStrategy(issueTypeStrategy);

        return issueTypeStrategyFactory;
    }

    public List<IssueTypeVO> getIssueTypeList(String connectId) {

        ServerInfo serverInfo = ServerInfoService.verifyServerInfo(connectId);

        ServerType serverType = ServerType.typeValueOf(serverInfo.getType());
        if (serverType == ServerType.CLOUD) {
            throw new IllegalArgumentException("지라 클라우드 타입은 이슈 유형 목록 가져오기를 사용할 수 없습니다.");
        }
        else if (serverType == ServerType.REDMINE_ON_PREMISS) {
            throw new IllegalArgumentException("레드마인 온프레미스 타입은 이슈 유형 목록 가져오기를 사용할 수 없습니다.");
        }

        IssueTypeStrategyFactory issueTypeStrategyFactory = this.settingIssueTypeStrategy(serverInfo);

        List<IssueTypeVO> returnIssueTypeList = issueTypeStrategyFactory.getIssueTypeList(serverInfo);
        return returnIssueTypeList;
    }

    public List<IssueTypeVO> getIssueTypeListByProject(String connectId, String projectId) {

        ServerInfo serverInfo = ServerInfoService.verifyServerInfo(connectId);

        ServerType serverType = ServerType.typeValueOf(serverInfo.getType());
        if (serverType == ServerType.ON_PREMISS) {
            throw new IllegalArgumentException("지라 온프레미스 타입은 프로젝트별 이슈 유형 목록 가져오기를 사용할 수 없습니다.");
        }

        IssueTypeStrategyFactory issueTypeStrategyFactory = this.settingIssueTypeStrategy(serverInfo);

        List<IssueTypeVO> returnIssueTypeList = issueTypeStrategyFactory.getIssueTypeListByProject(serverInfo, projectId);
        return returnIssueTypeList;
    }
}
