package com.arms.egovframework.javaservice.gcframework.service;

import com.arms.config.GiteaUserConfig;
import com.arms.egovframework.javaservice.gcframework.model.AbstractContentVO;
import com.arms.egovframework.javaservice.gcframework.model.FileContent;
import com.arms.egovframework.javaservice.gcframework.model.GitFileInfo;
import com.arms.egovframework.javaservice.gcframework.model.RepoType;
import com.arms.egovframework.javaservice.gcframework.parser.AbstractContentParser;
import com.arms.egovframework.javaservice.gcframework.provider.GitRepositoryProvider;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * GitFileService 인터페이스의 구현체.
 * Git Repository와 통신하여 파일 관리 및 파싱을 수행합니다.
 */
@Service // Spring Bean으로 등록
@Slf4j
public class GitFileServiceImpl implements GitFileService {

    private final GiteaUserConfig giteaUserConfig;
    private final Map<RepoType, GitRepositoryProvider> providers;
    private final Map<String, AbstractContentParser<?>> parsers;

    public GitFileServiceImpl(GiteaUserConfig giteaUserConfig,
                            List<GitRepositoryProvider> providers,
                            List<AbstractContentParser<?>> parsers) {
        this.giteaUserConfig = giteaUserConfig;
        this.providers = providers.stream()
                .collect(Collectors.toMap(GitRepositoryProvider::getType, Function.identity()));
        this.parsers = parsers.stream()
                .collect(Collectors.toMap(AbstractContentParser::getHandledType, Function.identity()));
    }

    @Override
    public FileContent getFileContent(RepoType repoType, String repoName, String filePath) {

        GitRepositoryProvider provider = providers.get(repoType);
        if (provider == null) {
            throw new IllegalArgumentException("Unsupported repository type: " + repoType);
        }

        String owner = giteaUserConfig.getOwner();
        String branch = giteaUserConfig.getGitBranch();

        log.debug("Getting file content from repo '{}', path '{}'", repoName, filePath);
        return provider.getFileContent(owner, repoName, branch, filePath);
    }

    @Override
    public <T extends AbstractContentVO> boolean upsertFileContent(RepoType repoType, String repoName, String filePath, T configVo, String commitMessage) {
        GitRepositoryProvider provider = providers.get(repoType);
        if (provider == null) {
            throw new IllegalArgumentException("Unsupported repository type: " + repoType);
        }

        String fileExtension = getFileExtension(filePath);
        AbstractContentParser<T> parser = (AbstractContentParser<T>) parsers.get(fileExtension);
        if (parser == null) {
            throw new UnsupportedOperationException("No parser found for file type: " + fileExtension);
        }

        String contentToSave = parser.serialize(configVo);

        String owner = giteaUserConfig.getOwner();
        String branch = giteaUserConfig.getGitBranch();

        log.debug("Upserting file content to repo '{}', path '{}'", repoName, filePath);
        return provider.upsertFile(owner, repoName, branch, filePath, contentToSave, commitMessage);
    }

    @Override
    public List<GitFileInfo> listFilesAndDirectories(RepoType repoType, String repoName, String directoryPath) {
        GitRepositoryProvider provider = providers.get(repoType);
        if (provider == null) {
            throw new IllegalArgumentException("Unsupported repository type: " + repoType);
        }

        String owner = giteaUserConfig.getOwner();
        String branch = giteaUserConfig.getGitBranch();

        log.debug("Listing files and directories in repo '{}', path '{}'", repoName, directoryPath);
        return provider.getListFilesAndDirectories(owner, repoName, directoryPath,branch);
    }

    @Override
    public boolean deleteFile(RepoType repoType, String repoName, String filePath, String commitMessage) {
        GitRepositoryProvider provider = providers.get(repoType);
        if (provider == null) {
            throw new IllegalArgumentException("Unsupported repository type: " + repoType);
        }

        String owner = giteaUserConfig.getOwner();
        String branch = giteaUserConfig.getGitBranch();

        log.debug("Deleting file from repo '{}', path '{}'", repoName, filePath);
        return provider.deleteFile(owner, repoName, branch, filePath, commitMessage);
    }

    @Override
    public <T extends AbstractContentVO> T parseFileContent(FileContent fileContent, String filePath, Class<T> targetVoType) {
        if (fileContent == null || fileContent.getContent() == null) {
            log.warn("Attempted to parse null or empty file content for path: {}", filePath);
            return null;
        }

        String fileExtension = getFileExtension(filePath);
        AbstractContentParser<T> parser = (AbstractContentParser<T>) parsers.get(fileExtension);
        if (parser == null) {
            throw new UnsupportedOperationException("No parser found for file type: " + fileExtension);
        }

        log.debug("Parsing file content for path '{}' with parser type '{}'", filePath, fileExtension);
        return parser.parse(fileContent.getContent(), targetVoType);
    }

    private String getFileExtension(String filePath) {
        int dotIndex = filePath.lastIndexOf('.');
        return (dotIndex == -1 || dotIndex == filePath.length() - 1) ? "" : filePath.substring(dotIndex + 1).toLowerCase();
    }
}
