package com.arms.api.analysis.resource.controller;

import com.arms.api.analysis.resource.model.dto.ResourceDTO;
import com.arms.api.analysis.resource.model.dto.ResourceSubgroupDTO;
import com.arms.api.analysis.resource.model.dto.ResourceWithAssigneesSubGroupDTO;
import com.arms.api.analysis.resource.model.dto.ResourceWithVersionIdNamesDTO;
import com.arms.api.analysis.resource.model.vo.*;
import com.arms.api.analysis.resource.model.vo.horizontalbar.HorizontalBarChartYAxisAndSeriesVO;
import com.arms.api.analysis.resource.model.vo.horizontalbar.ReqAndNotReqHorizontalBarChartVO;
import com.arms.api.analysis.resource.model.vo.pie.ReqAndNotReqPieChartVO;
import com.arms.api.analysis.resource.model.vo.pie.TotalIssueAndPieChartVO;
import com.arms.api.analysis.resource.model.vo.treemap.TreeMapWorkerVO;
import com.arms.api.analysis.resource.model.vo.treemap.TreemapExcelVO;
import com.arms.api.analysis.resource.service.ResourceService;
import com.arms.api.dashboard.model.SankeyData;
import com.arms.api.util.model.dto.PdServiceAndIsReqDTO;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.excel.ExcelUtilsBase;
import com.arms.egovframework.javaservice.treeframework.excel.ExcelUtilsFactory;
import feign.Param;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpHeaders;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

import javax.servlet.http.HttpServletResponse;
import java.util.List;
import java.util.Optional;

@Slf4j
@RestController
@RequestMapping(value = "/admin/arms/analysis/resource")
@RequiredArgsConstructor
public class ResourceController {

    private final ResourceService resourceService;

    // issue-status datatable  API
    @GetMapping("/issue-status-by-assignee")
    public ResponseEntity<List<UniqueAssigneeIssueStatusVO>> issueStatusDataByAssignee(ResourceDTO resourceDTO) throws Exception {

        log.info("[ResourceController :: issueStatusDataByAssignee] :: pdServiceId ==> {}, pdServiceVersionLinks ==> {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks());

        return ResponseEntity.ok(resourceService.issueStatusDataByAssignee(resourceDTO));
    }

    // Sankey Chart
    @GetMapping(value="/version-assignee-summary", name="resource-version-assignee-summary")
    public ResponseEntity<CommonResponse.ApiResult<SankeyData>> versionAssigneeIssueSummary(
            ResourceDTO resourceDTO, @RequestParam(required = false) Integer topN) throws Exception {

        log.info("[ResourceController :: versionAssigneeIssueSummary] :: pdServiceId ==> {}, pdServiceVersionLinks ==> {}, topN ==> {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks(), Optional.ofNullable(topN).orElse(0));

        return ResponseEntity.ok(CommonResponse.success(resourceService.sankeyChartAPIWithTopN(resourceDTO, topN)));
    }

    // Expand(모달) - 작업자 필터 API
    @GetMapping("/assignees-info")
    public ResponseEntity<List<UniqueAssigneeVO>> findAssigneesInfoData(ResourceDTO resourceDTO) throws Exception {

        log.info("[ResourceController :: findAssigneesInfoData] :: pdServiceId ==> {}, pdServiceVersionLinks ==> {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks());

        List<UniqueAssigneeVO> assigneesInfoData = resourceService.findAssigneesInfoData(resourceDTO);

        return ResponseEntity.ok(assigneesInfoData);
    }

    // top-contributors 대체 예정.
    @GetMapping("/top-contributors/pie-expand")
    public ResponseEntity<TotalIssueAndPieChartVO> findPieChartDataExpand(ResourceDTO resourceDTO) {

        log.info("[ResourceController :: findPieChartDataExpand] :: pdServiceId => {}, pdServiceVersionLinks => {}, IsReq => {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks(), resourceDTO.req());

        return ResponseEntity.ok(resourceService.findPieChartDataExpand(resourceDTO));
    }

    @GetMapping("/top-contributors/pie")
    public ResponseEntity<ReqAndNotReqPieChartVO> findPieChartData(ResourceDTO resourceDTO) {

        log.info("[ResourceController :: findPieChartData] :: pdServiceId => {}, pdServiceVersionLinks => {}, IsReq => {}, size => {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks(), resourceDTO.req(), resourceDTO.getNumSize());

        return ResponseEntity.ok(resourceService.findPieChartData(resourceDTO));
    }
    // 메인 - pie 차트 이후, isReq 및 accounts 활용 -> 차트 데이터 가져오는 API
    @PostMapping("/top-contributors/horizontal-bar")
    public ResponseEntity<List<HorizontalBarChartYAxisAndSeriesVO>> findHorizontalBarChartData(@RequestBody ResourceDTO resourceDTO) {
        log.info("[ResourceController :: findHorizontalBarChartData] :: pdServiceId => {}, pdServiceVersionLinks => {}, IsReq => {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks(), resourceDTO.req());
        return ResponseEntity.ok(resourceService.findHorizontalBarChartData(resourceDTO));
    }

    // 데이터테이블 클릭시, 같음 accouns로 한번에 그리는 API
    @PostMapping("/top-contributors/horizontal-bar-at-once")
    public ResponseEntity<ReqAndNotReqHorizontalBarChartVO> findHorizontalBarChartDataAtOnce(@RequestBody ResourceDTO resourceDTO) {
        log.info("[ResourceController :: findHorizontalBarChartDataAtOnce] :: pdServiceId => {}, pdServiceVersionLinks => {}, accounts.size => {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks(), resourceDTO.getAccounts().size());
        return ResponseEntity.ok(
                resourceService.findHorizontalBarChartDataBySameAccounts(resourceDTO));
    }

    // 리소스 트리맵v3
    @GetMapping(value="/assignees-requirements-involvements-v3", name="resource-treemap-v3")
    public ResponseEntity<CommonResponse.ApiResult<List<TreeMapWorkerVO>>> findTreeMapChartDataV3(ResourceWithVersionIdNamesDTO resourceWithVersionIdNamesDTO) throws Exception {
        return ResponseEntity.ok(CommonResponse.success(resourceService.findTreeMapChartDataV3(resourceWithVersionIdNamesDTO)));
    }

    // 리소스 트리맵 엑셀 API
    @GetMapping(value="/findAssigneeRequirementInvolvementData")
    public ResponseEntity<CommonResponse.ApiResult<List<TreemapExcelVO>>> assigneeRequirementInvolvementDetailData(ResourceWithVersionIdNamesDTO resourceWithVersionIdNamesDTO) throws Exception {
        return ResponseEntity.ok(CommonResponse.success(resourceService.assigneeRequirementInvolvementDetailData(resourceWithVersionIdNamesDTO)));
    }
    // 리소스 트리맵 엑셀 API - 엑셀다운로드
    @GetMapping(value="/findAssigneeRequirementInvolvementDataDownload")
    public void findAssigneeRequirementInvolvementDataDownload(
            ResourceWithVersionIdNamesDTO resourceWithVersionIdNamesDTO,
            @Param("fileName") String fileName,
            HttpServletResponse httpServletResponse) throws Exception {

        httpServletResponse.addHeader(HttpHeaders.CONTENT_DISPOSITION, "attachment; filename="+fileName);
        httpServletResponse.addHeader("Cache-Control", "no-cache, no-store, must-revalidate");
        httpServletResponse.addHeader("Pragma", "no-cache");
        httpServletResponse.addHeader("Expires", "0");
        httpServletResponse.setContentType("application/octet-stream");

        ExcelUtilsBase excelUtilsBase = ExcelUtilsFactory.getInstance(
                httpServletResponse.getOutputStream());

        excelUtilsBase.create(List.of(resourceService.assigneeRequirementInvolvementDetailData(resourceWithVersionIdNamesDTO)));
    }

    // word-cloud
    @GetMapping("/word-cloud-data")
    public ResponseEntity<List<WordCloudVO>> findWordCloudData(ResourceDTO resourceDTO) {

        log.info("[ResourceController :: findWordCloudData] :: pdServiceId => {}, pdServiceVersionLinks => {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks());

        return ResponseEntity.ok(resourceService.findWordCloudData(resourceDTO));
    }

    // stacked-horizontal-bar
    @GetMapping("/issue-status-stacked-bar")
    public ResponseEntity<CommonResponse.ApiResult<StackedHorizontalBarChartVO>> issueStatusStackedBarChartData(ResourceDTO resourceDTO) {

        log.info("[ResourceController :: issueStatusStackedBarChartData] :: pdServiceId => {}, pdServiceVersionLinks => {}",
                resourceDTO.pdServiceLink(), resourceDTO.pdServiceVersionLinks());

        return ResponseEntity.ok(CommonResponse.success(resourceService.issueStatusStackedBarChartData(resourceDTO)));
    }


}
