package com.arms.api.newsletter.service;

import com.arms.api.newsletter.model.NewsletterContentsDTO;
import com.arms.api.newsletter.model.NewsletterDTO;
import com.arms.api.newsletter.model.NewsletterEntity;
import com.arms.api.util.communicate.external.EngineService;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.service.TreeServiceImpl;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.Disjunction;
import org.hibernate.criterion.MatchMode;
import org.hibernate.criterion.Order;
import org.hibernate.criterion.Restrictions;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.select.Elements;
import org.modelmapper.ModelMapper;
import org.springframework.data.annotation.ReadOnlyProperty;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.transaction.Transactional;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
@RequiredArgsConstructor
@Service("newsletter")
public class NewsletterImpl extends TreeServiceImpl implements Newsletter {

    private final ModelMapper modelMapper;
    private final EngineService engineService;

    @Override
    @Transactional
    public NewsletterEntity addNewsletter(NewsletterEntity newsletterEntity) throws Exception {
        newsletterEntity.setRef(TreeConstant.First_Node_CID);  // root를 Parent로 두고있는 노드에 추가
        newsletterEntity.setC_type(TreeConstant.Leaf_Node_TYPE);  // Leaf 노드로 설정
        newsletterEntity.setC_newsletter_created(LocalDateTime.now()); // 생성 시간 설정
        newsletterEntity.setC_newsletter_hit(0); // 조회수 초기화: 조회 관련하여 외부에서 값을 컨트롤 할 수 있게끔 하면 안됨. DB 컬럼 생성 시 default 0 으로 설정 필요.);

        NewsletterEntity savedEntity = this.addNode(newsletterEntity);

        try {
            ResponseEntity<CommonResponse.ApiResult<NewsletterContentsDTO>> apiResultResponseEntity = engineService.addNewsletter(NewsletterContentsDTO.builder()
                    .id(String.valueOf(savedEntity.getC_id()))
                    .contents(newsletterEntity.getC_newsletter_contents())
                    .build());
        } catch (Exception e) {
            newsletterEntity.setC_newsletter_contents("");
            log.error("Failed to add newsletter with ID: {}", newsletterEntity);
        }

        return savedEntity;
    }

    @Override
    @Transactional
    public NewsletterEntity getNewsletter(Long newsletterId) throws Exception {
        NewsletterEntity newsletterEntity = new NewsletterEntity();
        newsletterEntity.setC_id(newsletterId);
        newsletterEntity = this.getNode(newsletterEntity);

        try {
            ResponseEntity<CommonResponse.ApiResult<NewsletterContentsDTO>> apiResultResponseEntity = engineService.getNewsletter(String.valueOf(newsletterId));
            log.info("apiResultResponseEntity.getBody()={}", apiResultResponseEntity.getBody());

            String contents = Optional.ofNullable(apiResultResponseEntity.getBody())
                    .map(CommonResponse.ApiResult::getResponse)
                    .map(NewsletterContentsDTO::getContents)
                    .orElse("");

            newsletterEntity.setC_newsletter_contents(contents);
        } catch (Exception e) {
            newsletterEntity.setC_newsletter_contents("");
            log.error("Failed to get newsletter with ID: {}", newsletterId);
        }

        // TODO: 트랜잭션 분리 및 별도 API 호출로 증가 고려. 사유: 수정페이지 갔다가 돌아올때도 조회수 증가함.
        // 조회수 1 증가
        newsletterEntity.setC_newsletter_hit(Optional.ofNullable(
                newsletterEntity.getC_newsletter_hit())
                .orElse(0) +1);
        this.updateNode(newsletterEntity); // 조회수 업데이트

        return newsletterEntity;
    }

    @Override
    @Transactional
    public NewsletterEntity updateNewsletter(NewsletterEntity newsletterEntity) throws Exception {
        newsletterEntity.setC_newsletter_updated(LocalDateTime.now()); // 수정 시간 설정
        int result = this.updateNode(newsletterEntity);
        if (result != 1) {  // 1이 반환되어야 정상 업데이트
            log.error("Failed to update newsletter with ID: {}", newsletterEntity.getC_id());
            throw new Exception("Failed to update newsletter with ID: " + newsletterEntity.getC_id());
        }

        try {
            ResponseEntity<CommonResponse.ApiResult<NewsletterContentsDTO>> apiResultResponseEntity = engineService.updateNewsletter(NewsletterContentsDTO.builder()
                    .id(String.valueOf(newsletterEntity.getC_id()))
                    .contents(newsletterEntity.getC_newsletter_contents())
                    .build());
            log.info("apiResultResponseEntity.getBody()={}", apiResultResponseEntity.getBody());
        } catch (Exception e) {
            newsletterEntity.setC_newsletter_contents("");
            log.error("Failed to update newsletter with ID: {}", newsletterEntity.getC_id());
        }

        return newsletterEntity;
    }

    @Override
    @Transactional
    public NewsletterEntity removeNewsletter(Long newsletterId) throws Exception {
        NewsletterEntity newsletterEntity = new NewsletterEntity();
        newsletterEntity.setC_id(newsletterId);
        int result = this.removeNode(newsletterEntity);
        if (result != 0) {  // 0이 반환되어야 정상 삭제
            log.error("Failed to remove newsletter with ID: {}", newsletterEntity.getC_id());
            throw new Exception("Failed to remove newsletter with ID: " + newsletterEntity.getC_id());
        }

        try {
            ResponseEntity<CommonResponse.ApiResult<NewsletterContentsDTO>> apiResultResponseEntity = engineService.deleteNewsletter(String.valueOf(newsletterId));
            log.info("apiResultResponseEntity.getBody()={}", apiResultResponseEntity.getBody());
        } catch (Exception e) {
            log.error("Failed to remove newsletter with ID: {}", newsletterId);
        }

        return newsletterEntity;
    }

    @Override
    @Transactional
    public List<NewsletterDTO> getNewsletters(NewsletterEntity newsletterEntity, int pageIndex, int pageUnit) throws Exception {

        setDefaultRestriction(newsletterEntity);
        Disjunction orCondition = Restrictions.disjunction();
        if (newsletterEntity.getC_newsletter_etc() != null) {
            orCondition.add(Restrictions.like("c_newsletter_etc", "%" + newsletterEntity.getC_newsletter_etc() + "%"));
            newsletterEntity.getCriterions().add(orCondition);  // 현재는 해당 조건만 있어서 여기에 삽입. 필터 조건이 더 늘어나면 if 문 밖으로 빼야 함
        }
        newsletterEntity.getOrder().add(Order.desc("c_newsletter_created")); // 최신순 정렬
        newsletterEntity.setPageIndex(pageIndex);
        newsletterEntity.setPageUnit(pageUnit);

        // Entity -> DTO 변환 및 썸네일 처리
        // Entity 썸네일 처리 시에는 @Transactional로 인해서 자동 업데이트가 발생할 수 있으므로 주의 필요
        List<NewsletterDTO> newsletterDTOList =  this.getPaginatedChildNode(newsletterEntity).stream()
                .map(entity -> {
                    NewsletterDTO dto = modelMapper.map(entity, NewsletterDTO.class);
                    Document doc = Jsoup.parse(dto.getC_newsletter_contents());
                    Element firstImage = doc.selectFirst("img");
                    if (firstImage != null) {
                        dto.setC_newsletter_thumbnail_url(firstImage.attr("src"));
                        dto.setC_newsletter_thumbnail_alt(firstImage.attr("alt"));

                        doc.select("img").remove();  // 본문 내용에서 이미지 제거
                        doc.select("p, div").stream()
                                .filter(e -> e.text().trim().isEmpty())  // 텍스트가 비어있는 요소 필터링
                                .filter(e -> e.html().replace("&nbsp;", "").trim().isEmpty())  // &nbsp; 만 있는 요소 필터링
                                .forEach(Element::remove);
                        dto.setC_newsletter_contents(doc.body().html());
                    }
                    return dto;
                }).collect(Collectors.toList());

        return newsletterDTOList;
    }

    @Override
    public Long getNewsletterCount(NewsletterEntity newsletterEntity) throws Exception {
        setDefaultRestriction(newsletterEntity);
        Disjunction orCondition = Restrictions.disjunction();
        if (newsletterEntity.getC_newsletter_etc() != null) {
            // 이렇게 검색할 시 불일치 데이터도 검색될 수 있음. ex) "테스트" 검색 시 "테스트입니다"도 검색됨.
            orCondition.add(Restrictions.like("c_newsletter_etc", newsletterEntity.getC_newsletter_etc(), MatchMode.ANYWHERE));
            newsletterEntity.getCriterions().add(orCondition);
        }
        return this.getNodesWithoutRoot(newsletterEntity).stream().count();
    }

    @Override
    @Transactional
    public List<NewsletterDTO> getPopularNewsletters(NewsletterEntity newsletterEntity, int pageIndex, int pageUnit) throws Exception {

        setDefaultRestriction(newsletterEntity);
        newsletterEntity.getOrder().add(Order.desc("c_newsletter_hit")); // 조회수 높은순 정렬
        newsletterEntity.setPageIndex(pageIndex);
        newsletterEntity.setPageUnit(pageUnit);

        // Entity -> DTO 변환 및 썸네일 처리
        // Entity 썸네일 처리 시에는 @Transactional로 인해서 자동 업데이트가 발생할 수 있으므로 주의 필요
        // stream().map() 대신 for문 사용
        List<NewsletterEntity> paginatedChildNode = this.getPaginatedChildNode(newsletterEntity);
        List<NewsletterDTO> newsletterDTOList = new ArrayList<>();
        for (NewsletterEntity entity : paginatedChildNode) {
            NewsletterDTO dto = modelMapper.map(entity, NewsletterDTO.class);
            Document doc = Jsoup.parse(dto.getC_newsletter_contents());
            Element firstImage = doc.selectFirst("img");
            if (firstImage != null) {
                dto.setC_newsletter_thumbnail_url(firstImage.attr("src"));
                dto.setC_newsletter_thumbnail_alt(firstImage.attr("alt"));
                doc.select("img").remove();  // 본문 내용에서 모든 이미지 태그 제거
                Elements selected = doc.select("p, div");  // p, div 태그 선택
                for (Element e : selected) {
                    if (e.text().trim().isEmpty()) {  // 텍스트가 비어있는 요소
                        e.remove();
                    } else if (e.html().replace("&nbsp;", "").trim().isEmpty()) {  // &nbsp; 만 있는 요소
                        e.remove();
                    }
                }
            }
            dto.setC_newsletter_contents(doc.body().html());
            newsletterDTOList.add(dto);
        }

        return newsletterDTOList;
    }


    // 기본적으로 Leaf 노드만 조회하도록 설정
    private void setDefaultRestriction(NewsletterEntity newsletterEntity) {
        newsletterEntity.getCriterions().add(Restrictions.eq(
                TreeConstant.Node_Type_Column, // c_type 컬럼
                TreeConstant.Leaf_Node_TYPE  // default 노드
        ));
    }
}
