package com.arms.api.report.export_service.template;

import lombok.extern.slf4j.Slf4j;
import org.apache.poi.xslf.usermodel.*;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * PPT 플레이스홀더 처리 컴포넌트
 *
 * <p>{{KEY}} 형식의 플레이스홀더를 실제 값으로 치환합니다.
 *
 * @author HS.Yang
 * @since 25.12.03
 */
@Slf4j
@Component
public class PlaceholderResolver {

    /** 플레이스홀더 패턴: {{...}} */
    private static final Pattern PLACEHOLDER_PATTERN = Pattern.compile("\\{\\{([^}]+)\\}\\}");

    /**
     * 슬라이드의 모든 플레이스홀더 치환
     *
     * @param slide 대상 슬라이드
     * @param placeholders 플레이스홀더 맵
     */
    public void resolvePlaceholders(XSLFSlide slide, Map<String, String> placeholders) {
        for (XSLFShape shape : slide.getShapes()) {
            if (shape instanceof XSLFTextShape) {
                XSLFTextShape textShape = (XSLFTextShape) shape;
                resolveTextShape(textShape, placeholders);
            } else if (shape instanceof XSLFTable) {
                XSLFTable table = (XSLFTable) shape;
                resolveTable(table, placeholders);
            } else if (shape instanceof XSLFGroupShape) {
                XSLFGroupShape groupShape = (XSLFGroupShape) shape;
                resolveGroupShape(groupShape, placeholders);
            }
        }
    }

    /**
     * 텍스트 Shape 플레이스홀더 치환
     */
    private void resolveTextShape(XSLFTextShape textShape, Map<String, String> placeholders) {
        for (XSLFTextParagraph paragraph : textShape.getTextParagraphs()) {
            for (XSLFTextRun run : paragraph.getTextRuns()) {
                String text = run.getRawText();
                if (text != null && containsPlaceholder(text)) {
                    String resolved = resolvePlaceholderString(text, placeholders);
                    run.setText(resolved);
                    log.debug("Replaced placeholder: '{}' -> '{}'", text, resolved);
                }
            }
        }
    }

    /**
     * 테이블 플레이스홀더 치환
     */
    private void resolveTable(XSLFTable table, Map<String, String> placeholders) {
        for (XSLFTableRow row : table.getRows()) {
            for (XSLFTableCell cell : row.getCells()) {
                resolveTextShape(cell, placeholders);
            }
        }
    }

    /**
     * 그룹 Shape 플레이스홀더 치환 (재귀)
     */
    private void resolveGroupShape(XSLFGroupShape groupShape, Map<String, String> placeholders) {
        for (XSLFShape shape : groupShape.getShapes()) {
            if (shape instanceof XSLFTextShape) {
                XSLFTextShape textShape = (XSLFTextShape) shape;
                resolveTextShape(textShape, placeholders);
            } else if (shape instanceof XSLFTable) {
                XSLFTable table = (XSLFTable) shape;
                resolveTable(table, placeholders);
            } else if (shape instanceof XSLFGroupShape) {
                XSLFGroupShape nestedGroup = (XSLFGroupShape) shape;
                resolveGroupShape(nestedGroup, placeholders);
            }
        }
    }

    /**
     * 문자열에서 플레이스홀더 치환
     */
    public String resolvePlaceholderString(String text, Map<String, String> placeholders) {
        if (text == null || placeholders == null) {
            return text;
        }

        String result = text;
        for (Map.Entry<String, String> entry : placeholders.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue() != null ? entry.getValue() : "";

            if (!key.startsWith("{{")) {
                key = "{{" + key + "}}";
            }

            result = result.replace(key, value);
        }

        return result;
    }

    /**
     * 플레이스홀더 포함 여부
     */
    public boolean containsPlaceholder(String text) {
        if (text == null) {
            return false;
        }
        return PLACEHOLDER_PATTERN.matcher(text).find();
    }

    /**
     * 플레이스홀더 키 추출
     */
    public List<String> extractPlaceholderKeys(String text) {
        List<String> keys = new ArrayList<>();
        if (text == null) {
            return keys;
        }

        Matcher matcher = PLACEHOLDER_PATTERN.matcher(text);
        while (matcher.find()) {
            keys.add(matcher.group(1));
        }
        return keys;
    }

    /**
     * 전체 프레젠테이션의 플레이스홀더 치환
     */
    public void resolveAllSlides(XMLSlideShow ppt, Map<String, String> placeholders) {
        for (XSLFSlide slide : ppt.getSlides()) {
            resolvePlaceholders(slide, placeholders);
        }
        log.info("Resolved placeholders in {} slides", ppt.getSlides().size());
    }
}
