/*
 * @author Dongmin.lee
 * @since 2023-03-21
 * @version 23.03.21
 * @see <pre>
 *  Copyright (C) 2007 by 313 DEV GRP, Inc - All Rights Reserved
 *  Unauthorized copying of this file, via any medium is strictly prohibited
 *  Proprietary and confidential
 *  Written by 313 developer group <313@313.co.kr>, December 2010
 * </pre>
 */
package com.arms.api.requirement.reqadd.controller;

import com.arms.api.product_service.pdservice.model.PdServiceEntity;
import com.arms.api.product_service.pdservice.service.PdService;
import com.arms.api.product_service.pdserviceversion.service.PdServiceVersion;
import com.arms.api.requirement.reqadd.excelupload.ExcelGantUpload;
import com.arms.api.requirement.reqadd.excelupload.WbsSchedule;
import com.arms.api.requirement.reqadd.model.dto.FollowReqLinkDTO;
import com.arms.api.requirement.reqadd.model.dto.ReqAddDTO;
import com.arms.api.requirement.reqadd.model.dto.ReqAddDateDTO;
import com.arms.api.requirement.reqadd.model.dto.ReqAddDetailDTO;
import com.arms.api.requirement.reqadd.model.dto.RequirementDTO;
import com.arms.api.requirement.reqadd.model.entity.ReqAddEntity;
import com.arms.api.requirement.reqadd.model.vo.ReqAddVO;
import com.arms.api.requirement.reqadd.service.ReqAdd;
import com.arms.api.requirement.reqadd_pure.model.ReqAddPureEntity;
import com.arms.api.requirement.reqdifficulty.model.ReqDifficultyEntity;
import com.arms.api.requirement.reqdifficulty.service.ReqDifficulty;
import com.arms.api.requirement.reqpriority.model.ReqPriorityEntity;
import com.arms.api.requirement.reqpriority.service.ReqPriority;
import com.arms.api.requirement.reqstate.model.ReqStateEntity;
import com.arms.api.requirement.reqstate.service.ReqState;
import com.arms.api.requirement.reqstatus.service.ReqStatus;
import com.arms.api.util.TreeServiceUtils;
import com.arms.api.util.communicate.internal.InternalService;
import com.arms.api.util.filerepository.model.FileRepositoryDTO;
import com.arms.api.util.filerepository.model.FileRepositoryEntity;
import com.arms.api.util.VersionUtil;
import com.arms.egovframework.javaservice.treeframework.TreeConstant;
import com.arms.egovframework.javaservice.treeframework.controller.CommonResponse;
import com.arms.egovframework.javaservice.treeframework.controller.TreeAbstractController;
import com.arms.egovframework.javaservice.treeframework.interceptor.SessionUtil;
import com.arms.egovframework.javaservice.treeframework.util.DateUtils;
import com.arms.egovframework.javaservice.treeframework.util.ParameterParser;
import com.arms.egovframework.javaservice.treeframework.util.StringUtils;
import com.arms.egovframework.javaservice.treeframework.validation.group.AddNode;
import com.arms.egovframework.javaservice.treeframework.validation.group.MoveNode;
import com.arms.egovframework.javaservice.treeframework.validation.group.UpdateNode;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.criterion.*;
import org.modelmapper.ModelMapper;
import org.modelmapper.PropertyMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Controller;
import org.springframework.ui.Model;
import org.springframework.ui.ModelMap;
import org.springframework.validation.BindingResult;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.MultipartHttpServletRequest;
import org.springframework.web.servlet.ModelAndView;

import javax.annotation.PostConstruct;
import javax.servlet.http.HttpServletRequest;
import java.util.*;

import static java.util.Comparator.comparing;
import static java.util.stream.Collectors.toList;

@Slf4j
@Controller
@RequestMapping(value = {"/arms/reqAdd"})
public class ReqAddController extends TreeAbstractController<ReqAdd, ReqAddDTO, ReqAddEntity> {

    @Autowired
    @Qualifier("reqAdd")
    private ReqAdd reqAdd;

    @Autowired
    @Qualifier("pdService")
    private PdService pdService;

    @Autowired
    @Qualifier("pdServiceVersion")
    private PdServiceVersion pdServiceVersion;

    @Autowired
    @Qualifier("reqPriority")
    private ReqPriority reqPriority;

    @Autowired
    @Qualifier("reqDifficulty")
    private ReqDifficulty reqDifficulty;

    @Autowired
    @Qualifier("reqState")
    private ReqState reqState;

    @Autowired
    @Qualifier("reqStatus")
    private ReqStatus reqStatus;

    @Autowired
    private InternalService internalService;

    @PostConstruct
    public void initialize() {
        setTreeService(reqAdd);
        setTreeEntity(ReqAddEntity.class);
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getMonitor.do"},
            method = {RequestMethod.GET}
    )
    public ModelAndView getMonitor(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            ModelMap model, HttpServletRequest request) throws Exception {

        SessionUtil.setAttribute("getMonitor", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();
        reqAddEntity.setOrder(Order.asc("c_position"));
        List<ReqAddEntity> list = reqAdd.getChildNodeWithoutPaging(reqAddEntity);

        SessionUtil.removeAttribute("getMonitor");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", list);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getNodesWithoutRoot.do"},
            method = {RequestMethod.GET}
    )
    public ModelAndView getNodesWithoutRoot(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            ModelMap model, HttpServletRequest request) throws Exception {

        SessionUtil.setAttribute("getNodesWithoutRoot", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();
        Criterion criterion = Restrictions.not(
                // replace "id" below with property name, depending on what you're filtering against
                Restrictions.in("c_id", new Object[]{TreeConstant.ROOT_CID, TreeConstant.First_Node_CID})
        );
        reqAddEntity.getCriterions().add(criterion);
        reqAddEntity.setOrder(Order.asc("c_position"));
        List<ReqAddEntity> list = reqAdd.getChildNodeWithoutPaging(reqAddEntity);

        SessionUtil.removeAttribute("getNodesWithoutRoot");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", list);
        return modelAndView;
    }


    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getReqNodesWithoutRoot.do"},
            method = {RequestMethod.GET}
    )
    public ResponseEntity<List<ReqAddEntity>> getReqNodesWithoutRoot(
            @PathVariable(value = "changeReqTableName") String changeReqTableName) throws Exception {

        SessionUtil.setAttribute("getReqNodesWithoutRoot", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();
        Criterion criterion = Restrictions.not(
                Restrictions.in("c_id", new Object[]{TreeConstant.ROOT_CID, TreeConstant.First_Node_CID})
        );
        reqAddEntity.getCriterions().add(criterion);
        reqAddEntity.setOrder(Order.asc("c_position"));
        List<ReqAddEntity> list = reqAdd.getChildNodeWithoutPaging(reqAddEntity);

        SessionUtil.removeAttribute("getReqNodesWithoutRoot");

        return ResponseEntity.ok(list);
    }


    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getChildNode.do"},
            method = {RequestMethod.GET}
    )
    public ModelAndView getSwitchDBChildNode(@PathVariable(value = "changeReqTableName") String changeReqTableName,
                                             ReqAddDTO reqAddDTO, HttpServletRequest request) throws Exception {

        ParameterParser parser = new ParameterParser(request);
        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException();
        }

        SessionUtil.setAttribute("getChildNode", changeReqTableName);

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);
        reqAddEntity.setWhere("c_parentid", reqAddEntity.getC_id());
        reqAddEntity.setOrder(Order.asc("c_position"));
        List<ReqAddEntity> list = reqAdd.getChildNode(reqAddEntity);

        SessionUtil.removeAttribute("getChildNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", list);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getChildNodeWithParent.do"},
            method = {RequestMethod.GET}
    )
    public ModelAndView getSwitchDBChildNodeWithParent(@PathVariable(value = "changeReqTableName") String changeReqTableName,
                                                        ReqAddDTO reqAddDTO, HttpServletRequest request) throws Exception {

        ParameterParser parser = new ParameterParser(request);
        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException();
        }

        SessionUtil.setAttribute("getChildNodeWithParent", changeReqTableName);

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);
        Criterion criterion1 = Restrictions.eq("c_parentid", reqAddEntity.getC_id());
        Criterion criterion2 = Restrictions.eq("c_id", reqAddEntity.getC_id());
        Criterion criterion3 = Restrictions.or(criterion1, criterion2);
        reqAddEntity.getCriterions().add(criterion3);
        reqAddEntity.setOrder(Order.asc("c_position"));

        List<ReqAddEntity> list = reqAdd.getChildNode(reqAddEntity);

        SessionUtil.removeAttribute("getChildNodeWithParent");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", list);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getNode.do"},
            method = {RequestMethod.GET}
    )
    public ModelAndView getSwitchDBNode(@PathVariable(value = "changeReqTableName") String changeReqTableName,
                                        ReqAddDTO reqAddDTO, HttpServletRequest request) throws Exception {

        ParameterParser parser = new ParameterParser(request);

        if (parser.getInt("c_id") <= 0) {
            throw new RuntimeException();
        }

        SessionUtil.setAttribute("getNode", changeReqTableName);

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);
        ReqAddEntity returnVO = reqAdd.getNode(reqAddEntity);

        SessionUtil.removeAttribute("getNode");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", returnVO);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/getReqAddListByFilter"},
            method = {RequestMethod.GET}
    )
    public ModelAndView getReqAddListByFilter(
            @PathVariable(value = "changeReqTableName") String changeReqTableName
            , ReqAddDTO reqAddDTO, HttpServletRequest request) throws Exception {


        SessionUtil.setAttribute("getReqAddListByFilter", changeReqTableName);

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);
        String[] versionStrArr = StringUtils.split(reqAddEntity.getC_req_pdservice_versionset_link(), ",");

        ModelAndView modelAndView = new ModelAndView("jsonView");
        List<ReqAddEntity> savedList = new ArrayList<>();
        if (versionStrArr == null || versionStrArr.length == 0) {
            reqAddEntity.setOrder(Order.asc("c_position"));
            savedList = reqAdd.getChildNodeWithoutPaging(reqAddEntity);
            SessionUtil.removeAttribute("getReqAddListByFilter");
        } else {
            Disjunction orCondition = Restrictions.disjunction();
            for (String versionStr : versionStrArr) {
                versionStr = "\\\"" + versionStr + "\\\"";
                orCondition.add(Restrictions.like("c_req_pdservice_versionset_link", versionStr, MatchMode.ANYWHERE));
            }

            if (reqAddEntity.getC_type() != null) {
                reqAddEntity.getCriterions().add(orCondition);
                reqAddEntity.getCriterions().add(Restrictions.eq("c_type", reqAddEntity.getC_type()));
            } else {
                orCondition.add(Restrictions.eq("c_type", TreeConstant.Branch_TYPE));
                reqAddEntity.getCriterions().add(orCondition);
            }

            reqAddEntity.setOrder(Order.asc("c_position"));

            savedList = reqAdd.getChildNodeWithoutPaging(reqAddEntity);
            SessionUtil.removeAttribute("getReqAddListByFilter");
        }

        modelAndView.addObject("result", savedList);
        return modelAndView;

    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/addNode.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> addReqNode(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @Validated({AddNode.class}) ReqAddDTO reqAddDTO,
            BindingResult bindingResult, ModelMap model
    ) throws Exception {

        log.info("ReqAddController :: addReqNode");

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);

        //제품(서비스) 셋팅
        reqAddEntity.setPdServiceEntity(TreeServiceUtils.getNode(pdService, reqAddDTO.getC_req_pdservice_link(), PdServiceEntity.class));

        //요구사항 우선순위 셋팅
        reqAddEntity.setReqPriorityEntity(TreeServiceUtils.getNode(reqPriority, reqAddDTO.getC_req_priority_link(), ReqPriorityEntity.class));
        
        //요구사항 난이도 셋팅
        reqAddEntity.setReqDifficultyEntity(TreeServiceUtils.getNode(reqDifficulty, reqAddDTO.getC_req_difficulty_link(), ReqDifficultyEntity.class));

        //요구사항 상태 셋팅
        reqAddEntity.setReqStateEntity(TreeServiceUtils.getNode(reqState, reqAddDTO.getC_req_state_link(), ReqStateEntity.class));

        //요구사항 생성일, 수정일 셋팅
        Date date = new Date();
        reqAddEntity.setC_req_create_date(date);
        reqAddEntity.setC_req_update_date(date);

        //버전 정보 리스트 = 글로벌 트리맵 조회
        List<Long> versionList = Optional.ofNullable(reqAddEntity.getC_req_pdservice_versionset_link())
                .map(VersionUtil::stringToLongArray)
                .map(Arrays::asList)
                .orElse(Collections.emptyList());

        //버전 시작일과 종료일을 요구사항 시작일과 종료일로 셋팅
        Date 버전시작일 = null;
        Date 버전종료일 = null;
        if (!versionList.isEmpty()) {
            Map<String, String> 시작일과_종료일 = pdServiceVersion.versionPeriod(versionList);

            버전시작일 = DateUtils.getDate(시작일과_종료일.get("earliestDate"), "yyyy/MM/dd HH:mm");
            버전종료일 = DateUtils.getDate(시작일과_종료일.get("latestDate"), "yyyy/MM/dd HH:mm");

            if (reqAddEntity.getC_req_start_date() == null) {
                reqAddEntity.setC_req_start_date(버전시작일);
            }

            if (reqAddEntity.getC_req_end_date() == null) {
                reqAddEntity.setC_req_end_date(버전종료일);
            }
        }

        //총 계획 기간일 수 셋팅
        long 총계획기간일수 = 0;
        if (reqAddEntity.getC_req_start_date() != null && reqAddEntity.getC_req_end_date() != null) {
            총계획기간일수 = DateUtils.getDiffDay(reqAddEntity.getC_req_start_date(), reqAddEntity.getC_req_end_date());
        }
        reqAddEntity.setC_req_plan_time(총계획기간일수);

        //총 기간일 수 셋팅
        long 총기간일수 = 0;
        if (버전시작일 != null && 버전종료일 != null) {
            총기간일수 = DateUtils.getDiffDay(버전시작일, 버전종료일);
        }
        reqAddEntity.setC_req_total_time(총기간일수);

        //총 작업 및 계획에 대한 일정 셋팅
        long 총작업MM = DateUtils.convertDaysToManMonth(총기간일수);
        long 총계획MM = DateUtils.convertDaysToManMonth(총계획기간일수);

        reqAddEntity.setC_req_total_resource(총작업MM);
        reqAddEntity.setC_req_plan_resource(총계획MM);
        ReqAddEntity savedNode = reqAdd.addReqNode(reqAddEntity, changeReqTableName);

        // 요구사항 default 타입일 경우에만 REQSTATUS 생성 후 ALM 서버로 요구사항 이슈 생성 로직 처리
        if (StringUtils.equals(savedNode.getC_type(),TreeConstant.Leaf_Node_TYPE)) {
            reqAdd.요구사항_생성_이후_상태정보_처리_프로세스(savedNode);
        }

        log.info("ReqAddController :: addReqNode");
        return ResponseEntity.ok(CommonResponse.success(savedNode));

    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/updateNode.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> updateReqNode(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @Validated({UpdateNode.class}) ReqAddDTO reqAddDTO, HttpServletRequest request,
            BindingResult bindingResult, ModelMap model
    ) throws Exception {

        log.info("ReqAddController :: updateReqNode");

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);

        if (reqAddDTO.getC_req_priority_link() != null) {
            reqAddEntity.setReqPriorityEntity(TreeServiceUtils.getNode(reqPriority, reqAddDTO.getC_req_priority_link(), ReqPriorityEntity.class));
        }

        if (reqAddDTO.getC_req_difficulty_link() != null) {
            reqAddEntity.setReqDifficultyEntity(TreeServiceUtils.getNode(reqDifficulty, reqAddDTO.getC_req_difficulty_link(), ReqDifficultyEntity.class));
        }

        if (reqAddDTO.getC_req_state_link() != null) {
            reqAddEntity.setReqStateEntity(TreeServiceUtils.getNode(reqState, reqAddDTO.getC_req_state_link(), ReqStateEntity.class));
        }

        Date date = new Date();
        reqAddEntity.setC_req_update_date(date);

        List<Long> versionList = Optional.ofNullable(reqAddEntity.getC_req_pdservice_versionset_link())
                .map(VersionUtil::stringToLongArray)
                .map(Arrays::asList)
                .orElse(Collections.emptyList());

        Date 버전시작일 = null;
        Date 버전종료일 = null;
        if (!versionList.isEmpty()) {
            Map<String, String> 시작일과_종료일 = pdServiceVersion.versionPeriod(versionList);

            버전시작일 = DateUtils.getDate(시작일과_종료일.get("earliestDate"), "yyyy/MM/dd HH:mm");
            버전종료일 = DateUtils.getDate(시작일과_종료일.get("latestDate"), "yyyy/MM/dd HH:mm");

            if (reqAddEntity.getC_req_start_date() == null) {
                reqAddEntity.setC_req_start_date(버전시작일);
            }

            if (reqAddEntity.getC_req_end_date() == null) {
                reqAddEntity.setC_req_end_date(버전종료일);
            }
        }

        long 총계획기간일수 = 0;
        if (reqAddEntity.getC_req_start_date() != null && reqAddEntity.getC_req_end_date() != null) {
            총계획기간일수 = DateUtils.getDiffDay(reqAddEntity.getC_req_start_date(), reqAddEntity.getC_req_end_date());
        }
        reqAddEntity.setC_req_plan_time(총계획기간일수);

        long 총기간일수 = 0;
        if (버전시작일 != null && 버전종료일 != null) {
            총기간일수 = DateUtils.getDiffDay(버전시작일, 버전종료일);
        }
        reqAddEntity.setC_req_total_time(총기간일수);

        long 총작업MM = DateUtils.convertDaysToManMonth(총기간일수);
        long 총계획MM = DateUtils.convertDaysToManMonth(총계획기간일수);

        reqAddEntity.setC_req_total_resource(총작업MM);
        reqAddEntity.setC_req_plan_resource(총계획MM);

        ResponseEntity<ReqAddVO> 요구사항조회 = internalService.요구사항조회(changeReqTableName, reqAddEntity.getC_id());
        ReqAddVO reqAddVO = 요구사항조회.getBody();

        if (reqAddVO == null) {
            log.error("ReqAddImpl :: updateReqNode :: 요구사항 수정 전 데이터 조회에 실패했습니다. 요구사항 ID : " + reqAddEntity.getC_id());
            throw new Exception("요구사항 수정 전 데이터 조회에 실패했습니다. 관리자에게 문의해 주세요.");
        }

        Integer result = reqAdd.updateReqNode(reqAddEntity, changeReqTableName);

        if (StringUtils.equals(reqAddVO.getC_type(),TreeConstant.Leaf_Node_TYPE)) {
            reqAdd.요구사항_수정_이후_상태정보_처리_프로세스(changeReqTableName, reqAddEntity, reqAddVO);
        }

        return ResponseEntity.ok(CommonResponse.success(result));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/updateDate.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> updateReqDate(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @Validated({UpdateNode.class}) ReqAddDateDTO reqAddDateDTO, HttpServletRequest request,
            BindingResult bindingResult, ModelMap model
    ) throws Exception {

        log.info("ReqAddController :: updateDate");

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDateDTO, ReqAddEntity.class);

        long 총계획기간일수 = 0;
        if (reqAddEntity.getC_req_start_date() != null && reqAddEntity.getC_req_end_date() != null) {
            총계획기간일수 = DateUtils.getDiffDay(reqAddEntity.getC_req_start_date(), reqAddEntity.getC_req_end_date());
        }
        reqAddEntity.setC_req_plan_time(총계획기간일수);

        long 총계획MM = DateUtils.convertDaysToManMonth(총계획기간일수);
        reqAddEntity.setC_req_total_resource(총계획MM);

        SessionUtil.setAttribute("updateDate", changeReqTableName);

        int result = reqAdd.updateNode(reqAddEntity);

        SessionUtil.removeAttribute("updateDate");

        return ResponseEntity.ok(CommonResponse.success(result));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/updateDataBase.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> updateDataBase(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @Validated({UpdateNode.class}) ReqAddDTO reqAddDTO, HttpServletRequest request,
            BindingResult bindingResult, ModelMap model
    ) throws Exception {

        log.info("ReqAddController :: updateDataBase"); // 요구사항 상태 우선순위 난이도 시작일 종료일 데이터 베이스 값 변경

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);
        if(reqAddDTO.getC_req_state_link() != null){
            reqAddEntity.setReqStateEntity(TreeServiceUtils.getNode(reqState, reqAddDTO.getC_req_state_link(), ReqStateEntity.class)); // 상태
        }
        if(reqAddDTO.getC_req_priority_link() != null){
            reqAddEntity.setReqPriorityEntity(TreeServiceUtils.getNode(reqPriority, reqAddDTO.getC_req_priority_link(), ReqPriorityEntity.class)); // 우선순위
        }
        if(reqAddDTO.getC_req_difficulty_link() != null){
            reqAddEntity.setReqDifficultyEntity(TreeServiceUtils.getNode(reqDifficulty, reqAddDTO.getC_req_difficulty_link(), ReqDifficultyEntity.class)); // 난이도
        }

        int result = reqAdd.updateDataBase(reqAddEntity,changeReqTableName);

        // reqStatus 업데이트 (상태, 우선순위, 난이도)
        reqAdd.요구사항_상태_디비_업데이트(reqAddEntity, changeReqTableName);

        return ResponseEntity.ok(CommonResponse.success(result));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/removeNode.do"},
            method = {RequestMethod.DELETE}
    )
    public ResponseEntity<?> removeReqNode(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @Validated({UpdateNode.class}) ReqAddDTO reqAddDTO, HttpServletRequest request,
            BindingResult bindingResult, ModelMap model) throws Exception {

        log.info("ReqAddController :: removeReqNode");
        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);

        ResponseEntity<ReqAddVO> 요구사항조회 = internalService.요구사항조회(changeReqTableName, reqAddEntity.getC_id());
        ReqAddVO reqAddVO = 요구사항조회.getBody();

        if (reqAddVO == null) {
            log.error("ReqAddImpl :: updateReqNode :: 요구사항 수정 전 데이터 조회에 실패했습니다. 요구사항 ID : " + reqAddEntity.getC_id());
            throw new Exception("요구사항 수정 전 데이터 조회에 실패했습니다. 관리자에게 문의해 주세요.");
        }

        int removedReqAddEntity = reqAdd.removeReqNode(reqAddEntity, changeReqTableName, request);

        if (StringUtils.equals(reqAddVO.getC_type(),TreeConstant.Leaf_Node_TYPE)) {
            reqAdd.요구사항_삭제_이후_상태정보_처리_프로세스(changeReqTableName, reqAddEntity, reqAddVO);
        }

        log.info("ReqAddController :: removeReqNode");
        return ResponseEntity.ok(CommonResponse.success(removedReqAddEntity));

    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/moveNode.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> moveReqNode(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @Validated({MoveNode.class}) ReqAddDTO reqAddDTO, HttpServletRequest request,
            BindingResult bindingResult, ModelMap model) throws Exception {

        log.info("ReqAddController :: moveReqNode");
        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);

        SessionUtil.setAttribute("moveNode", changeReqTableName);

        ReqAddEntity savedReqAddEntity = reqAdd.moveNode(reqAddEntity, request);

        SessionUtil.removeAttribute("moveNode");

        log.info("ReqAddController :: moveReqNode");
        return ResponseEntity.ok(CommonResponse.success(savedReqAddEntity));

    }

    @ResponseBody
    @RequestMapping(value = "/uploadFileToNode.do", method = RequestMethod.POST)
    public ModelAndView uploadFileToNode(final MultipartHttpServletRequest multiRequest,
                                         HttpServletRequest request, Model model) throws Exception {

        ParameterParser parser = new ParameterParser(request);
        long pdservice_link = parser.getLong("pdservice_link");

        HashMap<String, Set<FileRepositoryEntity>> map = new HashMap();

        Set<FileRepositoryEntity> entitySet = Collections.emptySet();
        map.put("files", entitySet);
        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", map);
        return modelAndView;
    }

    @ResponseBody
    @RequestMapping(value = "/getFilesByNode.do", method = RequestMethod.GET)
    public ModelAndView getFilesByNode(FileRepositoryDTO fileRepositoryDTO, HttpServletRequest request) throws Exception {

        ParameterParser parser = new ParameterParser(request);
        HashMap<String, Set<FileRepositoryEntity>> returnMap = new HashMap();

        ModelAndView modelAndView = new ModelAndView("jsonView");
        modelAndView.addObject("result", returnMap);

        return modelAndView;
    }

    @ResponseBody
    @PostMapping(value = "/sample/excel-to-list")
    public ResponseEntity excelUpload(@RequestPart("excelFile") MultipartFile excelFile, HttpServletRequest request) throws Exception {
        //확인후에 저장을 하기 위한 샘플입니다.
        return ResponseEntity.ok(CommonResponse.success(
                        new ExcelGantUpload(excelFile.getInputStream())
                                .getGetWebScheduleList()
                                .stream()
                                .sorted(comparing(WbsSchedule::getDepth).reversed())
                                .collect(toList())
                )
        );
    }

    @ResponseBody
    @GetMapping(value = "/{changeReqTableName}/getDetail.do")
    public ResponseEntity<ReqAddDetailDTO> followReqLink(FollowReqLinkDTO followReqLinkDTO, @PathVariable(value = "changeReqTableName") String changeReqTableName
    ) throws Exception {

        return ResponseEntity.ok(reqAdd.getDetail(followReqLinkDTO, changeReqTableName));
    }


    @GetMapping(value = "/{changeReqTableName}/getNodeDetail.do")
    public ResponseEntity<ReqAddVO> loadReqNode(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @RequestParam(value = "c_id") Long c_id, HttpServletRequest request
    ) throws Exception {

        log.info("ReqAddController :: getNodeDetail.do :: changeReqTableName {} :: c_id {}", changeReqTableName, c_id);

        SessionUtil.setAttribute("getNodeDetail", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();

        reqAddEntity.setC_id(c_id);

        ReqAddEntity response = reqAdd.getNode(reqAddEntity);

        log.info("ReqAddController :: getNodeDetail.do :: response :: " + response);

        ModelMapper customModelMapper = new ModelMapper();
        customModelMapper.addMappings(new PropertyMap<ReqAddEntity, ReqAddVO>() {
            @Override
            protected void configure() {
                map().setC_req_pdservice_link(source.getPdServiceEntity().getC_id());
                map().setC_req_priority_link(source.getReqPriorityEntity().getC_id());
                map().setC_req_state_link(source.getReqStateEntity().getC_id());
                map().setC_req_difficulty_link(source.getReqDifficultyEntity().getC_id());
            }
        });

        ReqAddVO reqAddDto = customModelMapper.map(response, ReqAddVO.class);

        log.info("ReqAddController :: getNodeDetail.do :: reqAddDto :: " + reqAddDto);

        SessionUtil.removeAttribute("getNodeDetail");

        return ResponseEntity.ok(reqAddDto);
    }


    @GetMapping(value = "/{changeReqTableName}/getNodesWhereInIds.do")
    public ResponseEntity<List<ReqAddVO>> getNodesWhereInIds(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @RequestParam List<Long> ids, HttpServletRequest request
    ) throws Exception {

        log.info("ReqAddController :: getNodesWhereInIds :: changeReqTableName :: {} :: ids {} ", changeReqTableName, ids);

        SessionUtil.setAttribute("getNodesWhereInIds", changeReqTableName);

        ReqAddEntity reqAddEntity = new ReqAddEntity();

        Criterion criterion = Restrictions.in("c_id", ids);

        reqAddEntity.getCriterions().add(criterion);

        reqAddEntity.setOrder(Order.asc("c_position"));

        List<ReqAddEntity> list = reqAdd.getChildNodeWithoutPaging(reqAddEntity);

        SessionUtil.removeAttribute("getNodesWhereInIds");

        ModelMapper customModelMapper = new ModelMapper();
        customModelMapper.addMappings(new PropertyMap<ReqAddEntity, ReqAddVO>() {
            @Override
            protected void configure() {
                map().setC_req_pdservice_link(source.getPdServiceEntity().getC_id());
                map().setC_req_priority_link(source.getReqPriorityEntity().getC_id());
                map().setC_req_state_link(source.getReqStateEntity().getC_id());
                map().setC_req_difficulty_link(source.getReqDifficultyEntity().getC_id());
            }
        });

        List<ReqAddVO> reqAddVOList = list.stream().map(entity -> customModelMapper.map(entity, ReqAddVO.class)).collect(toList());

        return ResponseEntity.ok(reqAddVOList);
    }

    @PostMapping(value = "/getRequirementAssignee")
    public ResponseEntity<?> getRequirementAssignee(@RequestBody RequirementDTO requirementDTO, HttpServletRequest request) throws Exception {
        log.info("ReqAddController :: getRequirementAssignee");

        return ResponseEntity.ok(CommonResponse.success( reqAdd.getRequirementAssignee(requirementDTO)));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/updateReqAddOnly.do"},
            method = {RequestMethod.POST}
    )
    public ResponseEntity<?> updateReqAddOnly(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @RequestBody ReqAddDTO reqAddDTO, HttpServletRequest request,
            BindingResult bindingResult, ModelMap model
    ) throws Exception {

        log.info("ReqAddController :: updateReqAddOnly"); // 요구사항 상태 우선순위 난이도 시작일 종료일 데이터 베이스 값 변경

        ReqAddEntity reqAddEntity = modelMapper.map(reqAddDTO, ReqAddEntity.class);
        if(reqAddDTO.getC_req_state_link() != null){
            reqAddEntity.setReqStateEntity(TreeServiceUtils.getNode(reqState, reqAddDTO.getC_req_state_link(), ReqStateEntity.class)); // 상태
        }
        if(reqAddDTO.getC_req_priority_link() != null){
            reqAddEntity.setReqPriorityEntity(TreeServiceUtils.getNode(reqPriority, reqAddDTO.getC_req_priority_link(), ReqPriorityEntity.class)); // 우선순위
        }
        if(reqAddDTO.getC_req_difficulty_link() != null){
            reqAddEntity.setReqDifficultyEntity(TreeServiceUtils.getNode(reqDifficulty, reqAddDTO.getC_req_difficulty_link(), ReqDifficultyEntity.class)); // 난이도
        }

        SessionUtil.setAttribute("updateReqAddOnly", changeReqTableName);
        int result = reqAdd.updateNode(reqAddEntity);
        SessionUtil.removeAttribute("updateReqAddOnly");

        return ResponseEntity.ok(CommonResponse.success(result));
    }

    @ResponseBody
    @RequestMapping(
            value = {"/{changeReqTableName}/updateReqAddState.do"},
            method = {RequestMethod.PUT}
    )
    public ResponseEntity<?> updateReqState(
            @PathVariable(value = "changeReqTableName") String changeReqTableName,
            @RequestParam(value = "stateBeforeChange") Long 변경_전_상태아이디,
            @RequestParam(value = "stateAfterChange") Long 변경_후_상태아이디) throws Exception {

        log.info("ReqAddController :: updateReqAddState");
        SessionUtil.setAttribute("updateReqAddState", changeReqTableName);
        List<ReqAddPureEntity> 요구사항_상태변경 = reqAdd.요구사항_상태변경(변경_전_상태아이디, 변경_후_상태아이디);
        SessionUtil.removeAttribute("updateReqAddState");

        return ResponseEntity.ok(CommonResponse.success(요구사항_상태변경));
    }

}
