package com.arms.api.util;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

public class DateRangeUtil {

    public static List<String> getDateRange(String startDateStr, String endDateStr) {
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");
        List<String> dateList = new ArrayList<>();
        LocalDate startDate = LocalDate.parse(startDateStr, formatter);
        LocalDate endDate = LocalDate.parse(endDateStr, formatter);

        if (endDate.isBefore(startDate)) {
            return dateList; // 빈 리스트 리턴
        }

        LocalDate current = startDate;
        while (!current.isAfter(endDate)) {
            dateList.add(current.format(formatter));
            current = current.plusDays(1);
        }
        return dateList;
    }

    /**
     * 주어진 날짜가 지정된 날짜 범위 안에 포함되는지 확인합니다.
     * 날짜 범위는 startDate, endDate의 존재 여부에 따라 동적으로 결정됩니다.
     *
     * @param targetDate     비교할 날짜 객체
     * @param startDateString 범위 시작 날짜 문자열 (예: "2025-08-01")
     * @param endDateString   범위 종료 날짜 문자열 (예: "2025-08-07")
     * @return targetDate가 범위 안에 포함되면 true, 아니면 false
     */
    public static boolean isWithinDateRange(Date targetDate, String startDateString, String endDateString) {
        if (targetDate == null) {
            return false;
        }

        // startDate와 endDate가 모두 없는 경우, 항상 true 반환
        if (startDateString == null && endDateString == null) {
            return true;
        }

        try {
            Date startDate = null;
            Date endDate = null;
            SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");

            // 1. startDate 파싱 및 시간 조정 (00:00:00)
            if (startDateString != null) {
                startDate = formatter.parse(startDateString);
            }

            // 2. endDate 파싱 및 시간 조정 (23:59:59)
            if (endDateString != null) {
                endDate = formatter.parse(endDateString);
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(endDate);
                calendar.set(Calendar.HOUR_OF_DAY, 23);
                calendar.set(Calendar.MINUTE, 59);
                calendar.set(Calendar.SECOND, 59);
                calendar.set(Calendar.MILLISECOND, 999);
                endDate = calendar.getTime();
            }

            // 3. 조건에 따른 날짜 비교
            if (startDate != null && endDate != null) {
                // Case 3: startDate와 endDate가 모두 있는 경우
                return !targetDate.before(startDate) && !targetDate.after(endDate);
            } else if (startDate != null) {
                // Case 1: startDate만 있는 경우
                return !targetDate.before(startDate); // targetDate >= startDate
            } else { // endDate != null
                // Case 2: endDate만 있는 경우
                return !targetDate.after(endDate); // targetDate <= endDate
            }

        } catch (ParseException e) {
            // 날짜 파싱 실패 시 예외 처리
            return false;
        }
    }

    public static boolean isNotWithinDateRange(Date targetDate, String startDateString, String endDateString) {
        return !isWithinDateRange(targetDate, startDateString, endDateString);
    }

}
