package com.arms.egovframework.javaservice.gcframework.service;

import com.arms.egovframework.javaservice.gcframework.model.GitFileInfo;
import com.arms.egovframework.javaservice.gcframework.model.AbstractContentVO;
import com.arms.egovframework.javaservice.gcframework.model.FileContent;
import com.arms.egovframework.javaservice.gcframework.model.RepoType;

import java.util.List;

/* *
 * GitFileService (<<interface>>): Git Repository에서 파일 단위의 CRUD 및 파싱 작업을 추상화한 인터페이스입니다.
 * 이 서비스는 하위 프로바이더와 파서를 지휘하여 파일을 관리합니다.
 */

/**
 * Git Repository에서 파일 내용을 조회, 업데이트, 삭제하고,
 * 파싱된 VO로 변환하는 공통 파일 관리 서비스 인터페이스.
 */
public interface GitFileService {

    /**
     * 특정 Repository의 파일 내용을 조회합니다.
     * @param repoType Git Repository 타입 (예: "gitea", "github")
     * @param repoName 대상 Repository 이름
     * @param filePath 조회할 파일 경로
     * @return 파일 내용과 메타데이터를 담은 FileContent 객체
     */
    FileContent getFileContent(RepoType repoType, String repoName, String filePath);

    /**
     * 특정 Repository에 파일 내용을 생성하거나 업데이트합니다.
     * @param repoType Git Repository 타입
     * @param repoName 대상 Repository 이름
     * @param filePath 생성/업데이트할 파일 경로
     * @param configVo 저장할 비즈니스 VO (이 VO는 파일 내용으로 직렬화됩니다)
     * @param commitMessage 커밋 메시지
     * @return 성공 여부
     */
    <T extends AbstractContentVO> boolean upsertFileContent(RepoType repoType, String repoName, String filePath, T configVo, String commitMessage);

    /**
     * 특정 Repository의 디렉토리 또는 파일 목록을 조회합니다.
     * @param repoType Git Repository 타입
     * @param repoName 대상 Repository 이름
     * @param directoryPath 조회할 디렉토리 경로
     * @return 디렉토리 내 GitFileInfo 목록
     */
    List<GitFileInfo> listFilesAndDirectories(RepoType repoType, String repoName, String directoryPath);

    /**
     * 특정 Repository의 파일을 삭제합니다.
     * @param repoType Git Repository 타입
     * @param repoName 대상 Repository 이름
     * @param filePath 삭제할 파일 경로
     * @param commitMessage 커밋 메시지
     * @return 성공 여부
     */
    boolean deleteFile(RepoType repoType, String repoName, String filePath, String commitMessage);

    /**
     * FileContent 객체로부터 파일 내용을 추출하여 지정된 비즈니스 VO 타입으로 파싱합니다.
     * @param fileContent 조회된 파일 내용 객체
     * @param filePath 파일 경로 (파서 선택을 위해 확장자 추출에 사용)
     * @param targetVoType 파싱할 VO의 Class 타입
     * @return 파싱된 비즈니스 VO
     */
    <T extends AbstractContentVO> T parseFileContent(FileContent fileContent, String filePath, Class<T> targetVoType);
}
